#ifndef LUM_OS_MAIN_H
#define LUM_OS_MAIN_H

/*
  This source is part of the Illumination library
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#include <Lum/Private/ImportExport.h>

#include <string>
#include <vector>

namespace Lum {
  namespace OS {

    /**
      Base class for main event loops defining fundamental order
      of program initialisation and deinitialisation.

      General execution order:
      * Initialize() - calls Cleanup() on error and then quits
      * ReadConfiguration() - calls Clanup() on error and then quits
      * Main() - calls Clanup() on error and then quits
      * Cleanup()

      Runtime callbacks:
      * OnDumpStatus()
    */
    class LUMAPI Main
    {
    public:
      std::wstring              appName;
      std::wstring              progName;
      std::vector<std::wstring> args;

    public:
      virtual ~Main();

      virtual bool Initialize(const std::wstring& appName,int argc, char* argv[]) = 0;
      virtual bool Prepare() = 0;
      virtual bool Body() = 0;
      virtual void Cleanup() = 0;

      virtual void OnDumpStatus() = 0;
    };

    class LUMAPI MainBase : public Main
    {
    public:
      bool Initialize(const std::wstring& appName,int argc, char* argv[]);
      bool Prepare();
      void Cleanup();

      void OnDumpStatus();
    };

    class LUMAPI MainIllumination : public MainBase
    {
    public:
      MainIllumination();

      bool Prepare();
      void Cleanup();
    };

    template<class D>
    class LUMAPI MainDialog : public MainIllumination
    {
    public:
      D *window;

    public:
      MainDialog()
      : window(NULL)
      {
        // no code
      }

      bool Prepare()
      {
        if (!MainIllumination::Prepare()) {
          return false;
        }

        window=new D();

        return window!=NULL;
      }

      bool Body()
      {
        if (window->Open()) {
          window->SetExitAction(window->GetClosedAction());
          window->EventLoop();
          window->Close();

          return true;
        }
        else {
          return false;
        }
      }

      void Cleanup()
      {
        delete window;

        MainIllumination::Cleanup();
      }
    };

#define LUM_MAIN(MAIN_CLASS,APP_NAME) \
int main(int argc, char* argv[])\
{\
  MAIN_CLASS mainClass;\
\
  if (!mainClass.Initialize(APP_NAME,argc,argv)) {\
    mainClass.Cleanup();\
    return 1;\
  }\
\
  if (!mainClass.Body()) {\
    mainClass.Cleanup();\
\
    return 2;\
  }\
\
  mainClass.Cleanup();\
\
  return 0;\
}

  }
}

#endif
