#ifndef LUM_OS_THEME_H
#define LUM_OS_THEME_H

/*
  This source is part of the Illumination library
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <vector>

#include <Lum/Private/ImportExport.h>

#include <Lum/OS/Color.h>
#include <Lum/OS/Display.h>
#include <Lum/OS/DrawInfo.h>
#include <Lum/OS/Fill.h>
#include <Lum/OS/Frame.h>
#include <Lum/OS/Image.h>

namespace Lum {
  namespace OS {

    /**
      A abstract base class for holding all information for theming a user interface.
    */
    class LUMAPI Theme
    {
    public:
      static const size_t stockImageCount = 8; //! Number of predefined stock images

      /**
        List of (optional!) images for enriching visual controls.
       */
      enum StockImage {
        positiveStockImage  =  0, //! Image for marking a positive action (e.g. for a button)
        negativeStockImage  =  1, //! Image for marking a negative action (e.g. for a button)
        defaultStockImage   =  2, //! Image for marking a default action (e.g. for a button)
        helpStockImage      =  3, //! Image for marking a help action (e.g. for a button)
        closeStockImage     =  4, //! Image for closing widget
        addStockImage       =  5, //! Image for adding somethng
        removeStockImage    =  6, //! Image for removing something
        editStockImage      =  7  //! Image for editing something
      };

      enum ScrollKnobMode {
        scrollKnobModeSingle,
        scrollKnobModeDouble
      };

      enum ToolbarPosition {
        toolbarTop,
        toolbarBottom
      };

      enum MenuType {
        menuTypeStrip,
        menuTypeTopPopup,
        menuTypeActionPopup
      };

      enum DialogActionPosition {
        actionPosHorizButtonRowBelow,
        actionPosVertButtonRowRight
      };

    public:
      Theme(Display *display);
      virtual ~Theme();

      /**
        @name Fonthandling
        Information about the to be used fonts for this theme.
      */
      //@{
      virtual bool HasFontSettings() const = 0;

      virtual std::wstring GetProportionalFontName() const = 0;
      virtual size_t GetProportionalFontSize() const = 0;

      virtual std::wstring GetFixedFontName() const = 0;
      virtual size_t GetFixedFontSize() const = 0;
      //@}

      /**
        @name Theme specific GUI primitives
        Colors, fills, frames and images for themes.
      */
      //@{
      virtual Color GetColor(Display::ColorIndex color) const = 0;
      virtual Fill* GetFill(Display::FillIndex fill) const = 0;
      virtual Frame* GetFrame(Display::FrameIndex frame) const = 0;
      virtual Image* GetImage(Display::ImageIndex image) const = 0;
      virtual Image* GetStockImage(StockImage image) const = 0;

      virtual void PushUniqueFill(DrawInfo* draw, size_t index, size_t maximum) = 0;
      virtual void PopUniqueFill(DrawInfo* draw, size_t index, size_t maximum) = 0;
      //@}

      /**
        @name Spaces
        Theme specific spacing.
      */
      //@{
      virtual size_t GetSpaceHorizontal(Display::Space space) const = 0;
      virtual size_t GetSpaceVertical(Display::Space space) const = 0;

      virtual size_t GetMinimalButtonWidth() const = 0;
      virtual size_t GetFirstTabOffset() const = 0;
      virtual size_t GetLastTabOffset() const = 0;

      /**
        How many pixels must the mouse move before this is recognized as drag start?
      */
      virtual size_t GetDragStartSensitivity() const = 0;
      /**
        How many pixels may the move move after click (and while holding) to get this
        recognizes as "click and hold" (right mouse button replacement)?
      */
      virtual size_t GetMouseClickHoldSensitivity() const = 0;
      //@}

      /**
        @name Variants and behaviours
        Choosing between display variants and different behaviours.
      */
      //@{
      virtual ScrollKnobMode GetScrollKnobMode() const = 0;
      virtual ToolbarPosition GetToolbarPosition() const = 0;
      virtual bool ShowButtonImages() const = 0;
      virtual bool RequestFingerFriendlyControls() const = 0;

      virtual MenuType GetMenuType() const = 0;
      virtual void GetMenuTypeActionPopupOffset(int& x, int& y) const = 0;
      virtual int GetSubMenuHorizontalOffset() const = 0;

      virtual bool SubdialogsAlwaysFlexHoriz() const = 0;
      virtual bool SubdialogsAlwaysFlexVert() const = 0;

      virtual DialogActionPosition GetDialogActionPosition() const =0;
      //@}
    };
  }
}

#endif
