#ifndef LUM_MODEL_TABLE_H
#define LUM_MODEL_TABLE_H

/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <algorithm>
#include <list>
#include <vector>

#include <Lum/Base/Object.h>

#include <Lum/Object.h>
#include <iostream>
namespace Lum {
  namespace Model {

    /**
      Abstract baseclass for all table models. Its purpose is,
      to define an common interface for all tablemodels. The
      table object itself will only talk with the model through
      this interface.
    */
    class LUMAPI Table : public Base::Model
    {
    public:
      class LUMAPI RefreshCell : public Base::ResyncMsg
      {
      public:
        int column;
        int row;
      };

      class LUMAPI RefreshRow : public Base::ResyncMsg
      {
      public:
        int row;
      };

      class LUMAPI InsertRow : public Base::ResyncMsg
      {
      public:
        int    row;
        size_t count;
      };

      class LUMAPI DeleteRow : public Base::ResyncMsg
      {
      public:
        int    row;
        size_t count;
      };

    public:
      Table();
      ~Table();

      void RedrawRow(size_t row);
      void RedrawCell(size_t column, size_t row);
      void NotifyInsert(size_t row, size_t count);
      void NotifyDelete(size_t row, size_t count);

      virtual size_t GetRows() const = 0;

      virtual std::wstring GetString(size_t column, size_t row) const = 0;
      virtual Lum::Object* GetObject(size_t column, size_t row) const = 0;

      virtual bool Sort(size_t column, bool down=true);
    };

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Base::Reference<Table>;
#endif

    typedef Base::Reference<Table> TableRef;

    /**
      This implementation of the Table interface is obsolete. Don't use it anymore!
     */
    class LUMAPI ListTable : public Table
    {
    public:
      class LUMAPI Entry
      {
      protected:
        ListTable *table;

      public:
        Entry(ListTable* table);
        virtual ~Entry();

        virtual std::wstring GetString(size_t column) const;
        virtual Lum::Object* GetObject(size_t column) const;
        virtual bool IsGreater(const Entry* other, size_t column) const;
      };

      class LUMAPI StdEntry : public Entry
      {
      private:
        class Cell
        {
        public:
          std::wstring string;
          Lum::Object  *object;

          Cell();
        };

      private:
        std::vector<Cell> row;

      public:
        StdEntry(ListTable* table);
        virtual ~StdEntry();

        std::wstring GetString(size_t column) const;
        Lum::Object* GetObject(size_t column) const;
        void SetString(size_t column, const std::wstring& string);
        void SetObject(size_t column, Lum::Object* object);
      };

    private:
      mutable std::list<Entry*>           list;
      mutable std::list<Entry*>::iterator current;
      mutable size_t                      currentPos;

    public:
      ListTable();
      virtual ~ListTable();

      size_t GetRows() const;

      std::wstring GetString(size_t column, size_t row) const;
      Lum::Object* GetObject(size_t column, size_t row) const;

      Entry* GetEntry(size_t row) const;
      size_t GetRow(Entry* entry) const;
      bool Sort(size_t column, bool down=true);
      // Insert
      // SwapWithNext
      void Delete(size_t row);
      void Prepend(Entry* entry);
      void Append(Entry* entry);
      void AppendString(const std::wstring& string);
      void Clear();
    };

    typedef Base::Reference<ListTable> ListTableRef;

    /**
      Abstract base class for TableModel implementations that wrap stdandard library
      containers.
     */
    template<typename Element, typename Container>
    class LUMAPI StdTableBase : public Table
    {
    public:
      typedef Container                                  ContainerType;
      typedef Element                                    ElementType;
      typedef typename ContainerType::iterator           Iterator;

    public:
      /**
        Maps element to the to be displayed data (eitehr string or an object).
       */
      class LUMAPI DataProvider
      {
      public:
        virtual ~DataProvider()
        {
          // no code
        }
 
        virtual std::wstring GetString(const Iterator& /*iter*/, size_t /*column*/) const
        {
          return L"";
        }

        virtual Lum::Object* GetObject(const Iterator& /*iter*/, size_t /*column*/) const
        {
          return NULL;
        }
      };

      /**
        Optionally assignable function object for sorting of tables. If no Comp
        no sorting is possible.
       */
      class LUMAPI Comparator
      {
      public:
        virtual bool operator()(const Element& a, const Element& b, size_t column, bool down) const = 0;
      };

    protected:
      /**
        Internal helper class.
       */
      class InternalComparator
      {
      private:
        Comparator *comparator;
        size_t     column;
        bool       down;

      public:
        InternalComparator(Comparator* comparator, size_t column, bool down)
        : comparator(comparator),
          column(column),
          down(down)
        {
          // no code
        }

        bool operator()(const Element& a, const Element& b) const
        {
          return comparator->operator()(a,b,column,down);
        }
      };

    private:
      mutable Iterator current;
      mutable size_t   currentPos;
      mutable bool     currentInvalid;
      DataProvider     *dataProvider;
      Comparator       *comparator;

    private:
      void GotoEntry(size_t row) const
      {
        assert(row>=1 && row<=GetContainer().size());

        if (currentInvalid) {
          // current is mutable,but compiler chooses const version of GetContainer()
          current=const_cast<StdTableBase*>(this)->GetContainer().begin();
          currentPos=1;
        }

        if (currentPos<row) {
          while (currentPos<row) {
            ++current;
            ++currentPos;
          }
        }
        else if (currentPos>row) {
          while (currentPos>row) {
            --current;
            --currentPos;
          }
        }

        assert(current!=GetContainer().end());
      }

    protected:
      void InvalidateCurrent()
      {
        currentInvalid=true;
      }

    public:
      virtual Container& GetContainer() = 0;
      virtual const Container& GetContainer() const = 0;

      StdTableBase(DataProvider* dataProvider, Comparator* comparator=NULL)
       : currentPos(0),
         currentInvalid(true),
         dataProvider(dataProvider),
         comparator(comparator)
      {
        // no code
      }

      ~StdTableBase()
      {
        delete dataProvider;
        delete comparator;
      }

      size_t GetRows() const
      {
        return GetContainer().size();
      }

      const Element& GetEntry(size_t row) const
      {
        GotoEntry(row);

        return *current;
      }

      /**
        You need to manually call RereshRow after changing the entry!
       */
      Element& GetEntry(size_t row)
      {
        GotoEntry(row);

        return *current;
      }

      std::wstring GetString(size_t column, size_t row) const
      {
        GotoEntry(row);
        return dataProvider->GetString(current,column);
      }

      Lum::Object* GetObject(size_t column, size_t row) const
      {
        GotoEntry(row);
        return dataProvider->GetObject(current,column);
      }

      // Insert
      // SwapWithNext
      Element Delete(size_t row)
      {
        GotoEntry(row);


        ElementType value=*current;

        GetContainer().erase(current);

        InvalidateCurrent();

        NotifyDelete(row,1);

        return value;
      }

      void Prepend(const Element& entry)
      {
        GetContainer().push_front(entry);
        currentPos++;
        NotifyInsert(1,1);
      }

      void Append(const Element& entry)
      {
        GetContainer().push_back(entry);

        InvalidateCurrent();

        NotifyInsert(GetContainer().size(),1);
      }

      void Clear()
      {
        InvalidateCurrent();

        GetContainer().clear();

        Notify();
      }

      Comparator* GetComparator() const
      {
        return comparator;
      }

      /**
        Sort by the given column up or down.
       */
      bool Sort(size_t /*column*/, bool /*down=true*/)
      {
        return false;
      }
    };

    template<typename Element, typename Container>
    class LUMAPI StdTableBaseSorted : public StdTableBase<Element,Container>
    {
    public:
      StdTableBaseSorted(typename StdTableBase<Element, Container>::DataProvider* dataProvider,
                         typename StdTableBase<Element, Container>::Comparator* comparator=NULL)
       : StdTableBase<Element,Container>(dataProvider,comparator)
      {
        // no code
      }

      bool Sort(size_t column, bool down=true)
      {
        if (this->GetComparator()!=NULL) {
          std::sort(this->GetContainer().begin(),
                    this->GetContainer().end(),
                    typename StdTableBase<Element, Container>::InternalComparator(this->GetComparator(),
                                                                                  column,down));
          this->InvalidateCurrent();

          this->Notify();

          return true;
        }
        else {
          return false;
        }
      }
    };

    template <typename Element>
    class LUMAPI StdTableBaseSorted<Element,std::list<Element> > : public StdTableBase<Element,std::list<Element> >
    {
    public:
      StdTableBaseSorted(typename StdTableBase<Element,std::list<Element> >::DataProvider* dataProvider,
                         typename StdTableBase<Element,std::list<Element> >::Comparator* comparator=NULL)
       : StdTableBase<Element,std::list<Element> >(dataProvider,comparator)
      {
        // no code
      }

      bool Sort(size_t column, bool down=true)
      {
        if (this->GetComparator()!=NULL) {
          this->GetContainer().sort(typename StdTableBase<Element,std::list<Element> >::InternalComparator(this->GetComparator(),
                                                                                                 column,down));
          this->InvalidateCurrent();

          this->Notify();

          return true;
        }
        else {
          return false;
        }
      }
    };

    /**
      TableModel implementation that wraps an instancance the given container of type
      Container holding elements of type Element.
     */
    template<typename Element, typename Container = std::list<Element> >
    class LUMAPI StdTable : public StdTableBaseSorted<Element,Container>
    {

    private:
      mutable typename StdTableBase<Element, Container>::ContainerType container;

    public:
      StdTable(typename StdTableBase<Element, Container>::DataProvider* dataProvider,
               typename StdTableBase<Element, Container>::Comparator* comparator=NULL)
       : StdTableBaseSorted<Element,Container>(dataProvider,comparator)
      {
        // no code
      }

      Container& GetContainer()
      {
        return container;
      }

      const Container& GetContainer() const
      {
        return container;
      }
    };

    /**
      TableModel implementation that wraps an reference to an instancance the given
      container of type Container holding elements of type Element.

      The reference to the container instance is passe din the constructor.
     */
    template<typename Element, typename Container = std::list<Element> >
    class LUMAPI StdRefTable : public StdTableBaseSorted<Element,Container>
    {

    private:
      mutable typename StdTableBase<Element, Container>::ContainerType& container;

    public:
      StdRefTable(typename StdTableBase<Element, Container>::ContainerType& container,
                  typename StdTableBase<Element, Container>::DataProvider* dataProvider,
                  typename StdTableBase<Element, Container>::Comparator* comparator=NULL)
       : StdTableBaseSorted<Element,Container>(dataProvider,comparator),
         container(container)
      {
        // no code
      }

      Container& GetContainer()
      {
        return container;
      }

      const Container& GetContainer() const
      {
        return container;
      }
    };

    /**
      Predefined table model implementation based on StdTable for a list of strings.
     */
    class LUMAPI StringTable : public StdTable<std::wstring, std::list<std::wstring> >
    {
    public:
      class StringDataProvider : public StdTable<std::wstring, std::list<std::wstring> >::DataProvider
      {
        std::wstring GetString(const StringTable::Iterator& iter, size_t /*column*/) const
        {
          return *iter;
        }
      };

    public:
      StringTable()
      : StdTable<std::wstring, std::list<std::wstring> >(new StringDataProvider())
      {
        // no code
      }
    };

    typedef Base::Reference<StringTable> StringTableRef;

    /**
      Predefined table model implementation based on StdRefTable for a list of strings.
     */
    class LUMAPI StringRefTable : public StdRefTable<std::wstring,std::list<std::wstring> >
    {
    public:
      class StringRefDataProvider : public StdRefTable<std::wstring,std::list<std::wstring> >::DataProvider
      {
        std::wstring GetString(const StringRefTable::Iterator& iter, size_t /*column*/) const
        {
          return *iter;
        }
      };

    public:
      StringRefTable(std::list<std::wstring>& container)
      : StdRefTable<std::wstring, std::list<std::wstring> >(container,new StringRefDataProvider())
      {
        // no code
      }
    };

    typedef Base::Reference<StringRefTable> StringRefTableRef;

    /**
      Predefined table model implementation based on StdTable for a list of arrays of strings
      (two dimensional dynamic array of strings).
     */
    class LUMAPI StringMatrix : public StdTable<std::vector<std::wstring>, std::list<std::vector<std::wstring> > >
    {
    public:
      class StringMatrixDataProvider : public StdTable<std::vector<std::wstring>, std::list<std::vector<std::wstring> > >::DataProvider
      {
        std::wstring GetString(const StringMatrix::Iterator& iter, size_t column) const
        {
          return (*iter)[column-1];
        }
      };

    public:
      StringMatrix()
      : StdTable<std::vector<std::wstring>, std::list<std::vector<std::wstring> > >(new StringMatrixDataProvider())
      {
        // no code
      }
    };

    typedef Base::Reference<StringMatrix> StringMatrixRef;

    /**
      Predefined table model implementation based on StdTable for a list of object (pointers).
     */
    class LUMAPI ObjectTable : public StdTable<Lum::Object*, std::list<Lum::Object*> >
    {
    public:
      class ObjectDataProvider : public StdTable<Lum::Object*, std::list<Lum::Object*> >::DataProvider
      {
        Lum::Object* GetObject(const ObjectTable::Iterator& iter, size_t /*column*/) const
        {
          return *iter;
        }
      };

    public:
      ObjectTable()
      : StdTable<Lum::Object*, std::list<Lum::Object*> >(new ObjectDataProvider())
      {
        // no code
      }
    };

    typedef Base::Reference<ObjectTable> ObjectTableRef;
  }
}

#endif
