/****************************************************************************
**
** This file is part of the Hyves PhoneGap container.
** Copyright (C) 2010-2011 Hyves (Startphone Ltd.)
** http://www.hyves.nl
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
****************************************************************************/

#ifndef MEMORYWATCHER_H
#define MEMORYWATCHER_H

#include <QObject>
#include <QTimer>


class MemoryWatcher : public QObject {

    Q_OBJECT

    public:
        explicit MemoryWatcher(QObject *parent = 0);

        /**
         * @returns memory usage in percent (0 - 100)
         */
        int usage() const;

        /**
         * @returns the amount of memory in use in bytes
         */
        int bytesUsed() const;

        /**
         * @returns the amount of free memory in bytes
         */
        int bytesFree() const;

        /**
         * @returns the memory usage threshold for the 'warning' signal to be emmitted (0 - 100)
         */
        int warningThreshold() const;

        /**
         * @returns the memory usage threshold for the 'critical' signal to be emmitted (0 - 100)
         */
        int criticalThreshold() const;

        /**
         * Sets threshold for the 'warning' signal
         * @param threshold in percent (0 - 100)
         */
        void setWarningThreshold(int threshold);

        /**
         * Sets threshold for the 'critical' signal
         * @param threshold in percent (0 - 100)
         */
        void setCriticalThreshold(int threshold);

        /**
         * Returns how many times the Hyves application got killed by Goom events
         * since last counter reset, @see clearKilledByGoom()
         */
        int killedByGoom() const;

        Q_PROPERTY(int usage READ usage);
        Q_PROPERTY(int bytesUsed READ bytesUsed);
        Q_PROPERTY(int bytesFree READ bytesFree);
        Q_PROPERTY(int warningThreshold READ warningThreshold WRITE setWarningThreshold);
        Q_PROPERTY(int criticalThreshold READ criticalThreshold WRITE setCriticalThreshold);
        Q_PROPERTY(int killedByGoom READ killedByGoom);

    signals:
        void warning(int usage);
        void critical(int usage);

    public slots:
        /**
         * Start monitoring memory
         */
        void start();

        /**
         * Stop monitoring memory
         */
        void stop();

        /**
         * Check memory manually. Will emit 'warning' or 'critical' signal if any threshold is met
         */
        void checkMemory();

        /**
         * Resets the counter for @see killedByGoom
         */
        void clearKilledByGoom();

    private:
        enum MemoryAlertState { NoAlertState = 0, WarningState, CriticalState };

        QTimer m_timer;
        int m_warningThreshold;
        int m_criticalThreshold;
        MemoryAlertState m_state;
};

#endif // MEMORYWATCHER_H
