#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <libalarm.h>
#include <libhildondesktop/libhildondesktop.h>
#include <lunar-date/lunar-date.h>
#include <glib/gi18n.h>
#include <cairo/cairo.h>

#include "hhcapplet.h"

#define HHC_ICON_LABEL          "/opt/home-calendar/background.png"
#define EXEC_PATH               "/opt/home-calendar/bin/lunar"
#define HHC_WIDTH               186
#define HHC_HEIGHT              170

#define HHC_LABEL_TOP_OFFSET    0
#define HHC_BACKGROUND_OFFSET   0

static gboolean calendar_lunar_update( calendar_lunar* obj, GError **error){
    obj->update();
    return TRUE;
}

#include "bind.h"

struct _HhcAppletPrivate
{
  GdkPixmap    *background_pixmap;
  cookie_t	resultCookie;
};

static GtkWidget    *label;

static gboolean hhc_applet_expose (GtkWidget *widget, GdkEventExpose *event);
static void     hhc_applet_update_background (HhcApplet *applet);
static void     hhc_applet_size_request (GtkWidget *widget,
                                         GtkRequisition *req);
static void     hhc_applet_size_allocate (GtkWidget *widget,
                                          GtkAllocation *alloc);
static void     hhc_applet_finalize (GObject *object);


HD_DEFINE_PLUGIN_MODULE (HhcApplet, hhc_applet, HD_TYPE_HOME_PLUGIN_ITEM)

static gboolean hhc_applet_timeout_cb ()
{
  LunarDate *date;
  GError *err=NULL;

  guint year,month,day;
  time_t timet;
  time (&timet);
  GDate *locdate = g_date_new();
  g_date_set_time_t(locdate,timet);
  year=g_date_year(locdate);
  month=g_date_month(locdate);
  day=g_date_day(locdate);

  date = lunar_date_new();
  lunar_date_set_solar_date(date, year, month, day, 1, &err);
  if (err)
  {
    g_warning (err->message);
    g_error_free (err);
    return FALSE;
  }

  gchar* leap = g_strdup_printf("%s",lunar_date_strftime(date, "%(yue)"));
  gchar *space = g_ascii_strncasecmp(leap,"*",1)?" ":"";
  gchar *holiday = g_strdup_printf("%s",lunar_date_get_jieri(date," "));
  gchar* text = g_strdup_printf("<span foreground='white' font_desc='14'>\n%u年%u月\n</span><span foreground='black' font_desc='33'>      %02u</span><span foreground='black' font_desc='15'>\n%s%s%s\n</span><span foreground='red' font_desc='15'>      %s%s</span>",
                                year,month, day,
                                lunar_date_strftime(date, "%(NIAN)%(shengxiao)年"),space,
                                lunar_date_strftime(date,"%(YUE)月%(RI)"),
                                lunar_date_strftime(date,"%(jieri)"),g_utf8_strlen(holiday,-1)>3?"...":"");

  gtk_label_set_markup (GTK_LABEL (label),text);

  lunar_date_free(date);
  g_date_free(locdate);
  g_free (leap);
  g_free (holiday);
  g_free (text);
  return TRUE;
}

static void calendar_lunar_class_init( calendar_lunar_class* klass );
static void calendar_lunar_init( calendar_lunar* obj );

GType calendar_lunar_get_type( void ){
    static GType calendar_lunar_type = 0;
    if ( !calendar_lunar_type ) {
        static const GTypeInfo calendar_lunar_info = {
            sizeof( calendar_lunar_class ),
            NULL,
            NULL,
            (GClassInitFunc)calendar_lunar_class_init,
            NULL,
            NULL,
            sizeof( calendar_lunar ),
            0,
            (GInstanceInitFunc)calendar_lunar_init
        };
        calendar_lunar_type = g_type_register_static( G_TYPE_OBJECT, "calendar_lunar", &calendar_lunar_info, 0 );
    }
    return calendar_lunar_type;
}

static void calendar_lunar_class_init( calendar_lunar_class* klass ){
    g_debug("calendar_lunar_class_init");
    dbus_g_object_type_install_info( CALENDAR_LUNAR_TYPE, &dbus_glib_calendar_lunar_object_info );
}
static void calendar_lunar_init( calendar_lunar* obj ){
    g_debug("calendar_lunar_init");

    DBusGConnection* conn;
    DBusGProxy* proxy;
    GError *error = NULL;
    guint request_name_result;

    conn = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
    proxy = dbus_g_proxy_new_for_name( conn, "org.freedesktop.DBus", "/org/freedesktop/DBus", "org.freedesktop.DBus" );
    dbus_g_proxy_call ( proxy, "RequestName", &error, G_TYPE_STRING, "org.calendar.lunar", G_TYPE_UINT, 0, G_TYPE_INVALID, G_TYPE_UINT, &request_name_result, G_TYPE_INVALID );
    dbus_g_connection_register_g_object ( conn, "/lunar", G_OBJECT(obj) );
}

calendar_lunar* calendar_lunar_new(void){
    g_debug("calendar_lunar_new");
    calendar_lunar* obj;
    obj = g_object_new( CALENDAR_LUNAR_TYPE, NULL );
    return obj;
}

static void calendar_lunar_set_key_func( calendar_lunar* conn, date_update_func func ){
    conn->update = func;
}

static gboolean
open_calendar(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  if(event->type == GDK_BUTTON_RELEASE){
    hhc_applet_timeout_cb ();
    char *argv[]={EXEC_PATH,NULL};
    g_spawn_async("/home/user",argv,NULL,G_SPAWN_SEARCH_PATH,NULL,NULL,NULL,NULL);
  }else if(event->type == GDK_LEAVE_NOTIFY){
    hhc_applet_timeout_cb ();
  }else if(event->type == GDK_BUTTON_PRESS){
    HhcAppletPrivate     *priv;
    priv = HHC_APPLET (widget)->priv;

    if (GTK_WIDGET_DRAWABLE (widget)){
      cairo_t                  *cr;
      cr = gdk_cairo_create (widget->window);
      cairo_set_source_rgba (cr, 1.0, 0.6, 0.2, 0.5);
      cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
      cairo_paint_with_alpha (cr,0.4);
      cairo_destroy (cr);
    }
  }
  return FALSE;
}

static calendar_lunar *con;

static void
hhc_applet_init (HhcApplet *applet)
{
  setlocale (LC_ALL, "");
  HhcAppletPrivate     *priv;
  GdkColormap          *colormap = NULL;

  if(!con)
    con = calendar_lunar_new();
  calendar_lunar_set_key_func(con, hhc_applet_timeout_cb);
  applet->priv = priv =
      G_TYPE_INSTANCE_GET_PRIVATE (applet, HHC_TYPE_APPLET, HhcAppletPrivate);

  label = gtk_label_new ("");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.08);
  gtk_misc_set_padding (GTK_MISC (label),
                        8,
                        0);

  alarm_event_t *newEvent = 0;
  alarm_recur_t *recur = 0;
  alarm_action_t *act = 0;
  cookie_t *list = 0;
  cookie_t cookie = 0;
 
  if( (list = alarmd_event_query(0,0, 0,0, "Lunar")) == 0 )
  {
    printf("query failed\n");
    goto cleanup;
  }
  int i;
  for( i = 0; list[i] != 0; ++i )
  {
    if(i) alarmd_event_del(cookie);
    else cookie = list[i];
  }
 
  cleanup:
    free(list);
 
  /* AlarmD will return a unique "alarm cookie" if successful; you can use this
   * later to identify and manipulate your alarm, or just ignore it. */
  if(!cookie){
  newEvent = alarm_event_create();
  alarm_event_set_alarm_appid(newEvent, "Lunar");
  alarm_event_set_title(newEvent, "LunarCalendar");

  newEvent->recur_secs = 0;
  newEvent->recur_count = -1;

  recur = alarm_event_add_recurrences(newEvent,1);
  recur->mask_min |= 1; // 1<<0
  recur->mask_hour |= 1; // 1<<0
  recur->mask_mday |= ALARM_RECUR_MDAY_ALL;
  recur->special |= ALARM_RECUR_SPECIAL_NONE;
 
  act = alarm_event_add_actions(newEvent, 1);
  act->flags = ALARM_ACTION_WHEN_TRIGGERED | ALARM_ACTION_DBUS_USE_ACTIVATION | ALARM_ACTION_TYPE_DBUS;
  alarm_action_set_dbus_interface(act, "org.calendar.lunar");
  alarm_action_set_dbus_service(act, "org.calendar.lunar");
  alarm_action_set_dbus_path(act, "/lunar");
  alarm_action_set_dbus_name(act, "update");
 
  priv->resultCookie = alarmd_event_add(newEvent);
  alarm_recur_delete(recur);
  alarm_event_delete(newEvent);
  }
  gtk_widget_show (label);

  gtk_container_add (GTK_CONTAINER (applet), label);
  g_signal_connect(G_OBJECT(applet),"event",G_CALLBACK(open_calendar),priv);
  /* declare the widget as having a RGBA colormap, so that we can draw
   * transparent stuff */
  colormap = gdk_screen_get_rgba_colormap (gdk_screen_get_default ());

  if (colormap)
    gtk_widget_set_colormap (GTK_WIDGET (applet), colormap);
  hhc_applet_timeout_cb ();
}

static void
hhc_applet_class_init (HhcAppletClass *klass)
{
  GtkWidgetClass       *widget_class;
  GObjectClass         *object_class;

  widget_class = GTK_WIDGET_CLASS (klass);
  object_class = G_OBJECT_CLASS (klass);

  object_class->finalize = hhc_applet_finalize;

  widget_class->expose_event  = hhc_applet_expose;
  widget_class->size_allocate = hhc_applet_size_allocate;
  widget_class->size_request  = hhc_applet_size_request;

  g_type_class_add_private (klass, sizeof (HhcAppletPrivate));

}

static void
hhc_applet_finalize (GObject *object)
{
  HhcAppletPrivate     *priv = HHC_APPLET (object)->priv;

  if (priv->resultCookie) {
    alarmd_event_del(priv->resultCookie);
  }

  if (priv->background_pixmap)
  {
    g_object_unref (priv->background_pixmap);
    priv->background_pixmap = NULL;
  }
}

static void
hhc_applet_class_finalize (HhcAppletClass *class) {}

static gboolean
hhc_applet_expose (GtkWidget *widget, GdkEventExpose *event)
{
  HhcAppletPrivate     *priv;

  priv = HHC_APPLET (widget)->priv;

  if (GTK_WIDGET_DRAWABLE (widget))
  {
    cairo_t                  *cr;

    if (!priv->background_pixmap)
      hhc_applet_update_background (HHC_APPLET (widget));

    cr = gdk_cairo_create (widget->window);

    /* Draw the background, we can use the SOURCE operator to
     * be a tad faster */
    gdk_cairo_set_source_pixmap (cr,
                                 priv->background_pixmap,
                                 HHC_BACKGROUND_OFFSET,
                                 HHC_BACKGROUND_OFFSET);
    cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
    cairo_paint_with_alpha (cr, 0.8);

    cairo_destroy (cr);

    /* Set app_paintable flag on. This prevents widget's default expose-event
    to overwrite background with default content */
    gtk_widget_set_app_paintable(widget, TRUE);

    /* Finally let GTK draw the label */
    return GTK_WIDGET_CLASS (hhc_applet_parent_class)->expose_event (widget,
                                                                     event);

  }

  return FALSE;

}

static void
hhc_applet_size_allocate (GtkWidget *widget, GtkAllocation *alloc)
{
  GTK_WIDGET_CLASS (hhc_applet_parent_class)->size_allocate (widget, alloc);

}

static void
hhc_applet_size_request (GtkWidget *widget, GtkRequisition *requisition)
{
  GdkPixbuf            *pixbuf;
  HhcAppletPrivate     *priv;
  GError               *error = NULL;
  priv = HHC_APPLET (widget)->priv;

  pixbuf = gdk_pixbuf_new_from_file (HHC_ICON_LABEL, &error);
  if (error)
  {
    g_warning (error->message);
    g_error_free (error);
    requisition->width = HHC_WIDTH;
    requisition->height = HHC_HEIGHT;
    return;
  }
  requisition->width = gdk_pixbuf_get_width (pixbuf);
  requisition->height = gdk_pixbuf_get_height (pixbuf);

  g_object_unref (pixbuf);
}

static void
hhc_applet_update_background (HhcApplet *applet)
{
  HhcAppletPrivate     *priv = applet->priv;
  GtkWidget            *widget;
  GdkPixbuf            *pixbuf;
  GError               *error = NULL;
  cairo_t              *cr;

  widget = GTK_WIDGET (applet);

  pixbuf = gdk_pixbuf_new_from_file (HHC_ICON_LABEL, &error);
  if (error)
  {
    g_warning (error->message);
    g_error_free (error);
    return;
  }

  if (priv->background_pixmap)
    g_object_unref (priv->background_pixmap);

  priv->background_pixmap = gdk_pixmap_new (widget->window,
                                            gdk_pixbuf_get_width (pixbuf),
                                            gdk_pixbuf_get_height (pixbuf),
                                            32);
  g_return_if_fail (priv->background_pixmap);

  /* Initialize the pixmap with a full empty transparent rectangle */
  cr = gdk_cairo_create (priv->background_pixmap);
  cairo_set_source_rgba (cr, 0.0, 0.0, 0.0, 0.0);
  cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
  cairo_paint (cr);
  cairo_destroy (cr);

  gdk_draw_pixbuf (priv->background_pixmap,
                   NULL,
                   pixbuf,
                   0, 0,
                   0, 0,
                   gdk_pixbuf_get_width (pixbuf),
                   gdk_pixbuf_get_height (pixbuf),
                   GDK_RGB_DITHER_NONE,
                   0, 0);

  g_object_unref (pixbuf);
}

