/****************************************************************************

    FILE:    mainwindow.cpp

 ****************************************************************************

    LICENSE:

    HexTool -  a simple program to convert numbers on the fly
    Copyright (C) 2003 Ryan Harkin and 2014 Christian Weniger; www.harkin.org/HexTool

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2 as
    published by the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/
#include <stdio.h>

#include <common/debug.h>
#include <common/handlebinarydigit.h>
#include <common/numbits.h>
#include <common/register.h>
#include <common/registeroperation.h>
#include <common/signed.h>
#include <common/value.h>
#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QtCore/QCoreApplication>
//#include <qmath.h>
#include <QMessageBox>

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent), ui(new Ui::MainWindow)
{
    SetSigned(UNSIGNED);
    SetNumBits(NUMBITS_32);
    SetRegister(REG_A);
    ui->setupUi(this);

    ui->buttonGroup->setId(ui->pushButton_bit1, 1);
    ui->buttonGroup->setId(ui->pushButton_bit2, 2);
    ui->buttonGroup->setId(ui->pushButton_bit3, 3);
    ui->buttonGroup->setId(ui->pushButton_bit4, 4);
    ui->buttonGroup->setId(ui->pushButton_bit5, 5);
    ui->buttonGroup->setId(ui->pushButton_bit6, 6);
    ui->buttonGroup->setId(ui->pushButton_bit7, 7);
    ui->buttonGroup->setId(ui->pushButton_bit8, 8);
    ui->buttonGroup->setId(ui->pushButton_bit9, 9);
    ui->buttonGroup->setId(ui->pushButton_bit10, 10);
    ui->buttonGroup->setId(ui->pushButton_bit11, 11);
    ui->buttonGroup->setId(ui->pushButton_bit12, 12);
    ui->buttonGroup->setId(ui->pushButton_bit13, 13);
    ui->buttonGroup->setId(ui->pushButton_bit14, 14);
    ui->buttonGroup->setId(ui->pushButton_bit15, 15);
    ui->buttonGroup->setId(ui->pushButton_bit16, 16);
    ui->buttonGroup->setId(ui->pushButton_bit17, 17);
    ui->buttonGroup->setId(ui->pushButton_bit18, 18);
    ui->buttonGroup->setId(ui->pushButton_bit19, 19);
    ui->buttonGroup->setId(ui->pushButton_bit20, 20);
    ui->buttonGroup->setId(ui->pushButton_bit21, 21);
    ui->buttonGroup->setId(ui->pushButton_bit22, 22);
    ui->buttonGroup->setId(ui->pushButton_bit23, 23);
    ui->buttonGroup->setId(ui->pushButton_bit24, 24);
    ui->buttonGroup->setId(ui->pushButton_bit25, 25);
    ui->buttonGroup->setId(ui->pushButton_bit26, 26);
    ui->buttonGroup->setId(ui->pushButton_bit27, 27);
    ui->buttonGroup->setId(ui->pushButton_bit28, 28);
    ui->buttonGroup->setId(ui->pushButton_bit29, 29);
    ui->buttonGroup->setId(ui->pushButton_bit30, 30);
    ui->buttonGroup->setId(ui->pushButton_bit31, 31);
    ui->buttonGroup->setId(ui->pushButton_bit32, 32);
    /*ui->rbuttonGroup->setId(ui->radioButton_8bit, 8);
    ui->rbuttonGroup->setId(ui->radioButton_16bit, 16);
    ui->rbuttonGroup->setId(ui->radioButton_32bit, 32);*/
    connect(ui->buttonGroup, SIGNAL(buttonClicked(int)), this, SLOT(groupButton_clicked(int)));
    UpdateDisplay();
    QAction *exit = new QAction("Exit", this);
    QAction *about = new QAction("About",this);
    this->menuBar()->addAction(exit);
    this->menuBar()->addAction(about);
    connect(exit, SIGNAL(triggered()), this, SLOT(close()));
    connect(about, SIGNAL(triggered()), this, SLOT(about()));

}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::setOrientation(ScreenOrientation orientation)
{
#if defined(Q_OS_SYMBIAN)
    // If the version of Qt on the device is < 4.7.2, that attribute won't work
    if (orientation != ScreenOrientationAuto) {
        const QStringList v = QString::fromAscii(qVersion()).split(QLatin1Char('.'));
        if (v.count() == 3 && (v.at(0).toInt() << 16 | v.at(1).toInt() << 8 | v.at(2).toInt()) < 0x040702) {
            qWarning("Screen orientation locking only supported with Qt 4.7.2 and above");
            return;
        }
    }
#endif // Q_OS_SYMBIAN

    Qt::WidgetAttribute attribute;
    switch (orientation) {
#if QT_VERSION < 0x040702
    // Qt < 4.7.2 does not yet have the Qt::WA_*Orientation attributes
    case ScreenOrientationLockPortrait:
        attribute = static_cast<Qt::WidgetAttribute>(128);
        break;
    case ScreenOrientationLockLandscape:
        attribute = static_cast<Qt::WidgetAttribute>(129);
        break;
    default:
    case ScreenOrientationAuto:
        attribute = static_cast<Qt::WidgetAttribute>(130);
        break;
#else // QT_VERSION < 0x040702
    case ScreenOrientationLockPortrait:
        attribute = Qt::WA_LockPortraitOrientation;
        break;
    case ScreenOrientationLockLandscape:
        attribute = Qt::WA_LockLandscapeOrientation;
        break;
    default:
    case ScreenOrientationAuto:
        attribute = Qt::WA_AutoOrientation;
        break;
#endif // QT_VERSION < 0x040702
    };
    setAttribute(attribute, true);
}

void MainWindow::showExpanded()
{
#if defined(Q_OS_SYMBIAN) || defined(Q_WS_SIMULATOR)
    showFullScreen();
#elif defined(Q_WS_MAEMO_5)
    showMaximized();
#else
    show();
#endif
}
void MainWindow::UpdateDisplay()
{
    static bool updatingDisplay = false;

    if (!updatingDisplay)
    {
    updatingDisplay = true;


    int value = GetValue(REG_CURRENT);

    char str[128];
    if (GetSigned()==UNSIGNED)
        sprintf(str, "%u", value);
    else
    {
         switch (GetNumBits())
         {
             case NUMBITS_8:     sprintf(str, "%d", (char) value);    break;
             case NUMBITS_16:    sprintf(str, "%d", (short)value);    break;
             case NUMBITS_32:    sprintf(str, "%d", (int)  value);    break;
         }
    }
    ui->lineEdit_dec->setText(str);
    sprintf(str, "%X", value);
    ui->lineEdit_hex->setText(str);

    /*ui->radioButton_8bit-> setChecked(false);
    ui->radioButton_16bit->setChecked(false);
    ui->radioButton_32bit->setChecked(false);

    switch (GetNumBits())
    {
        case NUMBITS_8:  ui->radioButton_8bit-> setChecked(true); break;
        case NUMBITS_16: ui->radioButton_16bit->setChecked(true); break;
        case NUMBITS_32: ui->radioButton_32bit->setChecked(true); break;
    }

    ui->radioButton_registerA->setChecked(false);
    ui->radioButton_registerB->setChecked(false);
    ui->radioButton_registerC->setChecked(false);
    switch (GetRegister())
    {
    default:
    case REG_A:
        ui->radioButton_registerA->setChecked(true);
        break;
    case REG_B:
        ui->radioButton_registerB->setChecked(true);
        break;
    case REG_C:
        ui->radioButton_registerC->setChecked(true);
        break;
    }*/

    sprintf(str, "%d", (unsigned char)((value & 0x000000FF) >> 0));
    ui->lineEdit_ipaddress0->setText(str);
    sprintf(str, "%d", (unsigned char)((value & 0x0000FF00) >> 8));
    ui->lineEdit_ipaddress1->setText(str);
    sprintf(str, "%d", (unsigned char)((value & 0x00FF0000) >> 16));
    ui->lineEdit_ipaddress2->setText(str);
    sprintf(str, "%d", (unsigned char)((value & 0xFF000000) >> 24));
    ui->lineEdit_ipaddress3->setText(str);

    sprintf(str, "%c", (unsigned char)((value & 0x000000FF) >> 0));
    ui->lineEdit_ascii0->setText(str);
    sprintf(str, "%c", (unsigned char)((value & 0x0000FF00) >> 8));
    ui->lineEdit_ascii1->setText(str);
    sprintf(str, "%c", (unsigned char)((value & 0x00FF0000) >> 16));
    ui->lineEdit_ascii2->setText(str);
    sprintf(str, "%c", (unsigned char)((value & 0xFF000000) >> 24));
    ui->lineEdit_ascii3->setText(str);

    ////////////////////////////////////////////////////////////////////////
    int bitVal;

    for (int bit=0; bit<8; bit++)
    {
        bitVal = (value & (1<<bit))?1:0;
        sprintf(str, "%d", bitVal);
        ui->buttonGroup->button(bit+1)->setText(str);
    }



    if (GetNumBits() != 8)
    {

        ui->frame_bit16->setDisabled(false);
        ui->lineEdit_ascii1 ->setDisabled(false);
        ui->lineEdit_ipaddress1 ->setDisabled(false);

        for (int bit=8; bit<16; bit++)
        {
            bitVal = (value & (1<<bit))?1:0;
            sprintf(str, "%d", bitVal);
            ui->buttonGroup->button(bit+1)->setText(str);
        }



        if (GetNumBits() != 16)
        {

            ui->frame_bit32->setDisabled(false);

            ui->lineEdit_ascii3->setDisabled(false);
            ui->lineEdit_ascii2->setDisabled(false);
            ui->lineEdit_ipaddress3->setDisabled(false);
            ui->lineEdit_ipaddress2->setDisabled(false);

            for (int bit=16; bit<32; bit++)
            {
                bitVal = (value & (1<<bit))?1:0;
                sprintf(str, "%d", bitVal);
                ui->buttonGroup->button(bit+1)->setText(str);
            }


        }
        else
        {

            for (int i=1; i<=16; i++)
            {
                ui->buttonGroup->button(i+16)->setText("");
            }


            ui->frame_bit32->setDisabled(true);

            ui->lineEdit_ascii3->setDisabled(true);
            ui->lineEdit_ascii2->setDisabled(true);
            ui->lineEdit_ipaddress3->setDisabled(true);
            ui->lineEdit_ipaddress2->setDisabled(true);
        }
    }
    else
    {

        for (int i=1; i<=24; i++)
        {
            ui->buttonGroup->button(i+8)->setText("");
        }

        ui->frame_bit16->setDisabled(true);
        ui->frame_bit32->setDisabled(true);
        ui->lineEdit_ascii1->setDisabled(true);
        ui->lineEdit_ipaddress1->setDisabled(true);
    }
    updatingDisplay = false;
    }
}

void MainWindow::ToggleBit(int bit)
{
    HandleBinaryDigit(bit);
    UpdateDisplay();
}

void MainWindow::groupButton_clicked(int bit) { ToggleBit(bit-1); }

/****************************************************************************

    Qt Signal Handler

    control:    lineEdit_dec
    signal:     textChanged

    The decimal number edit box has changed

****************************************************************************/
void MainWindow::on_lineEdit_dec_textChanged()
{
    int value;
    if (GetSigned()==UNSIGNED)
    {
        value = (int)ui->lineEdit_dec->text().toUInt();
    }
    else
    {
        value = (int)ui->lineEdit_dec->text().toInt();
    }
    SetValue(value, REG_CURRENT);
    UpdateDisplay();
}

/****************************************************************************

    Qt Signal Handler

    control:    lineEdit_hex
    signal:     textChanged

    The hexadecimal number edit box has changed

****************************************************************************/
void MainWindow::on_lineEdit_hex_textChanged()
{
    int value;
    value = (int)ui->lineEdit_hex->text().toUInt(0,16);
    SetValue(value, REG_CURRENT);
    UpdateDisplay();
}

/****************************************************************************

    Qt Signal Handler

    control:    radioButton_8bit
    signal:     pressed

    The radio button for 8 bit mode has been selected (aka pressed)

****************************************************************************/
void MainWindow::on_radioButton_8bit_clicked()
{
    SetNumBits(NUMBITS_8);
    UpdateDisplay();
    ui->lineEdit_hex->setMaxLength(2);
    if (ui->checkBox_signed->checkState()) ui->lineEdit_dec->setMaxLength(4);
    else ui->lineEdit_dec->setMaxLength(3);
}

/****************************************************************************

    Qt Signal Handler

    control:    radioButton_16bit
    signal:     pressed

    The radio button for 16 bit mode has been selected (aka pressed)

****************************************************************************/
void MainWindow::on_radioButton_16bit_clicked()
{
    SetNumBits(NUMBITS_16);
    UpdateDisplay();
    ui->lineEdit_hex->setMaxLength(4);
    if (ui->checkBox_signed->checkState()) ui->lineEdit_dec->setMaxLength(6);
    else ui->lineEdit_dec->setMaxLength(5);
}

/****************************************************************************

    Qt Signal Handler

    control:    radioButton_32bit
    signal:     pressed

    The radio button for 32 bit mode has been selected (aka pressed)

****************************************************************************/
void MainWindow::on_radioButton_32bit_clicked()
{
    SetNumBits(NUMBITS_32);
    UpdateDisplay();
    ui->lineEdit_hex->setMaxLength(8);
    if (ui->checkBox_signed->checkState()) ui->lineEdit_dec->setMaxLength(11);
    else ui->lineEdit_dec->setMaxLength(10);
}
/****************************************************************************

    Qt Signal Handler

    control:    pushButton_not
    signal:     pressed

    The NOT button has been pressed

****************************************************************************/
void MainWindow::on_pushButton_not_pressed()
{
    RegisterOperation(REG_CURRENT, REG_CURRENT, REG_CURRENT, OP_INVERSE);
    UpdateDisplay();
}

/****************************************************************************

   Qt Signal Handler

    control:    checkBox_signed
    signal:     toggled

    The checkbox for signed has been toggled for signed/unsigned operation

****************************************************************************/
void MainWindow::on_checkBox_signed_toggled(bool isChecked)
{
    eSigned sig;

    if (isChecked)
        sig=SIGNED;
    else
        sig=UNSIGNED;

    SetSigned(sig);
    if (isChecked)
    {
        switch (GetNumBits())
        {
            case NUMBITS_8:  ui->lineEdit_dec->setMaxLength(4); break;
            case NUMBITS_16: ui->lineEdit_dec->setMaxLength(6); break;
            case NUMBITS_32: ui->lineEdit_dec->setMaxLength(11); break;
        }
    }
    UpdateDisplay();
    if (!isChecked)
    {
        switch (GetNumBits())
        {
            case NUMBITS_8:  ui->lineEdit_dec->setMaxLength(3); break;
            case NUMBITS_16: ui->lineEdit_dec->setMaxLength(5); break;
            case NUMBITS_32: ui->lineEdit_dec->setMaxLength(10); break;
        }
    }
}

/****************************************************************************

    Qt Signal Handler

    control:    pushButton_shiftRight
    signal:     pressed

    The right shift button has been pressed to shift the number 1 place

****************************************************************************/
void MainWindow::on_pushButton_shiftLeft_pressed()
{
    RegisterOperation(REG_CURRENT, REG_CURRENT, REG_CURRENT, OP_LEFT_SHIFT);
    UpdateDisplay();
}

/****************************************************************************

    Qt Signal Handler

    control:    pushButton_shiftRight
    signal:     pressed

    The right shift button has been pressed to shift the number 1 place

****************************************************************************/
void MainWindow::on_pushButton_shiftRight_pressed()
{
    RegisterOperation(REG_CURRENT, REG_CURRENT, REG_CURRENT, OP_RIGHT_SHIFT);
    UpdateDisplay();
}

/****************************************************************************

    Qt Signal Handler

    control:    pushButton_bitNumbering
    signal:     pressed

    Toggle the labelling of the bits from zero based to 1 based

****************************************************************************/
void MainWindow::on_pushButton_bitNumbering_pressed()
{
    static bool zeroBased = false;

    if (zeroBased)
    {
        zeroBased = false;
        ui->pushButton_bitNumbering->setText("1");
        ui->label_bit4->setText("4");
        ui->label_bit8->setText("8");
        ui->label_bit12->setText("12");
        ui->label_bit16->setText("16");
        ui->label_bit20->setText("20");
        ui->label_bit24->setText("24");
        ui->label_bit28->setText("28");
        ui->label_bit32->setText("32");
    }
    else
    {
        zeroBased = true;
        ui->pushButton_bitNumbering->setText("0");
        ui->label_bit4->setText("3");
        ui->label_bit8->setText("7");
        ui->label_bit12->setText("11");
        ui->label_bit16->setText("15");
        ui->label_bit20->setText("21");
        ui->label_bit24->setText("23");
        ui->label_bit28->setText("27");
        ui->label_bit32->setText("31");
    }
}

/****************************************************************************

    Qt Signal Handler

    control:    pushButton_endianSwap
    signal:     pressed

    Toggles the endian between Big and Little endian

****************************************************************************/
void MainWindow::on_pushButton_endianSwap_pressed()
{
    unsigned int value = GetValue(REG_CURRENT);
    unsigned int newValue = 0;

    switch (GetNumBits())
    {
      case 32:
        {
          newValue |= (value & 0x000000ff) << 24;
          newValue |= (value & 0x0000ff00) <<  8;
          newValue |= (value & 0x00ff0000) >>  8;
          newValue |= (value & 0xff000000) >> 24;
        }
        break;

      case 16:
        {
          newValue |= (value & 0x000000ff) << 8;
          newValue |= (value & 0x0000ff00) >> 8;
        }
        break;

      default:
      case 8:
        // don't do anything!
        newValue = value;
        break;
    }
    SetValue(newValue, REG_CURRENT);
    UpdateDisplay();
}

/****************************************************************************

    Qt Signal Handler

    control:    pushButton increase and decrease
    signal:     clicked



****************************************************************************/
void MainWindow::on_pushButton_increase_clicked()
{
    int value = GetValue(REG_CURRENT);
    value++;
    SetValue(value, REG_CURRENT);
    UpdateDisplay();
}

void MainWindow::on_pushButton_decrease_clicked()
{
    int value = GetValue(REG_CURRENT);
    value--;
    SetValue(value, REG_CURRENT);
    UpdateDisplay();
}


/****************************************************************************

    Qt Signal Handler

    control:    radioButton_register[A|B|C]
    signal:     clicked

    Select the current register that is being displayed

****************************************************************************/
void MainWindow::on_radioButton_registerA_clicked()
{
    SetRegister(REG_A);
    UpdateDisplay();
}

void MainWindow::on_radioButton_registerB_clicked()
{
    SetRegister(REG_B);
    UpdateDisplay();
}

void MainWindow::on_radioButton_registerC_clicked()
{
    SetRegister(REG_C);
    UpdateDisplay();
}

/****************************************************************************

    This is the generic function to handle register operations after the user
    has clicked on one of the buttons.

****************************************************************************/
int MainWindow::HandleOperation
(
    eOpType op
)
{
    // Perform the operation on the registers
    int retVal = RegisterOperation(REG_A, REG_B, REG_C, op);

    // change the display to show register C
    SetRegister(REG_C);
    ui->radioButton_registerC->setChecked(true);

    // Update the display
    UpdateDisplay();

    return (retVal);
}

/****************************************************************************

    Qt Signal Handler

    control:    pushButton_XXX
    signal:     pressed

    Where XXX is a register operation like AND, OR, XOR, ...
    These handlers just pass the processing to the generic HandleOperation
    function.

****************************************************************************/
void MainWindow::on_pushButton_AND_pressed()
{
    HandleOperation(OP_AND);
}

void MainWindow::on_pushButton_OR_pressed()
{
    HandleOperation(OP_OR);
}

void MainWindow::on_pushButton_XOR_pressed()
{
    HandleOperation(OP_XOR);
}

void MainWindow::on_pushButton_plus_pressed()
{
    HandleOperation(OP_ADD);
}

void MainWindow::on_pushButton_minus_pressed()
{
    HandleOperation(OP_SUBTRACT);
}

void MainWindow::on_pushButton_multiply_pressed()
{
    HandleOperation(OP_MULTIPLY);
}

void MainWindow::on_pushButton_divide_pressed()
{
    HandleOperation(OP_DIVIDE);
}




/****************************************************************************

    This generic handler deals with the user editing the IP Address edit
    boxes.
    It will automatically move the focus to the next control if the user
    enters a 3 digit number, because each part of an IP address is always
    in the range 0...255.
    Users are also forbidden from entering numbers greater than 255.

****************************************************************************/
void MainWindow::HandleIpAddrEditBox
(
    int field,
    unsigned int val
)
{
    if (field < 4)
    {
        if ((val & 0xFFFFFF00) == 0)
        {
            int currentValue    = GetValue(REG_CURRENT);    // the current value stored in the current register
            int newValue        = 0;                        // the new value that we want to to set the register to

            switch (field)   // which byte was modified (i.e. which ASCI edit box)
            {
                case 0: newValue = (currentValue & 0xFFFFFF00) | (val      ); break;
                case 1: newValue = (currentValue & 0xFFFF00FF) | (val <<  8); break;
                case 2: newValue = (currentValue & 0xFF00FFFF) | (val << 16); break;
                case 3: newValue = (currentValue & 0x00FFFFFF) | (val << 24); break;
            }

            SetValue(newValue, REG_CURRENT);
            UpdateDisplay();

            // If the value has changed, skip to the next character
            if ((newValue != currentValue) && (val > 99))
            {
                QLineEdit * nextCtrl;
                switch (field)
                {
                case 0:
                    switch (GetNumBits())
                    {
                    case NUMBITS_8:  nextCtrl = ui->lineEdit_ipaddress0; break;
                    case NUMBITS_16: nextCtrl = ui->lineEdit_ipaddress1; break;
                    case NUMBITS_32: nextCtrl = ui->lineEdit_ipaddress3; break;
                    }
                    break;
                case 1: nextCtrl = ui->lineEdit_ipaddress0; break;
                case 2: nextCtrl = ui->lineEdit_ipaddress1; break;
                case 3: nextCtrl = ui->lineEdit_ipaddress2; break;
                }
                nextCtrl->setFocus();
                nextCtrl->setSelection(0,10);
            }
        }
        else
        {
            // refuse to accept more than 255 in an IP Addr edit box
            UpdateDisplay();
        }
    }
}


/****************************************************************************

    Qt Signal Handler

    control:    lineEdit_ipaddressX
    signal:     textChanged

    When the users edits the ip address fields, these handlers pass control
    to the generic HandleIpAddrEditBox function.

****************************************************************************/
void MainWindow::on_lineEdit_ipaddress0_textChanged()
{
    HandleIpAddrEditBox(0, ui->lineEdit_ipaddress0->text().toUInt());
}

void MainWindow::on_lineEdit_ipaddress1_textChanged()
{
    HandleIpAddrEditBox(1, ui->lineEdit_ipaddress1->text().toUInt());
}

void MainWindow::on_lineEdit_ipaddress2_textChanged()
{
    HandleIpAddrEditBox(2, ui->lineEdit_ipaddress2->text().toUInt());
}

void MainWindow::on_lineEdit_ipaddress3_textChanged()
{
    HandleIpAddrEditBox(3, ui->lineEdit_ipaddress3->text().toUInt());
}


/****************************************************************************

    The generic handler called when the users types into the ASCII edit boxes.
    The UI is programmed to only allow 1 character using the form designer.
    This function automatically moves focus to the next ASCII edit control,
    wrapping back to the first.

****************************************************************************/
void MainWindow::HandleAsciiEditBox
(
    int field,
    unsigned int val
)
{
    if (field < 4)
    {
        int newByte         = val & 0x000000FF;// the byte that was just typed in the ASCII edit box
        int currentValue    = GetValue(REG_CURRENT);    // the current value stored in the current register
        int newValue        = 0;                        // the new value that we want to to set the register to

        switch (field)   // which byte was modified (i.e. which ASCI edit box)
        {
            case 0: newValue = (currentValue & 0xFFFFFF00) | (newByte      ); break;
            case 1: newValue = (currentValue & 0xFFFF00FF) | (newByte <<  8); break;
            case 2: newValue = (currentValue & 0xFF00FFFF) | (newByte << 16); break;
            case 3: newValue = (currentValue & 0x00FFFFFF) | (newByte << 24); break;
        }

        SetValue(newValue, REG_CURRENT);
        UpdateDisplay();

        // If the value has changed, skip to the next character
        if (newValue != currentValue)
        {
            QLineEdit * nextCtrl;
            switch (field)
            {
            case 0:
                switch (GetNumBits())
                {
                case NUMBITS_8:  nextCtrl = ui->lineEdit_ascii0; break;
                case NUMBITS_16: nextCtrl = ui->lineEdit_ascii1; break;
                case NUMBITS_32: nextCtrl = ui->lineEdit_ascii3; break;
                }
                break;
            case 1: nextCtrl = ui->lineEdit_ascii0; break;
            case 2: nextCtrl = ui->lineEdit_ascii1; break;
            case 3: nextCtrl = ui->lineEdit_ascii2; break;
            }
            nextCtrl->setFocus();
            nextCtrl->setSelection(0,1);
        }
    }
}
/****************************************************************************

    Qt Signal Handler

    control:    lineEdit_asciiX
    signal:     textChanged

    Passes control on to a generic handler after extracting the new ASCII
    value that was entered.

****************************************************************************/
void MainWindow::on_lineEdit_ascii0_textChanged()
{
    QString text;
    text = ui->lineEdit_ascii0->text();
    int val = (int)text[0].unicode();
    HandleAsciiEditBox(0, val);
}
void MainWindow::on_lineEdit_ascii1_textChanged()
{
    QString text;
    text = ui->lineEdit_ascii1->text();
    int val = (int)text[0].unicode();
    HandleAsciiEditBox(1, val);
}
void MainWindow::on_lineEdit_ascii2_textChanged()
{
    QString text;
    text = ui->lineEdit_ascii2->text();
    int val = (int)text[0].unicode();
    HandleAsciiEditBox(2, val);
}
void MainWindow::on_lineEdit_ascii3_textChanged()
{
    QString text;
    text = ui->lineEdit_ascii3->text();
    int val = (int)text[0].unicode();
    HandleAsciiEditBox(3, val);
}
void MainWindow::about()
{
    QMessageBox::about(this,tr("HexTool Maemo Version 1.7.1"),tr("Ported and code changes by Halftux.\nBased on Version 1.7.0.0 by Ryan Harkin."));
}
