import re

class Provider:
    """The notional `Provider' which defines the interface which all service
       providers for Hermes must meet. Implementations of this class are expected
       to load their preferences on construction, display them through open_preferences
       and pass them to the service object as necessary. 
       
       Copyright (c) Andrew Flegg <andrew@bleb.org> 2010.
       Released under the Artistic Licence."""


    # -----------------------------------------------------------------------
    def get_name(self):
        """Return the display name of this service. An icon, of with the lower-case,
           all-alphabetic version of this name is expected to be provided."""
           
        return None


    # -----------------------------------------------------------------------
    def get_id(self):
        """Return the ID of this service. This should be alphanumeric characters
           only. The default implementation is the lowercase version of get_name()
           with all non-alphanumeric characters removed.
           
           Overridding this method is NOT recommended."""
           
        return re.sub('[^a-z0-9]+', '', self.get_name().lower())


    # -----------------------------------------------------------------------
    def get_account_detail(self):
        """Return information on the configured account. This would, typically,
           be the user's login for the service. If None is returned, no futher
           information is available."""
           
        return None
    
    
    # -----------------------------------------------------------------------
    def has_preferences(self):
        """Whether or not this provider has any preferences. If it does not,
           open_preferences must NOT be called; as the behaviour is undetermined."""
           
        return False
    
    
    # -----------------------------------------------------------------------
    def open_preferences(self, parent):
        """Open the preferences for this provider as a child of the 'parent' widget.
           The dialogue box should have a _('Disable') and _('Enable') button; and
           return True if the user chooses to enable the service; False if not.
           If the user cancels the dialogue, return None."""
        
        pass

    
    # -----------------------------------------------------------------------
    def service(self, gui_callback):
        """Return the service backend. This must be a class which implements the
           following methods:
               * get_friends
               * process_contact
               * finalise
        
           See Service for more details."""
           
        return None
