//    Group SMS
//    Copyright (C) 2011 Paolo Iommarini
//    sakya_tg@yahoo.it
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#include "mainwindow.h"
#include "ui_mainwindow.h"
#include <QDebug>
#include <QSqlQuery>
#include <QFile>
#include <QLabel>
#include <QMenuBar>
#include <QMenu>
#include <QTimer>
#ifdef Q_WS_MAEMO_5
#include "hildon-uri.h"
#endif
#include "commonstatic.h"
#include "editgroupwindow.h"
#include "groupactionswindow.h"
#include "aboutwindow.h"

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);    
    setWindowTitle("Group SMS");
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5NonComposited, true);
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
#endif

    QMenu *menu = menuBar()->addMenu(tr("Group SMS"));
    QAction* action = new QAction(tr("About"), this);
    connect(action, SIGNAL(triggered()), this, SLOT(AboutSlot()));
    menu->addAction(action);

    action = new QAction(tr("Exit"), this);
    connect(action, SIGNAL(triggered()), this, SLOT(ExitSlot()));
    menu->addAction(action);

    connect(ui->btnNewGroup, SIGNAL(clicked()), this, SLOT(NewGroupSlot()));

    m_GroupMenu = new QMenu(this);
    action = new QAction(tr("Remove group"), this);
    connect(action, SIGNAL(triggered()), this, SLOT(RemoveGroupSlot()));
    m_GroupMenu->addAction(action);
    action = new QAction(tr("Send SMS to group"), this);
    connect(action, SIGNAL(triggered()), this, SLOT(SendSmsToGroupSlot()));
    m_GroupMenu->addAction(action);

    ui->tblGroups->setContextMenuPolicy(Qt::CustomContextMenu);
    connect(ui->tblGroups, SIGNAL(cellClicked(int,int)), this, SLOT(GroupClickedSlot(int, int)));
    connect(ui->tblGroups, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(GroupMenuSlot(QPoint)));
    OpenDB();
    PopulateGroupsTable();
}

MainWindow::~MainWindow()
{
    m_Db->close();
    delete m_Db;
    m_Db = NULL;
    delete ui;
}


void MainWindow::OpenDB()
{
    QString dbName = QString("%1/groups.db").arg(CommonStatic::GetSettingsFolder());
    m_Db = new QSqlDatabase();
    *m_Db = QSqlDatabase::addDatabase( "QSQLITE" );
    m_Db->setDatabaseName(dbName);
    if (!m_Db->open()){
        qDebug() << QString("WARNING: error opening groups database");
        delete m_Db;
        m_Db = NULL;
    }else{        
        QSqlQuery qry;
        if (qry.prepare("PRAGMA synchronous=OFF;"))
            qry.exec();
        CreateTables();
    }
}

bool MainWindow::CreateTables()
{
    QStringList queries;
    queries.append("Create table system(s_id INTEGER UNIQUE PRIMARY KEY, \
                                        s_db_version varchar(255) ) ");

    queries.append(QString("delete from system"));
    queries.append(QString("insert into system(s_db_version) \
                                   values('%1')").arg(QUOTE(DB_VERSION)));

    queries.append("Create table groups(g_id INTEGER UNIQUE PRIMARY KEY, \
                                        g_name varchar(255) ) ");
    queries.append("Create table group_contacts(gc_id INTEGER UNIQUE PRIMARY KEY, \
                                                gc_group_id INTEGER, \
                                                gc_contact_id integer, \
                                                gc_number varchar(30)) ");
    queries.append("Create index group_contacts_group_id on group_contacts (gc_group_id) ");

    QSqlQuery qry;
    foreach(QString query, queries){
        if (!qry.prepare(query))
            return false;
        if (!qry.exec())
            return false;
    }
    return true;
}

void MainWindow::NewGroupSlot()
{
    Group* g = new Group();
    EditGroupWindow* frm = new EditGroupWindow(this, g);
    if (frm->exec() == QDialog::Accepted){
        g->SaveToDb();
        PopulateGroupsTable();
    }
    delete frm;
}

void MainWindow::PopulateGroupsTable()
{
    ui->tblGroups->clear();
    ui->tblGroups->verticalHeader()->hide();
    ui->tblGroups->horizontalHeader()->hide();
    ui->tblGroups->horizontalHeader()->setStretchLastSection(true);

    m_Groups = Group::GetGroupList();

    ui->tblGroups->setRowCount(m_Groups.count());
    ui->tblGroups->setColumnCount(2);

    QPixmap* cImage = new QPixmap(":/images/group.png");
    int i = 0;
    QTableWidgetItem* item;
    foreach(Group* g, m_Groups){
        ui->tblGroups->setRowHeight(i, 56);

        QLabel* lbl = new QLabel(this);
        lbl->setAlignment(Qt::AlignCenter);
        lbl->setPixmap(cImage->scaled(48, 48, Qt::KeepAspectRatio));
        ui->tblGroups->setCellWidget(i, 0, lbl);

        item = new QTableWidgetItem(0);
        item->setText(QString("%1 (%2)").arg(g->Name).arg(g->ContactsId.count()));
        ui->tblGroups->setItem(i, 1, item);

        i++;
    }
    delete cImage;
}

void MainWindow::GroupClickedSlot(int row, int)
{    
    Group* g = m_Groups.at(row);

    GroupActionsWindow* frm = new GroupActionsWindow(this, g);
    frm->exec();
    PopulateGroupsTable();
}

void MainWindow::ExitSlot()
{
    close();
}

void MainWindow::AboutSlot()
{
    AboutWindow* frm = new AboutWindow(this);
    frm->show();
    frm->Init();
    frm->exec();

    delete frm;
}

void MainWindow::show()
{
    QMainWindow::show();
    QTimer::singleShot(300, this, SLOT(TimerSlot()));
}

void MainWindow::TimerSlot()
{
    CommonStatic::LoadContactsCache();
}

void MainWindow::GroupMenuSlot(QPoint point)
{
    m_GroupMenu->exec(mapToGlobal(point));
}

void MainWindow::SendSmsToGroupSlot()
{
#ifdef Q_WS_MAEMO_5
    Group* g = m_Groups.at(ui->tblGroups->currentRow());
    QString strTo;
    foreach (GroupContact* gc, g->ContactsId){
        if (!strTo.isEmpty())
            strTo.append(", ");
        strTo.append(gc->Number);
    }

    QUrl smsUrl((QString("sms:%1").arg(strTo)));
    smsUrl.addQueryItem("body", "");
    hildon_uri_open(smsUrl.toString().toStdString().c_str(),NULL,NULL);
#endif

}

void MainWindow::RemoveGroupSlot()
{
    Group* g = m_Groups.at(ui->tblGroups->currentRow());
    if (CommonStatic::ConfirmMessage(this,
                                     tr("Confirm remove"),
                                     QString(tr("Remove group %1?"))
                                     .arg(g->Name)) == QMessageBox::Yes){
        g->Delete();
        PopulateGroupsTable();
    }
}
