/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef GPXVIEW_H
#define GPXVIEW_H

#ifdef USE_MAEMO
#if MAEMO_VERSION_MAJOR < 5
#define USE_BREAD_CRUMB_TRAIL
#else
#define FREMANTLE

// these are on maemo fremantle
#include <hildon/hildon-button.h>
#include <hildon/hildon-text-view.h>
#include <hildon/hildon-window-stack.h>
#define USE_HILDON_TEXT_VIEW
#define USE_STACKABLE_WINDOW
#define NO_COPY_N_PASTE
#define MENU_GPXLIST   0
#define MENU_CACHELIST 1
#define MENU_CACHE     2
#endif
#endif

#ifdef BCT
#include "bct.h"
#endif

#include "config.h"

#include <string.h>
#include <stdlib.h>

#include <locale.h>
#include <libintl.h>

#define _(String) gettext(String)
#define N_(String) (String)

#ifdef USE_MAEMO
#include <hildon/hildon-program.h>
#include <hildon/hildon-file-chooser-dialog.h>
#include <hildon/hildon-file-system-model.h>
#include <hildon/hildon-bread-crumb-trail.h>
#include <hildon/hildon-number-editor.h>
#include <libosso.h>      /* required for screen saver timeout */
#ifdef HILDON_HELP
#include <hildon/hildon-help.h>
#endif
#if MAEMO_VERSION_MAJOR >= 5
#define USE_PANNABLE_AREA
#include <hildon/hildon-pannable-area.h>
/* panning a gtkhtml view currently doesn't work well */
#define PANNABLE_HTML
#endif
#endif

#include <gtk/gtk.h>
#include <gtkhtml/gtkhtml.h>

/* list of current gpx files is being kept in gconf */
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-inet-connection.h>

/* all include appdata_t relies on go here ... */
#include "gpx.h"
#include "settings.h"

#define GPXLIST_ITEM_VALID    (1<<0)
#define GPXLIST_ITEM_DATE     (1<<1)
#define GPXLIST_ITEM_CNUM     (1<<2)
#define GPXLIST_ITEM_FILENAME (1<<3)
#define GPXLIST_ITEM_DEFAULT  (GPXLIST_ITEM_VALID | GPXLIST_ITEM_CNUM)

#define CACHELIST_ITEM_VALID  (1<<0)
#define CACHELIST_ITEM_ID     (1<<1)
#define CACHELIST_ITEM_SIZE   (1<<2)
#define CACHELIST_ITEM_RATING (1<<3)
#define CACHELIST_ITEM_DEFAULT  (CACHELIST_ITEM_VALID | CACHELIST_ITEM_ID)

typedef struct {
  pos_t home;                 /* current home geo position */
  gboolean use_gps;           /* use GPS if avbailable */
  char *path;                 /* current path to select GPX from */

  location_t *location;       /* extra home locations */
  int active_location;        /* index of active (extra) home location */

  gboolean imperial;          /* display imperial units */

  pos_t gps;                  /* saved last gps position */

  char *image_path;           /* path to cache images in */

  int search;                 /* "search in" items */
  char *search_str; 
  int search_days;

  struct proxy_s *proxy;      /* proxy settings */

  int gpxlist_items;
  int cachelist_items;
  guint cachelist_handler_id; /* update timer */
  gboolean cachelist_hide_found;

  /* geotext related */
  char *geotext_text;
  int  geotext_shift;

  gpx_t *gpx;
  GtkListStore *gpxstore;
  GtkWidget    *gpxview;
  GConfClient *gconf_client;

  gboolean compass_locked;
  int compass_damping;

  gboolean disable_gcvote;

  struct gps_state *gps_state;

  /* keep track of all html views so zoom events can be delivered */
  struct html_view *html_view;

#ifndef REQUIRES_CLOSE_BUTTON
  /* this is being activated and de-activated on selection/deselection */
  GtkWidget *menu_remove, *menu_close;
#endif

#ifndef NO_COPY_N_PASTE
  /* do enable/disable the edit entries */
  GtkWidget *menu_cut, *menu_copy, *menu_paste;
  GtkTextBuffer *active_buffer;
#endif

  GtkClipboard *clipboard;     /* clipboard for copy/paste */

#ifdef USE_STACKABLE_WINDOW
  GtkWidget *export_menu, *tools_menu;
#endif

  /* save result here to be able to "goto" it */
  pos_t geomath;

  int cur_items;
  GtkWidget *cur_view;

  char *mmpoi_path;           /* path to save poi to */
  float mmpoi_radius;
  gboolean mmpoi_use_radius;
  gboolean mmpoi_dont_export_found;
  gboolean mmpoi_dont_export_disabled;

  char *fieldnotes_path;       /* path to save field notes to */

  char *garmin_path;           /* path to save garmin data to */
  gboolean garmin_ign_found;

#ifdef USE_MAEMO
  gboolean fullscreen;
#ifdef ENABLE_MAEMO_MAPPER
  pos_t mmpos;                /* position received from Maemo Mapper */
  gboolean mmpos_valid;
#endif

  gboolean mmpoi_dontlaunch;

  gboolean cachelist_disable_screensaver;
  gboolean goto_disable_screensaver;

  HildonProgram *program;
  HildonWindow *window;
  osso_context_t *osso_context;
#else
  GtkWidget *window;
#endif

  gboolean cachelist_update;

#if defined(USE_BREAD_CRUMB_TRAIL) || defined(BCT)
  /* reference to enable/disable these while walking the crumb trail */
  GtkWidget *menu_import, *menu_export;
  GtkWidget *menu_search;

  GtkWidget *bct;
#endif

  /* keep track of current "path" in the bread crumb trail */
  gpx_t *cur_gpx;
  cache_t *cur_cache;
  struct cache_context_s *cache_context;

  gpx_t *search_results;

#ifdef ENABLE_OSM_GPS_MAP
  struct {
    struct map_context_s *context;
    int source;
    int zoom;
    pos_t pos;
  } map;
#endif

  /* username is also used to detect own caches (by log entry) */
  char *username;

  /* stuff to call geotoad */
  struct {
    char *password, *filename;
    float distance, lat, lon;
    int flags;
  } gt;

  GtkWidget *vbox;    
} appdata_t;

/* all includes that rely on appdata_t go here ... */
#include "gps.h"
#include "goto.h"
#include "gconf.h"
#include "html.h"
#include "icons.h"
#include "gcvote.h"
#include "cache.h"
#include "geomath.h"
#include "geotext.h"
#include "notes.h"
#include "help.h"
#include "mm_poi.h"
#include "precpos.h"
#include "garmin_export.h"
#include "geotoad.h"
#include "about.h"

#ifdef ENABLE_OSM_GPS_MAP
#include "map-tool.h"
#endif

#ifdef USE_MAEMO
#include "dbus.h"
#endif

#define SEARCH_ID    (1<<0)
#define SEARCH_NAME  (1<<1)
#define SEARCH_DESC  (1<<2)
#define SEARCH_OWNER (1<<3)
#define SEARCH_FINDS (1<<4)

#ifdef USE_MAEMO
#define DIALOG_WIDTH  800
#define DIALOG_HEIGHT 480
#define GTK_FM_OK  GTK_RESPONSE_OK
#else
#define DIALOG_WIDTH  560
#define DIALOG_HEIGHT 340
#define GTK_FM_OK  GTK_RESPONSE_ACCEPT
#endif

#include "misc.h"

#define CHANGE_FLAG_POS   (1<<0)
#define CHANGE_FLAG_MASK  (1<<1)

extern void main_after_settings_redraw(appdata_t *appdata, int flags);

#include <stdarg.h>
extern void errorf(const char *fmt, ...);
extern gboolean on_window_key_press(GtkWidget *widget, 
				    GdkEventKey *event, appdata_t *appdata);
#ifdef USE_STACKABLE_WINDOW
HildonAppMenu *menu_create(appdata_t *appdata, int mode);
#endif
void cachelist_goto_cache(appdata_t *appdata, cache_t *cache);
void gpxlist_goto_cachelist(appdata_t *appdata, gpx_t *gpx);
gboolean gpxlist_find(appdata_t *appdata, GtkTreeIter *iter, gpx_t *gpx);
void gpxlist_set(GtkListStore *store, GtkTreeIter *iter, gpx_t *gpx);
void gpxlist_add(appdata_t *appdata, gpx_t *new);
void cachelist_redraw(appdata_t *appdata);

#endif // GPXVIEW_H
