/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <string.h>
#include <math.h>

#include "gpxview.h"


#ifdef USE_MAEMO
#ifdef ENABLE_GPSBT
#include <gpsbt.h>
#include <gpsmgr.h>
#endif
#include <errno.h>
#endif

static void gps_unregister_all(appdata_t *appdata);

#ifndef ENABLE_LIBLOCATION

/* maybe user configurable later on ... */
#define GPSD_HOST "127.0.0.1"
#define GPSD_PORT 2947

gps_sat_t *gps_get_sats(appdata_t *appdata) {
  gps_sat_t *retval = NULL;

  g_mutex_lock(appdata->gps_state->mutex);
  if(appdata->gps_state->gpsdata.set & SATELLITE_SET) {
    int i;
    retval = malloc(sizeof(gps_sat_t));
    retval->num = appdata->gps_state->gpsdata.satellites;

    retval->PRN  = malloc(sizeof(int)*retval->num);
    retval->used = malloc(sizeof(int)*retval->num);
    retval->ss   = malloc(sizeof(int)*retval->num);

    if(retval->num) {
      for(i=0;i<retval->num;i++) {
	retval->PRN[i]  = appdata->gps_state->gpsdata.PRN[i];
	retval->ss[i]   = appdata->gps_state->gpsdata.ss[i];
	retval->used[i] = appdata->gps_state->gpsdata.used[i];
      }
    }
  }

  g_mutex_unlock(appdata->gps_state->mutex);

  return retval;
}

pos_t *gps_get_pos(appdata_t *appdata) {
  static pos_t retval;

  retval.lat = NAN;

  g_mutex_lock(appdata->gps_state->mutex);
  if(appdata->gps_state->gpsdata.set & STATUS_SET)
    if(appdata->gps_state->gpsdata.status != STATUS_NO_FIX)
      if(appdata->gps_state->gpsdata.set & LATLON_SET) 
	retval = appdata->gps_state->gpsdata.fix.pos;

  g_mutex_unlock(appdata->gps_state->mutex);

  if(isnan(retval.lat))
    return NULL;

  return &retval;
}

float gps_get_heading(appdata_t *appdata) {
  float retval = NAN;

  g_mutex_lock(appdata->gps_state->mutex);
  if(appdata->gps_state->gpsdata.set & STATUS_SET)
    if(appdata->gps_state->gpsdata.status != STATUS_NO_FIX) 
      if(appdata->gps_state->gpsdata.set & TRACK_SET) 
	retval = appdata->gps_state->gpsdata.fix.track;

  g_mutex_unlock(appdata->gps_state->mutex);
  return retval;
}

float gps_get_eph(appdata_t *appdata) {
  float retval = NAN;

  g_mutex_lock(appdata->gps_state->mutex);
  if(appdata->gps_state->gpsdata.set & STATUS_SET)
    if(appdata->gps_state->gpsdata.status != STATUS_NO_FIX) 
      retval = appdata->gps_state->gpsdata.fix.eph;

  g_mutex_unlock(appdata->gps_state->mutex);
  return retval;
}

static int gps_connect(gps_state_t *gps_state) {
  GnomeVFSResult vfs_result;
#ifdef USE_MAEMO
  char errstr[256] = "";

  if(!gps_state) {
    printf("No gps state\n");
    return -1;
  }

  /* We need to start gpsd (via gpsbt) first. */
  memset(&gps_state->context, 0, sizeof(gpsbt_t));
  errno = 0;

  if(gpsbt_start(NULL, 0, 0, 0, errstr, sizeof(errstr),
		 0, &gps_state->context) < 0) {
    printf("gps: Error connecting to GPS receiver: (%d) %s (%s)\n",
	   errno, strerror(errno), errstr);
  }
#endif

  /************** from here down pure gnome/gtk/gpsd ********************/

  /* try to connect to gpsd */
  /* Create a socket to interact with GPSD. */

  int retries = 5;
  while(retries && 
	(GNOME_VFS_OK != (vfs_result = gnome_vfs_inet_connection_create(
		&gps_state->iconn, GPSD_HOST, GPSD_PORT, NULL)))) {
    printf("gps: Error creating connection to GPSD, retrying ...\n");

    retries--;
    sleep(1);
  }

  if(!retries) {
    printf("gps: Finally failed ...\n");
    return -1;
  }

  retries = 5;
  while(retries && ((gps_state->socket = 
     gnome_vfs_inet_connection_to_socket(gps_state->iconn)) == NULL)) {
    printf("gps: Error creating connecting GPSD socket, retrying ...\n");

    retries--;
    sleep(1);
  }

  if(!retries) {
    printf("gps: Finally failed ...\n");
    return -1;
  }

  GTimeVal timeout = { 10, 0 };
  if(GNOME_VFS_OK != (vfs_result = gnome_vfs_socket_set_timeout(
	gps_state->socket, &timeout, NULL))) {
    printf("gps: Error setting GPSD timeout\n");
    return -1;
  } 

  printf("gps: GPSD connected ...\n");

  return 0;
}	

void gps_clear_fix(struct gps_fix_t *fixp) {
  fixp->mode = MODE_NOT_SEEN;
  fixp->pos.lat = fixp->pos.lon = NAN;
  fixp->track = NAN;
  fixp->eph = NAN;
}

/* unpack a daemon response into a status structure */
static void gps_unpack(char *buf, struct gps_data_t *gpsdata) {
  char *ns, *sp, *tp;

  for(ns = buf; ns; ns = strstr(ns+1, "GPSD")) {
    if(strncmp(ns, "GPSD", 4) == 0) {
      /* the following should execute each time we have a good next sp */
      for (sp = ns + 5; *sp != '\0'; sp = tp+1) {
	tp = sp + strcspn(sp, ",\r\n");
	if (*tp == '\0') tp--;
	else *tp = '\0';
	
	switch (*sp) {
	  /* A - altitude is not supported */
	  /* B - baudrate isn't supported by gpxview */
	  /* C - cycle isn't supported by gpxview */
	  /* D - utc time isn't supported by gpxview */
	case 'E':
	  gpsdata->fix.eph = NAN;
	  /* epe should always be present if eph or epv is */
	  if (sp[2] != '?') {
	    char epe[20], eph[20], epv[20];
	    (void)sscanf(sp, "E=%s %s %s", epe, eph, epv);
#define DEFAULT(val) (val[0] == '?') ? NAN : g_ascii_strtod(val, NULL)
	    gpsdata->fix.eph = DEFAULT(eph);
#undef DEFAULT
	  }
	  break;
	  /* F - device name isn't supported by gpxview */
	  /* I - gps id isn't supported by gpxview */
	  /* K - known devices list isn't supported by gpxview */
	case 'M':
	  if (sp[2] == '?') {
	    gpsdata->fix.mode = MODE_NOT_SEEN;
	  } else {
	    gpsdata->fix.mode = atoi(sp+2);
	    gpsdata->set |= MODE_SET;
	  }
	  break;
	  /* N - driver mode reporting isn't supported by gpxview */
	case 'O':
	  if (sp[2] == '?') {
	    gpsdata->set = 
	      (gpsdata->set & SATELLITE_SET) | // fix for below
	      MODE_SET | STATUS_SET;  // this clears sat info??
	    gpsdata->status = STATUS_NO_FIX;
	    gps_clear_fix(&gpsdata->fix);
	  } else {
	    struct gps_fix_t nf;
	    char tag[MAXTAGLEN+1], alt[20];
	    char eph[20], epv[20], track[20],speed[20], climb[20];
	    char epd[20], eps[20], epc[20], mode[2];
	    char timestr[20], ept[20], lat[20], lon[20];
	    int st = sscanf(sp+2, 
			    "%8s %19s %19s %19s %19s %19s %19s %19s %19s %19s %19s %19s %19s %19s %1s",
			    tag, timestr, ept, lat, lon,
			    alt, eph, epv, track, speed, climb,
			    epd, eps, epc, mode);
	    if (st >= 14) {
#define DEFAULT(val) (val[0] == '?') ? NAN : g_ascii_strtod(val, NULL)
	      nf.pos.lat = DEFAULT(lat);
	      nf.pos.lon = DEFAULT(lon);
	      nf.eph = DEFAULT(eph);
	      nf.track = DEFAULT(track);
#undef DEFAULT
	      if (st >= 15)
		nf.mode = (mode[0] == '?') ? MODE_NOT_SEEN : atoi(mode);
	      else
		nf.mode = (alt[0] == '?') ? MODE_2D : MODE_3D;
	      if (isnan(nf.eph)==0)
		gpsdata->set |= HERR_SET;
	      if (isnan(nf.track)==0)
		gpsdata->set |= TRACK_SET;
	      gpsdata->fix = nf;
	      gpsdata->set |= LATLON_SET|MODE_SET;
	      gpsdata->status = STATUS_FIX;
	      gpsdata->set |= STATUS_SET;
	    }
	  }
	  break;
	case 'P':
	  if (sp[2] == '?') {
	    gpsdata->fix.pos.lat = NAN;
	    gpsdata->fix.pos.lon = NAN;
	  } else {
	    char lat[20], lon[20];
	    (void)sscanf(sp, "P=%19s %19s", lat, lon);
	    gpsdata->fix.pos.lat = g_ascii_strtod(lat, NULL);
	    gpsdata->fix.pos.lon = g_ascii_strtod(lon, NULL);
	    gpsdata->set |= LATLON_SET;
	  }
	  break;
	  /* Q is not supported */
	case 'S':
	  if (sp[2] == '?') {
	    gpsdata->status = -1;
	  } else {
	    gpsdata->status = atoi(sp+2);
	    gpsdata->set |= STATUS_SET;
	  }
	  break;
	case 'T':
	  if (sp[2] == '?') {
	    gpsdata->fix.track = NAN;
	  } else {
	    (void)sscanf(sp, "T=%f", &gpsdata->fix.track);
	    gpsdata->set |= TRACK_SET;
	  }
	  break;
	  /* U - climb is not supported */
	  /* V - is not supported */
	  /* X - online is not supported */
	case 'Y':
	  if (sp[2] == '?') {
	    gpsdata->satellites = 0;
	  } else {
	    int j, i1, i2, i3, i4, i5;
	    int PRN[MAXCHANNELS];
	    int elevation[MAXCHANNELS], azimuth[MAXCHANNELS];
	    int ss[MAXCHANNELS], used[MAXCHANNELS];
	    char tag[MAXTAGLEN+1], timestamp[21];
	    
	    (void)sscanf(sp, "Y=%8s %20s %d ", 
			 tag, timestamp, &gpsdata->satellites);
	    for (j = 0; j < gpsdata->satellites; j++) {
	      PRN[j]=elevation[j]=azimuth[j]=ss[j]=used[j]=0;
	    }
	    //	    printf("gps: sats = %d\n", gpsdata->satellites);
	    for (j = 0, gpsdata->satellites_used = 0; j < gpsdata->satellites; j++) {
	      if ((sp != NULL) && ((sp = strchr(sp, ':')) != NULL)) {
		sp++;
		(void)sscanf(sp, "%d %d %d %d %d", &i1, &i2, &i3, &i4, &i5);
		PRN[j] = i1;
		elevation[j] = i2; azimuth[j] = i3;
		ss[j] = i4; used[j] = i5;
		if (i5 == 1)
		  gpsdata->satellites_used++;
	      }
	    }
	    memcpy(gpsdata->PRN, PRN, sizeof(PRN));
	    memcpy(gpsdata->elevation, elevation, sizeof(elevation));
	    memcpy(gpsdata->azimuth, azimuth,sizeof(azimuth));
			memcpy(gpsdata->ss, ss, sizeof(ss));
			memcpy(gpsdata->used, used, sizeof(used));
			/*@ +compdef @*/
	  }
	  gpsdata->set |= SATELLITE_SET;
	  break;
	  /* Z and $ - profiling isn't supported by gpxview */
	}
      }
    }
  }
}

/* call one of the application provided gps callbacks */
static void do_app_cb(gpointer data, gpointer user_data) {
  appdata_t *appdata = (appdata_t*)user_data;
  gps_cb_t *cb = (gps_cb_t*)data;
  cb->cb(appdata->gps_state, cb->data);
}

/* walk though list of all application provided callbacks */
static gboolean gps_idle_cb(gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  //  printf("gps: idle callback, calling app callbacks\n");

  g_slist_foreach(appdata->gps_state->cb, do_app_cb, appdata);

  return FALSE;
}

gpointer gps_thread(gpointer data) {
  GnomeVFSFileSize bytes_read;
  GnomeVFSResult vfs_result;
  char str[512];
  appdata_t *appdata = (appdata_t*)data;
  int cnt=1000;

  const char *msg_pos = "o\r\n";   /* pos request */
  const char *msg_sat = "y\r\n";   /* sat request */

  appdata->gps_state->gpsdata.set = 0;

  gboolean connected = FALSE;

  while(1) {
    if(appdata->use_gps) {
      if(!connected) {
	printf("gps: trying to connect\n");

	if(gps_connect(appdata->gps_state) < 0)
	  sleep(10);
	else 
	  connected = TRUE;
      } else {
	const char *msg;
	if(!cnt) msg = msg_sat;
	else     msg = msg_pos;
      
	if(GNOME_VFS_OK == 
	   (vfs_result = gnome_vfs_socket_write(appdata->gps_state->socket,
		      msg, strlen(msg)+1, &bytes_read, NULL))) {

	  /* update every second, wait here to make sure a complete */
	  /* reply is received */
	  if(cnt <= 1) usleep(500000);
	  else 	       sleep(1);
	  
	  if(bytes_read == (strlen(msg)+1)) {
	    vfs_result = gnome_vfs_socket_read(appdata->gps_state->socket,
			       str, sizeof(str)-1, &bytes_read, NULL);
	    if(vfs_result == GNOME_VFS_OK) {
	      str[bytes_read] = 0; 
	    
	      //	  printf("gps: msg: %s (%d)\n", str, strlen(str));
	      
	      g_mutex_lock(appdata->gps_state->mutex);
	      
	      if(!cnt) appdata->gps_state->gpsdata.set &= ~SATELLITE_SET;
	      else     appdata->gps_state->gpsdata.set &= 
			 ~(LATLON_SET|MODE_SET|STATUS_SET);
	    
	      gps_unpack(str, &appdata->gps_state->gpsdata);
	      g_mutex_unlock(appdata->gps_state->mutex);
	      g_idle_add(gps_idle_cb, appdata); 
	    }
	  }
	}
	if(cnt++ >= 5) cnt = 0;
      }
    } else {
      if(connected) {
	printf("gps: stopping GPS connection due to user request\n");
	gnome_vfs_inet_connection_destroy(appdata->gps_state->iconn, NULL);

#ifdef USE_MAEMO
	gpsbt_stop(&appdata->gps_state->context);
#endif
	connected = FALSE;
      } else
	sleep(1);
    }
  }

  printf("gps: thread ended???\n");
  return NULL;
}

void gps_init(appdata_t *appdata) {
  appdata->gps_state = g_new0(gps_state_t, 1);

  /* start a new thread to listen to gpsd */
  appdata->gps_state->mutex = g_mutex_new();
  appdata->gps_state->thread_p = 
    g_thread_create(gps_thread, appdata, FALSE, NULL);
}

void gps_release(appdata_t *appdata) { 
  gps_unregister_all(appdata);
#ifdef USE_MAEMO
  gpsbt_stop(&appdata->gps_state->context);
#endif
  g_free(appdata->gps_state);
}

#else

static void
location_changed(LocationGPSDevice *device, gps_state_t *gps_state) {

  gps_state->fields = device->fix->fields;

  if(gps_state->fields & LOCATION_GPS_DEVICE_LATLONG_SET) { 
    gps_state->fix.pos.lat = device->fix->latitude;
    gps_state->fix.pos.lon = device->fix->longitude;
    gps_state->fix.eph = device->fix->eph/100.0;  // we want eph in meters
  }

  if(gps_state->fields & LOCATION_GPS_DEVICE_TRACK_SET)
    gps_state->fix.track = device->fix->track;
  
  /* update list of sattelites */

  /* free old list */
  if(gps_state->sats.num) {
    g_free(gps_state->sats.PRN);
    g_free(gps_state->sats.used);
    g_free(gps_state->sats.ss);
    gps_state->sats.num = 0;
  }

  /* build new one */
  if(device->satellites_in_view) {
    gps_state->sats.PRN  = g_new0(int, device->satellites_in_view);
    gps_state->sats.used = g_new0(int, device->satellites_in_view);
    gps_state->sats.ss   = g_new0(int, device->satellites_in_view);

    int i;
    for(i=0;i<device->satellites_in_view;i++) {
      LocationGPSDeviceSatellite *sat = 
	g_ptr_array_index(device->satellites, i);

      gps_state->sats.PRN[i]  = sat->prn;
      gps_state->sats.used[i] = sat->in_use;
      gps_state->sats.ss[i]   = sat->signal_strength;
    }

    gps_state->sats.num  = device->satellites_in_view;
  }
}

void gps_init(appdata_t *appdata) {
  gps_state_t *gps_state = appdata->gps_state = g_new0(gps_state_t, 1);

  printf("gps: init: Using liblocation\n");

  gps_state->device = g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);  
  if(!gps_state->device) {
    printf("gps: Unable to connect to liblocation\n");
    return;
  }

  gps_state->idd_changed = 
    g_signal_connect(gps_state->device, "changed", 
		     G_CALLBACK(location_changed), gps_state);

  gps_state->control = location_gpsd_control_get_default();

  if(gps_state->control
#if MAEMO_VERSION_MAJOR < 5
     && gps_state->control->can_control
#endif
     ) {

    printf("gps: Having control over GPSD and GPS is to be enabled, starting it\n");
    location_gpsd_control_start(gps_state->control);
  }
}

void gps_release(appdata_t *appdata) {
  gps_state_t *gps_state = appdata->gps_state;
  gps_unregister_all(appdata);
  
  if(!gps_state->device) return;

  if(gps_state->control
#if MAEMO_VERSION_MAJOR < 5
     && gps_state->control->can_control
#endif     
     ) {
    printf("gps: Having control over GPSD, stopping it\n");
    location_gpsd_control_stop(gps_state->control);
  }
  
  /* Disconnect signal */
  g_signal_handler_disconnect(gps_state->device, gps_state->idd_changed);
  
  g_free(appdata->gps_state);
  appdata->gps_state = NULL;
}

pos_t *gps_get_pos(appdata_t *appdata) {   
  static pos_t pos;

  if(!appdata->use_gps)
    return NULL;

  gps_state_t *gps_state = appdata->gps_state;

  if(!(gps_state->fields & LOCATION_GPS_DEVICE_LATLONG_SET))
    return NULL;

  pos.lat = gps_state->fix.pos.lat;
  pos.lon = gps_state->fix.pos.lon;

  return &pos; 
}

float gps_get_heading(appdata_t *appdata) { 
  gps_state_t *gps_state = appdata->gps_state;
 
  if(!(gps_state->fields & LOCATION_GPS_DEVICE_TRACK_SET))
    return NAN;

  return gps_state->fix.track; 
}

float gps_get_eph(appdata_t *appdata) { 
  gps_state_t *gps_state = appdata->gps_state;
 
  if(!(gps_state->fields & LOCATION_GPS_DEVICE_LATLONG_SET))
    return NAN;

  return gps_state->fix.eph; 
}

gps_sat_t *gps_get_sats(appdata_t *appdata) { 
  gps_sat_t *retval = NULL;
  gps_state_t *gps_state = appdata->gps_state;

  if(gps_state->sats.num) {
    retval = g_new0(gps_sat_t, 1);
    retval->num = gps_state->sats.num;
  
    retval->PRN  = g_memdup(gps_state->sats.PRN, 
			    sizeof(int)*gps_state->sats.num);
    retval->used = g_memdup(gps_state->sats.used, 
			    sizeof(int)*gps_state->sats.num);
    retval->ss   = g_memdup(gps_state->sats.ss, 
			    sizeof(int)*gps_state->sats.num);
  }

  return retval;
}


#endif

void *gps_register_callback(appdata_t *appdata, gps_cb cb, gpointer data) {
  printf("gps: register gps callback\n");

  if(!appdata->gps_state)
    return NULL;

  /* allocate callback info strcuture */
  gps_cb_t *cb_info = g_new0(gps_cb_t, 1);
  cb_info->cb = cb;
  cb_info->data = data;

  /* and insert it into list of callbacks */
  appdata->gps_state->cb = g_slist_append(appdata->gps_state->cb, cb_info);

  return cb_info;
}


void gps_unregister_callback(appdata_t *appdata, void *cb) {
  printf("gps: unregister gps callback\n");

  if(!appdata->gps_state)
    return;

  /* the item must be in the list */
  g_assert(g_slist_find(appdata->gps_state->cb, cb));
    
  g_free(cb);
  appdata->gps_state->cb = g_slist_remove(appdata->gps_state->cb, cb);
}

static void gps_unregister_all(appdata_t *appdata) {
  printf("gps: unregister all callbacks: ");

  while(appdata->gps_state->cb) {
    printf(".");
    g_free(appdata->gps_state->cb->data);
    appdata->gps_state->cb = g_slist_remove(appdata->gps_state->cb,
			    appdata->gps_state->cb->data);
  }
  printf("\n");
}
