/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <libxml/parser.h>
#include <libxml/tree.h>

#include <libxml/xmlreader.h>

#include <glib.h>
#include <glib/gstdio.h>

#include <zlib.h>

#include "gpxview.h"
#include "unzip.h"

void gpx_free_wpt(wpt_t *wpt) {
  if(wpt->id)   xmlFree(wpt->id);
  if(wpt->cmt)  xmlFree(wpt->cmt);
  if(wpt->desc) xmlFree(wpt->desc);
  free(wpt);
}

void gpx_free_log(log_t *log) {
  if(log->finder)  xmlFree(log->finder);
  if(log->text)    xmlFree(log->text);
  free(log);
}

void gpx_free_tb(tb_t *tb) {
  if(tb->name) xmlFree(tb->name);
  if(tb->ref)  xmlFree(tb->ref);
  free(tb);
}

void gpx_free_cache(cache_t *cache) {
  log_t *log = cache->log;
  wpt_t *wpt = cache->wpt;
  tb_t  *tb  = cache->tb;

  if(cache->id)                xmlFree(cache->id);
  if(cache->name)              xmlFree(cache->name);
  if(cache->owner)             xmlFree(cache->owner);
  if(cache->short_description) xmlFree(cache->short_description);
  if(cache->long_description)  xmlFree(cache->long_description);
  if(cache->hint)              xmlFree(cache->hint);
  if(cache->url)               xmlFree(cache->url);

  /* free all logs */
  while(log) { log_t *next = log->next; gpx_free_log(log); log = next; }

  /* free all waypoints */
  while(wpt) { wpt_t *next = wpt->next; gpx_free_wpt(wpt); wpt = next; }

  /* free all tbs */
  while(tb) { tb_t *next = tb->next; gpx_free_tb(tb); tb = next; }

  if(cache->notes)             notes_free(cache->notes);

  free(cache);
}

void gpx_free_caches(gpx_t *gpx) {
  cache_t *cache = gpx->cache;

  /* free all caches */
  while(cache) {
    cache_t *next = cache->next;
    gpx_free_cache(cache);
    cache = next;
  }

  gpx->cache = NULL;
}

void gpx_free(gpx_t *gpx) {

  if(gpx->name)     xmlFree(gpx->name);
  if(gpx->desc)     xmlFree(gpx->desc);
  if(gpx->filename) free(gpx->filename);

  gpx_free_caches(gpx);
  
  free(gpx);
}

void gpx_free_all(gpx_t *gpx) {
  while(gpx) {
    gpx_t *next = gpx->next;
    gpx_free(gpx);
    gpx = next;
  }
}

static const char *cache_type_str[] = { "<Unknown>",
  "Traditional Cache|Traditional|Geocache", "Multi-cache|Multi", 
  "Unknown Cache|Other",
  "Virtual Cache|Virtual", "Webcam Cache|Webcam", "Event Cache|Event|Geocoins:",
  "Letterbox Hybrid|Letterbox", "Earthcache", "Wherigo Cache",
  "Mega-Event Cache", "Cache In Trash Out Event",
  ""};

static const char *cache_container_str[] = { "<Unknown>",
  "Regular", "Small", "Micro", "Not chosen|Unknown", 
  "Other", "Large", "Virtual"
  ""};

static const char *log_type_str[] = { "<Unknown>",
  "Found it|Found", "Didn't find it|Not Found", "Owner Maintenance", 
  "Write Note|Note|Other", 
  "Post Reviewer Note", "Enable Listing", "Publish Listing", "Will Attend", 
  "Attended", "Webcam Photo taken", 
  "Temporarily Disable Listing|Cache Disabled!", 
  "Needs Maintenance", "Update Coordinates", "Unarchive|Archive (show)",
  "Needs Archived", "Archive",
  ""};

static const char *wpt_sym_str[] = { "<Unknown>",
  "Stages of a Multicache", "Parking Area", "Final Location", 
  "Question to Answer", "Trailhead", "Reference Point",
  ""};

#define DLG_DIV 10

/* create the dialog box shown while loading in progress */
gpx_dialog_t *gpx_busy_dialog_new(GtkWidget *parent) {
#ifdef USE_MAEMO
  gpx_dialog_t *dialog = malloc(sizeof(gpx_dialog_t));
  memset(dialog, 0, sizeof(gpx_dialog_t));

  dialog->dialog = gtk_dialog_new();

  gtk_dialog_set_has_separator(GTK_DIALOG(dialog->dialog), FALSE);
  gtk_window_set_title(GTK_WINDOW(dialog->dialog), _("Loading"));
  gtk_window_set_default_size(GTK_WINDOW(dialog->dialog), 300, 10);

  gtk_window_set_modal(GTK_WINDOW(dialog->dialog), TRUE);
  gtk_window_set_transient_for(GTK_WINDOW(dialog->dialog), GTK_WINDOW(parent));

  dialog->label = gtk_label_new("---");
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog->dialog)->vbox), 
			      dialog->label);

  dialog->pbar = gtk_progress_bar_new();
  gtk_progress_bar_set_pulse_step(GTK_PROGRESS_BAR(dialog->pbar), 
				  0.0025 * DLG_DIV);

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog->dialog)->vbox), 
			      dialog->pbar);

  gtk_widget_show_all(dialog->dialog);

  return dialog;
#else
  return NULL;
#endif
}

void gpx_busy_dialog_destroy(gpx_dialog_t *dialog) {
  if(!dialog) return;

  gtk_widget_destroy(dialog->dialog);
  free(dialog);
}

static void gpx_busy_dialog_progress(gpx_dialog_t *dialog) {
  static int sub_dlg = 0;

  if(!dialog) return;

  if(sub_dlg++ >= DLG_DIV) {
    sub_dlg = 0;

    gtk_progress_bar_pulse(GTK_PROGRESS_BAR(dialog->pbar));

    /* wait for main gui to appear */
    while(gtk_events_pending()) 
      gtk_main_iteration();
  }
}

static void gpx_busy_dialog_set(gpx_dialog_t *dialog, char *name) {
  if(!dialog) return;

  if(strrchr(name, '/'))
    name = strrchr(name, '/')+1;

  gtk_label_set_text(GTK_LABEL(dialog->label), name);

  /* wait for main gui to appear */
  while(gtk_events_pending()) 
    gtk_main_iteration();
}

static int str_search(const char *pstr[], char *str, char *type) {
  int i=0;

  while(pstr[i+1][0]) {
    char *p = (char*)pstr[i+1];

    /* multiple substrings in pattern? */
    while(strchr(p, '|')) {
      if(!strncasecmp(p, str, strchr(p, '|')-p))
	return i;

      p = strchr(p, '|')+1;
    }

    if(!strcasecmp(p, str))
      return i;

    i++;
  }

  fprintf(stderr, "ERROR parsing \"%s\": Unknown \"%s\"\n", type, str);
  return -1;
}

static int log_is_older(log_t *a, log_t *b) {
  if(a->year < b->year)  return TRUE;
  else if(a->year == b->year) {
    if(a->month < b->month) return TRUE;
    else if(a->month == b->month) {
      if(a->day   < b->day)   return TRUE;
    }
  }

  return FALSE;
}

int is_white(char c) {
  return((c==' ')||(c=='\r')||(c=='\n'));
}

static int all_is_white(char *str) {
  while(*str) {
    if(!is_white(*str))
      return FALSE;

    str++;
  }
  return TRUE;
}

void gpx_display_log(log_t *log) {
  printf("  Log:\n");
  printf("    date:     %d.%d.%d\n", log->day, log->month, log->year);
  printf("    type:     %s\n", log_type_str[log->type+1]);
  printf("    finder:   %s\n", log->finder);
  //  printf("    text:     %s\n", log->text);
}

void gpx_display_cache(cache_t *cache) {
  log_t *log = cache->log;

  printf("\nCache:\n");
  printf("  id:         %s\n", cache->id);
  printf("  name:       %s\n", cache->name);
  printf("  latitude:   %f\n", cache->pos.lat);
  printf("  longitude:  %f\n", cache->pos.lon);
  printf("  owner:      %s\n", cache->owner);
  printf("  type:       %s\n", cache_type_str[cache->type+1]);
  printf("  container:  %s\n", cache_container_str[cache->container+1]);
  printf("  difficulty: %.1f\n", cache->difficulty);
  printf("  terrain:    %.1f\n", cache->terrain);
  //  printf("  short:      %s\n", cache->short_description);
  //  printf("  long:       %s\n", cache->long_description);
  //  printf("  hint:       %s\n", cache->hint);  

  while(log) {
    gpx_display_log(log);
    log = log->next;
  }
}

void gpx_display_all(gpx_t *gpx) {
  while(gpx) {
    cache_t *cache = gpx->cache;

    printf("GPX name: %s\n", gpx->name);
    printf("GPX desc: %s\n", gpx->desc);
    printf("GPX date: %d.%d.%d\n", gpx->day, gpx->month, gpx->year);
    while(cache) {
      gpx_display_cache(cache);
      cache = cache->next;
    }
    gpx = gpx->next;
  }
}

static gint my_strcmp(const xmlChar *a, const xmlChar *b) {
  if(!a && !b) return 0;
  if(!a) return -1;
  if(!b) return +1;
  return strcmp((char*)a,(char*)b);
}

static float xml_get_prop_float(xmlTextReaderPtr reader, char *name) {
  float ret = NAN;
  char *prop;
  if((prop = (char*)xmlTextReaderGetAttribute(reader, BAD_CAST name))) {
    ret = g_ascii_strtod(prop, NULL);
    xmlFree(prop);
  }
  return ret;
}

static int xml_prop_is(xmlTextReaderPtr reader, char *name, char *value, 
		       int def_value) {
  int match = def_value;
  char *prop;
  if((prop = (char*)xmlTextReaderGetAttribute(reader, BAD_CAST name))) {
    match = (strcasecmp(prop, value) == 0);
    xmlFree(prop);
  }
  return match;
}

/* skip current element incl. everything below (mainly for testing) */
/* returns FALSE if something failed */
static gboolean skip_element(xmlTextReaderPtr reader) {
  g_assert(xmlTextReaderNodeType(reader) == XML_READER_TYPE_ELEMENT);
  const xmlChar *name = xmlTextReaderConstName(reader);
  g_assert(name);
  int depth = xmlTextReaderDepth(reader);

  if(xmlTextReaderIsEmptyElement(reader))
    return TRUE;

  int ret = xmlTextReaderRead(reader);
  while((ret == 1) && 
	((xmlTextReaderNodeType(reader) != XML_READER_TYPE_END_ELEMENT) ||
	 (xmlTextReaderDepth(reader) > depth) ||
	 (my_strcmp(xmlTextReaderConstName(reader), name) != 0))) {
    ret = xmlTextReaderRead(reader);
  }
  return(ret == 1);
}

static char *process_text(xmlTextReaderPtr reader) {
  char *text = NULL;
  int depth = xmlTextReaderDepth(reader);
  int ret = xmlTextReaderRead(reader);
  while((ret == 1) && 
	((xmlTextReaderNodeType(reader) != XML_READER_TYPE_END_ELEMENT) ||
	 (xmlTextReaderDepth(reader) != depth))) {

    /* found a text fragment */
    if((xmlTextReaderNodeType(reader) == XML_READER_TYPE_TEXT) ||
       (xmlTextReaderNodeType(reader) == XML_READER_TYPE_CDATA)) {
      char *frag = (char*)xmlTextReaderConstValue(reader);

      if(!text) text = strdup(frag);
      else {
	char *old = text;
	text = malloc(strlen(old) + strlen(frag) + 1);
	strcpy(text, old);
	strcat(text, frag);
	free(old);
      }
    }
    ret = xmlTextReaderRead(reader);
  }

  return text;
}

static int xml_str_search(xmlTextReaderPtr reader, 
			  const char *pstr[], char *type, int def) {
  char *text = process_text(reader);
  int result = def;
  if(text) {
    result = str_search(pstr, text, type);
    free(text);
  }
  return result;
}

static float xml_float(xmlTextReaderPtr reader, float def) {
  char *text = process_text(reader);
  float result = def;
  if(text) {
    result = g_ascii_strtod(text, NULL);
    free(text);
  }
  return result;
}

static void xml_get_date(xmlTextReaderPtr reader, int *year, int *month, int *day) {
  char *str = process_text(reader);
  if(str) {
    sscanf(str, "%d-%d-%d", year, month, day);
    free(str);
  }
}

static log_t *process_gpx_wpt_gc_logs_log(xmlTextReaderPtr reader) {

  if(xmlTextReaderIsEmptyElement(reader))
    return NULL;

  /* create a new log entry */
  log_t *log = malloc(sizeof(log_t));
  memset(log, 0, sizeof(log_t));

  /* process all sub-nodes */
  int depth = xmlTextReaderDepth(reader);
  int ret = xmlTextReaderRead(reader);
  while(ret == 1) {
    
    switch(xmlTextReaderNodeType(reader)) {
    case XML_READER_TYPE_ELEMENT:
      g_assert(xmlTextReaderDepth(reader) == depth+1);
      char *name = (char*)xmlTextReaderConstName(reader);
      if(strrchr(name, ':')) name = strrchr(name, ':')+1;
      if(name) {
	if((strcasecmp(name, "date") == 0) ||
	   (strcasecmp(name, "time") == 0)) {
	  xml_get_date(reader, &log->year, &log->month, &log->day);
	} else if(strcasecmp(name, "type") == 0) {
	  log->type = xml_str_search(reader, log_type_str, "log", 0);
	} else if((strcasecmp(name, "finder") == 0) ||
		  (strcasecmp(name, "geocacher") == 0)) {
	  if(!log->finder) log->finder = process_text(reader);
	} else if(strcasecmp(name, "text") == 0) {
	  if(!log->text) log->text = process_text(reader);
	} else
	  skip_element(reader);
      } else
	skip_element(reader);
      break;
      
    case XML_READER_TYPE_END_ELEMENT:
      /* end element must be for the current element */
      g_assert(xmlTextReaderDepth(reader) == depth);
      return log;
      break;
      
    default:
      break;
    }
    ret = xmlTextReaderRead(reader);
  }

  gpx_free_log(log);
  return NULL;
}

static log_t *process_gpx_wpt_gc_logs(xmlTextReaderPtr reader) {
  log_t *log_chain = NULL;

  if(xmlTextReaderIsEmptyElement(reader))
    return NULL;

  /* process all sub-nodes */
  int depth = xmlTextReaderDepth(reader);
  int ret = xmlTextReaderRead(reader);
  while(ret == 1) {
    
    switch(xmlTextReaderNodeType(reader)) {
    case XML_READER_TYPE_ELEMENT:
      g_assert(xmlTextReaderDepth(reader) == depth+1);
      char *name = (char*)xmlTextReaderConstName(reader);
      if(strrchr(name, ':')) name = strrchr(name, ':')+1;
      if(name) {
	if(strcasecmp(name, "log") == 0) {
	  log_t *log = process_gpx_wpt_gc_logs_log(reader);
	  if(log) {
	    /* add log to chain */
	    log_t **cur = &log_chain;
	    while(*cur && log_is_older(log, *cur))
	      cur = &((*cur)->next);
	    
	    log->next = *cur;
	    *cur = log;
	  }
	} else
	  skip_element(reader);
      } else
	skip_element(reader);
      break;
      
    case XML_READER_TYPE_END_ELEMENT:
      /* end element must be for the current element */
      g_assert(xmlTextReaderDepth(reader) == depth);
      return log_chain;
      break;
      
    default:
      break;
    }
    ret = xmlTextReaderRead(reader);
  }

  /* free the entire log chain */
  while(log_chain) {
    log_t *next = log_chain->next;
    gpx_free_log(log_chain);
    log_chain = next;
  }

  return NULL;
}

static tb_t *process_gpx_wpt_gc_tbs_travelbug(xmlTextReaderPtr reader) {

  if(xmlTextReaderIsEmptyElement(reader))
    return NULL;

  /* create a new tb entry */
  tb_t *tb = malloc(sizeof(tb_t));
  memset(tb, 0, sizeof(tb_t));

  char *prop;
  if((prop = (char*)xmlTextReaderGetAttribute(reader, BAD_CAST "ref"))) 
    tb->ref = strdup(prop);
  else
    tb->ref = strdup("<NONE>");

  /* process all sub-nodes */
  int depth = xmlTextReaderDepth(reader);
  int ret = xmlTextReaderRead(reader);
  while(ret == 1) {
    
    switch(xmlTextReaderNodeType(reader)) {
    case XML_READER_TYPE_ELEMENT:
      g_assert(xmlTextReaderDepth(reader) == depth+1);
      char *name = (char*)xmlTextReaderConstName(reader);
      if(strrchr(name, ':')) name = strrchr(name, ':')+1;
      if(name) {
	if(strcasecmp(name, "name") == 0) {
	  if(!tb->name) tb->name = process_text(reader);
	} else
	  skip_element(reader);
      } else
	skip_element(reader);
      break;
      
    case XML_READER_TYPE_END_ELEMENT:
      /* end element must be for the current element */
      g_assert(xmlTextReaderDepth(reader) == depth);
      return tb;
      break;
      
    default:
      break;
    }
    ret = xmlTextReaderRead(reader);
  }

  gpx_free_tb(tb);
  return NULL;
}

static tb_t *process_gpx_wpt_gc_tbs(xmlTextReaderPtr reader) {
  tb_t *tb = NULL, **tbP = &tb;

  if(xmlTextReaderIsEmptyElement(reader))
    return tb;

  /* process all sub-nodes */
  int depth = xmlTextReaderDepth(reader);
  int ret = xmlTextReaderRead(reader);
  while(ret == 1) {
    
    switch(xmlTextReaderNodeType(reader)) {
    case XML_READER_TYPE_ELEMENT:
      g_assert(xmlTextReaderDepth(reader) == depth+1);
      char *name = (char*)xmlTextReaderConstName(reader);
      if(strrchr(name, ':')) name = strrchr(name, ':')+1;
      if(name) {
	if(strcasecmp(name, "travelbug") == 0) {
	  *tbP = process_gpx_wpt_gc_tbs_travelbug(reader);
	  if(*tbP) tbP = &(*tbP)->next;
	} else
	  skip_element(reader);
      } else
	skip_element(reader);
      break;
      
    case XML_READER_TYPE_END_ELEMENT:
      /* end element must be for the current element */
      g_assert(xmlTextReaderDepth(reader) == depth);
      return tb;
      break;
      
    default:
      break;
    }
    ret = xmlTextReaderRead(reader);
  }

  while(tb) {
    tb_t *next = tb;
    gpx_free_tb(tb);
    tb = next;
  }

  return NULL;
}

static void process_gpx_wpt_gc(xmlTextReaderPtr reader, cache_t *cache) {
  cache->available = xml_prop_is(reader, "available", "true", TRUE);
  cache->archived = xml_prop_is(reader, "archived", "true", FALSE);

  if(xmlTextReaderIsEmptyElement(reader))
    return;

  /* process all sub-nodes */
  int depth = xmlTextReaderDepth(reader);
  int ret = xmlTextReaderRead(reader);
  while(ret == 1) {
    
    switch(xmlTextReaderNodeType(reader)) {
    case XML_READER_TYPE_ELEMENT:
      g_assert(xmlTextReaderDepth(reader) == depth+1);
      char *name = (char*)xmlTextReaderConstName(reader);
      if(strrchr(name, ':')) name = strrchr(name, ':')+1;
      if(name) {
	if(strcasecmp(name, "name") == 0) {
	  if(!cache->name) cache->name = process_text(reader);
	} else if(strcasecmp(name, "owner") == 0) {
	  if(!cache->owner) cache->owner = process_text(reader);
	} else if(strcasecmp(name, "type") == 0) {
	  cache->type = xml_str_search(reader, cache_type_str, 
				       "cache type", CACHE_TYPE_UNKNOWN);
	} else if(strcasecmp(name, "container") == 0) {
	  cache->container = xml_str_search(reader, cache_container_str, 
					    "container", CACHE_CONT_UNKNOWN);
	} else if((strcasecmp(name, "short_description") == 0) ||
		  (strcasecmp(name, "summary") == 0))	{
	  if(!cache->short_description) {
	    cache->short_description = process_text(reader);
	    cache->short_is_html = xml_prop_is(reader, "html", "true", FALSE);
	  }
	} else if((strcasecmp(name, "long_description") == 0) ||
		  (strcasecmp(name, "description") == 0))	{
	  if(!cache->long_description) {
	    cache->long_description = process_text(reader);
	    cache->long_is_html = xml_prop_is(reader, "html", "true", FALSE);
	  }
	} else if((strcasecmp(name, "encoded_hints") == 0) ||
		  (strcasecmp(name, "hints") == 0))	{
	  if(!cache->hint) {
	    cache->hint = process_text(reader);
	    /* often hints aren't more than just a bunch of blanks ... */
	    if(cache->hint && all_is_white(cache->hint)) {
	      free(cache->hint);
	      cache->hint = NULL;
	    } else
	       cache->hint_is_html = xml_prop_is(reader, "html", "true", FALSE);
	  }
	} else if(strcasecmp(name, "difficulty") == 0) {
	  cache->difficulty = xml_float(reader, 0.0);
	} else if(strcasecmp(name, "terrain") == 0) {
	  cache->terrain = xml_float(reader, 0.0);
	} else if(strcasecmp(name, "logs") == 0) {
	  if(!cache->log) cache->log = process_gpx_wpt_gc_logs(reader);
	} else if(strcasecmp(name, "travelbugs") == 0) {
	  if(!cache->tb) cache->tb = process_gpx_wpt_gc_tbs(reader);
	} else {
	  //	  printf("unhandled item found: gpx/wpt/cache/%s\n", name);
	  skip_element(reader);
	}
      } else
	skip_element(reader);
      break;
      
    case XML_READER_TYPE_END_ELEMENT:
      /* end element must be for the current element */
      g_assert(xmlTextReaderDepth(reader) == depth);
      return;
      break;
      
    default:
      break;
    }
    ret = xmlTextReaderRead(reader);
  }
}

/* parse waypoint entry */
static cache_t *process_gpx_wpt(xmlTextReaderPtr reader,  gpx_dialog_t *dialog,
				gpx_t *gpx) {
  char *cmt = NULL, *desc = NULL;
  char *sym = NULL;

  gpx_busy_dialog_progress(dialog);

  if(xmlTextReaderIsEmptyElement(reader))
    return NULL;

  cache_t *cache = malloc(sizeof(cache_t));
  memset(cache, 0, sizeof(cache_t));

  /* set some defaults */
  cache->type = CACHE_TYPE_UNKNOWN;
  cache->container = CACHE_CONT_UNKNOWN;
  cache->available = TRUE;

  /* parse attributes */
  cache->pos.lat = xml_get_prop_float(reader, "lat");
  cache->pos.lon = xml_get_prop_float(reader, "lon");

  /* process all sub-nodes */
  int depth = xmlTextReaderDepth(reader);
  int ret = xmlTextReaderRead(reader);
  while(ret == 1) {
    
    switch(xmlTextReaderNodeType(reader)) {
    case XML_READER_TYPE_ELEMENT:
      g_assert(xmlTextReaderDepth(reader) == depth+1);
      char *name = (char*)xmlTextReaderConstName(reader);

      if(strrchr(name, ':')) name = strrchr(name, ':')+1;

      if(name) {
	if(strcasecmp(name, "name") == 0) {
	  if(!cache->id) cache->id = process_text(reader);
	} else if(strcasecmp(name, "url") == 0) {
	  if(!cache->url) cache->url = process_text(reader);
	} else if((strcasecmp(name, "cache") == 0) || 
		  (strcasecmp(name, "geocache") == 0)) {
	  process_gpx_wpt_gc(reader, cache);

	  /* the following are used if the current entry is a waypoint */
	} else if(strcasecmp(name, "cmt") == 0) {
	  if(!cmt) cmt = process_text(reader);
	} else if(strcasecmp(name, "desc") == 0) {
	  if(!desc) desc = process_text(reader);
	} else if(strcasecmp(name, "sym") == 0) {
	  if(!sym) sym = process_text(reader);
	} else {
	  skip_element(reader);
	}
      } else
	skip_element(reader);
      break;
      
    case XML_READER_TYPE_END_ELEMENT:
      /* end element must be for the current element */
      g_assert(xmlTextReaderDepth(reader) == depth);

      /* ------------ cleanup -------------- */

      /* special handling for opencaching.de caches */
      if(cache->id && strncasecmp(cache->id, "OC", 2) == 0) {
	/* the html attributes are either missing or wrong on OC ... */
	cache->long_is_html = TRUE;
	cache->hint_is_html = TRUE;
	cache->logs_are_html = TRUE;
      }

      /* neither geocaching.com GC* nor opencaching.com OC* nor */
      /* geocaching australia GA* waypoint */
      if(cache->id &&
	 (strncasecmp(cache->id, "__", 2) != 0) &&
	 (strncasecmp(cache->id, "GC", 2) != 0) &&
	 (strncasecmp(cache->id, "OC", 2) != 0) &&
	 (strncasecmp(cache->id, "GA", 2) != 0)) {
	cache_t *parent = gpx->cache;

	/* check if the gpx file contains a cache with matching name */
	while(parent && strcasecmp(parent->id+2, cache->id+2)) 
	  parent = parent->next;

	if(parent && parent != cache) {
	  wpt_t **wpt = &parent->wpt;

	  /* search end of list */
	  while(*wpt && (strcmp((*wpt)->id, cache->id)<0))
	    wpt = &(*wpt)->next;

	  *wpt = malloc(sizeof(wpt_t));
	  memset(*wpt, 0, sizeof(wpt_t));
      
	  /* transfer name to waypoint entry */
	  (*wpt)->id = cache->id;
	  cache->id = NULL;

	  (*wpt)->pos.lat = cache->pos.lat;
	  (*wpt)->pos.lon = cache->pos.lon;

	  (*wpt)->cmt = cmt;   cmt = NULL;
	  (*wpt)->desc = desc; desc = NULL;
	  (*wpt)->sym = str_search(wpt_sym_str, sym, "wpt sym");
	  
	  /* and just free the current cache entry as we now have used */
	  /* the data for a caches waypoint */
	  gpx_free_cache(cache);
	  cache = NULL;
	} else {
	  /* if it doesn't have a name etc, it's probably not a real */
	  /* cache, so drop it */
	  if(!cache->name || !cache->id) {
	    printf("Orphaned waypoint: %s\n", cache->id);
	    gpx_free_cache(cache);
	    cache = NULL;
	  }
	}
      } else {
	if(!cache->id)
	  cache->id = g_strdup_printf("NO ID");

	/* this is known to be a geocache due to its waypoint name */
	/* (gc*, oc*, ga*) and is thus forces to be an entry */
	if(!cache->name) 
	  cache->name = g_strdup_printf("Unnamed(%s)", cache->id);
      }

      if(desc) free(desc);
      if(cmt) free(cmt);
      if(sym) free(sym);
      
      return cache;
      break;
      
    default:
      break;
    }
    ret = xmlTextReaderRead(reader);
  }

  gpx_free_cache(cache);
  return NULL;
}

static gboolean process_gpx(xmlTextReaderPtr reader, gpx_dialog_t *dialog, 
			    gpx_t *gpx) {

  /* no attributes of interest */

  /* the following might be optimized for speed reasons! */

  /* find end of cache chain */
  cache_t **cache = &gpx->cache;
  while(*cache) cache = &(*cache)->next;

  const xmlChar *name = xmlTextReaderConstName(reader);
  if(!name) return FALSE;

  /* read next node */
  int ret = xmlTextReaderRead(reader);
  while(ret == 1) {

    switch(xmlTextReaderNodeType(reader)) {
    case XML_READER_TYPE_ELEMENT:

      g_assert(xmlTextReaderDepth(reader) == 1);
      char *name = (char*)xmlTextReaderConstName(reader);
      if(name && !gpx->name && strcasecmp(name, "name") == 0) {
	gpx->name = process_text(reader);
      } else if(name && !gpx->desc && strcasecmp(name, "desc") == 0) {
	gpx->desc = process_text(reader);
      } else if(name && ((strcasecmp(name, "time") == 0) || 
			 (strcasecmp(name, "date") == 0))) {
	xml_get_date(reader, &gpx->year, &gpx->month, &gpx->day);
      } else if(name && strcasecmp(name, "wpt") == 0) {
	*cache = process_gpx_wpt(reader, dialog, gpx);
	if(*cache) cache = &(*cache)->next;
      } else {
	//	printf("something unknown (%s) found\n", name);
	skip_element(reader);
      }
      break;
      
    case XML_READER_TYPE_END_ELEMENT:
      /* end element must be for the current element */
      g_assert(xmlTextReaderDepth(reader) == 0);
      return TRUE;
      break;
      
    default:
      break;
    }
    ret = xmlTextReaderRead(reader);
  }

  return FALSE;
}

/* parse loc waypoint entry */
static cache_t *process_loc_waypoint(xmlTextReaderPtr reader,
				     gpx_dialog_t *dialog) {

  gpx_busy_dialog_progress(dialog);
  
  if(xmlTextReaderIsEmptyElement(reader))
    return NULL;

  cache_t *cache = malloc(sizeof(cache_t));
  memset(cache, 0, sizeof(cache_t));

  /* set some defaults */
  cache->type = CACHE_TYPE_TRADITIONAL;
  cache->container = CACHE_CONT_UNKNOWN;
  cache->available = TRUE;

  /* process all sub-nodes */
  int depth = xmlTextReaderDepth(reader);
  int ret = xmlTextReaderRead(reader);
  while(ret == 1) {
    
    switch(xmlTextReaderNodeType(reader)) {
    case XML_READER_TYPE_ELEMENT:
      g_assert(xmlTextReaderDepth(reader) == depth+1);
      char *name = (char*)xmlTextReaderConstName(reader);
      if(strrchr(name, ':')) name = strrchr(name, ':')+1;

      if(name) {
	if(strcasecmp(name, "name") == 0) {
	  cache->id = (char*)xmlTextReaderGetAttribute(reader, BAD_CAST "id");
	  cache->name = process_text(reader);
	} else if(strcasecmp(name, "link") == 0) {
	  cache->url = process_text(reader); 
	} else if(strcasecmp(name, "coord") == 0) {
	  cache->pos.lat = xml_get_prop_float(reader, "lat");
	  cache->pos.lon = xml_get_prop_float(reader, "lon");
	  skip_element(reader);
	} else 
	  skip_element(reader);
      } else 
	skip_element(reader);
      break;
      
      case XML_READER_TYPE_END_ELEMENT:
      /* end element must be for the current element */
      g_assert(xmlTextReaderDepth(reader) == depth);
      return cache;
      break;
      
    default:
      break;
    }
    ret = xmlTextReaderRead(reader);
  }
  
  g_assert(0);
  return NULL;
}

static void process_loc(xmlTextReaderPtr reader, gpx_dialog_t *dialog, 
			gpx_t *gpx) {

  /* find end of cache chain */
  cache_t **cache = &gpx->cache;
  while(*cache) cache = &(*cache)->next;

  const xmlChar *name = xmlTextReaderConstName(reader);
  g_assert(name);

  /* read next node */
  int ret = xmlTextReaderRead(reader);
  while(ret == 1) {

    switch(xmlTextReaderNodeType(reader)) {
    case XML_READER_TYPE_ELEMENT:

      g_assert(xmlTextReaderDepth(reader) == 1);
      char *name = (char*)xmlTextReaderConstName(reader);

      if(name && strcasecmp(name, "waypoint") == 0) {
	*cache = process_loc_waypoint(reader, dialog);
	if(*cache) cache = &(*cache)->next;
      } else
	skip_element(reader);
      break;
      
    case XML_READER_TYPE_END_ELEMENT:
      /* end element must be for the current element */
      g_assert(xmlTextReaderDepth(reader) == 0);
      return;
      break;
      
    default:
      break;
    }
    ret = xmlTextReaderRead(reader);
  }

  g_assert(0);
  return;
}

static gpx_t *process_root(xmlTextReaderPtr reader, gpx_dialog_t *dialog, 
			   char *fname, gpx_t *in) {

  /* no gpx entry given, create a new one */
  gpx_t *gpx = NULL;
  if(!in) {
    /* allocate memory to hold gpx file description */
    gpx = malloc(sizeof(gpx_t));
    memset(gpx, 0, sizeof(gpx_t));
    gpx->filename = strdup(fname);
  } else
    gpx = in;  

  int ret = xmlTextReaderRead(reader);
  while(ret == 1) {
    switch(xmlTextReaderNodeType(reader)) {
    case XML_READER_TYPE_ELEMENT:
      g_assert(xmlTextReaderDepth(reader) == 0);
      char *name = (char*)xmlTextReaderConstName(reader);
      if(name && strcasecmp(name, "gpx") == 0) {
	process_gpx(reader, dialog, gpx);
      } else if(name && strcasecmp(name, "loc") == 0) {
	process_loc(reader, dialog, gpx);
      } else {
	printf("something unknown found\n");
	skip_element(reader);
      }
      break;
      
    case XML_READER_TYPE_END_ELEMENT:
      /* end element must be for the current element */
      g_assert(xmlTextReaderDepth(reader) == 0);
      ret = -1;
      break;
      
    default:
      break;
    }

    if(ret == 1)
      ret = xmlTextReaderRead(reader);
  }
  
  /* check if a name has been set and use filename if not */
  if(!in && !gpx->name) {
    if(!gpx->desc) { 
      char *str = strrchr(fname, '/');
      if(str) gpx->name = strdup(str+1);
      else    gpx->name = strdup(fname);
    } else
      gpx->name = strdup(gpx->desc);
  }

  return gpx;
}

static gpx_t *gpx_parse_file(gpx_dialog_t *dialog, char *filename) {
  gpx_t *gpx = NULL;

  LIBXML_TEST_VERSION;

  gpx_busy_dialog_set(dialog, filename);

  xmlTextReaderPtr reader = xmlReaderForFile(filename, NULL, 0);
  if (reader != NULL) {
    gpx = process_root(reader, dialog, filename, NULL);
    xmlFreeTextReader(reader);
  } else {
    fprintf(stderr, "Unable to open %s\n", filename);
  }

  /* check of there's a waypoints file (*-wpts.gpx) for this */
  if(strrchr(filename, '.')) {
    char *dot = strrchr(filename, '.');
    char wpts_name[128];
    *dot = 0;
    snprintf(wpts_name, sizeof(wpts_name), "%s-wpts.gpx", filename);
    *dot = '.';
    if(g_file_test(wpts_name,  G_FILE_TEST_EXISTS)) {
      xmlTextReaderPtr reader = xmlReaderForFile(wpts_name, NULL, 0);
      if (reader != NULL) {
	gpx = process_root(reader, dialog, wpts_name, gpx);
	xmlFreeTextReader(reader);
      } else {
	fprintf(stderr, "Unable to open %s\n", filename);
      }
    }
  }

  return gpx;
}

static gpx_t *decompress_file(unzFile file, gpx_dialog_t *dialog, 
			      char *name, char *filename,
			      gpx_t *gpx_in) {
  unz_file_info info;
  gpx_t *gpx = NULL;
  
  gpx_busy_dialog_set(dialog, name);

  if((unzLocateFile(file, name, FALSE) != Z_OK) ||
     (unzGetCurrentFileInfo(file, &info, NULL,0, NULL,0, NULL,0) != Z_OK) ||
     (unzOpenCurrentFile(file) != UNZ_OK)) {

    /* do not complain if we are processing a waypoints file */
    if(!gpx_in)
      errorf("Unable to locate/get info/open\n%s\ninside\n%s",
	     name, filename);
    else
      printf("Unable to locate/get info/open %s inside %s\n",
	     name, filename);

    return gpx_in;
  }

  printf("file size is %ld\n", info.uncompressed_size);

  char *buffer = malloc(info.uncompressed_size);
  if(!buffer) {
    errorf("Out of memory while uncompressing file");
    unzCloseCurrentFile(file);
    return gpx_in;
  }

  if(unzReadCurrentFile(file, buffer, info.uncompressed_size) < 0) {
    errorf("Read error on compressed file");
    free(buffer);
    unzCloseCurrentFile(file);
    return gpx_in;
  }

  /* fire up libxml */
  LIBXML_TEST_VERSION;

  xmlTextReaderPtr reader = 
    xmlReaderForMemory(buffer, info.uncompressed_size, 
		       NULL, NULL, 0);
  if (reader != NULL) {
    gpx = process_root(reader, dialog, filename, gpx_in);
    xmlFreeTextReader(reader);
  } else {
    fprintf(stderr, "Unable to open %s\n", filename);
  }

  free(buffer);
  unzCloseCurrentFile(file);
  return gpx;
}

static gpx_t *decompress_zip(gpx_dialog_t *dialog, char *filename) {
  char *gpx_name, *fbase;
  gpx_t *gpx = NULL;

  /* extract base name and allocate space for file names */
  fbase = strrchr(filename, '/');
  if(!fbase) fbase = filename;
  else       fbase++;  /* skip '/' */
  gpx_name = malloc(strlen(fbase)+strlen("-wpts")+1);

  unzFile file = unzOpen(filename);
  if(!file) {
    errorf("Error opening file %s for unzip", filename);
    free(gpx_name);
    return NULL;
  }

  printf("ZIP file successfully opened\n");

  /* try to open gpx file inside */
  strcpy(gpx_name, fbase);
  strcpy(gpx_name+strlen(gpx_name)-4, ".gpx");
  printf("gpx file name is %s\n", gpx_name);

  gpx = decompress_file(file, dialog, gpx_name, filename, NULL);

  /* try to open -wpts.gpx file inside */
  strcpy(gpx_name, fbase);
  strcpy(gpx_name+strlen(gpx_name)-4, "-wpts.gpx");
  printf("gpx wpts file name is %s\n", gpx_name);

  gpx = decompress_file(file, dialog, gpx_name, filename, gpx);

  unzClose(file);
  free(gpx_name);
  return gpx;
}

gpx_t *gpx_parse(gpx_dialog_t *dialog, char *filename) {
  gpx_t *gpx = NULL;

  /* show busy dialog */
  printf("load file %s\n", filename);

  if((strlen(filename) > 4) &&
     !strcasecmp(filename+strlen(filename)-4, ".zip")) {
    printf("trying to load a zip file!\n");

    gpx = decompress_zip(dialog, filename);
  } else 
    gpx = gpx_parse_file(dialog, filename);

  return gpx;
}

/* scan entire directory */
gpx_t *gpx_parse_dir(gpx_dialog_t *dialog, char *dirname) {
  GnomeVFSResult result;
  GnomeVFSDirectoryHandle *handle;
  GnomeVFSFileInfo *finfo = gnome_vfs_file_info_new();;

  gpx_t *gpx = NULL;
  
  /* show busy dialog */
  printf("load dir %s\n", dirname);
  gpx_busy_dialog_set(dialog, dirname);

  LIBXML_TEST_VERSION;

  result = gnome_vfs_directory_open(&handle, dirname, 
				    GNOME_VFS_FILE_INFO_DEFAULT);

  if(result != GNOME_VFS_OK) {
    errorf("Unable to open directory \"%s\":\n%s",
	   dirname,  gnome_vfs_result_to_string(result));
    return NULL;
  }

  while(GNOME_VFS_OK == gnome_vfs_directory_read_next(handle, finfo)) {
    if(finfo->type == GNOME_VFS_FILE_TYPE_REGULAR) {
      char *ext = finfo->name+strlen(finfo->name)-4;

      /* check if file ends with .gpx or .loc */
      if((strcasecmp(ext, ".gpx") == 0) || (strcasecmp(ext, ".loc") == 0)) {
	char *filename = malloc(strlen(dirname)+strlen(finfo->name)+2);

	strcpy(filename, dirname);
	if(strlastchr(filename) != '/')
	  strcat(filename, "/");
	strcat(filename, finfo->name);

	xmlTextReaderPtr reader = xmlReaderForFile(filename, NULL, 0);
	if (reader != NULL) {
	  gpx = process_root(reader, dialog, filename, gpx);
	  xmlFreeTextReader(reader);
	} else {
	  fprintf(stderr, "Unable to open %s\n", filename);
	}

	free(filename);
      }
    }
  }

  if(gpx) {
    /* replace file name with directory name */
    free(gpx->filename);
    gpx->filename = strdup(dirname);

    /* replace gpx name with directory name */
    free(gpx->name);

    /* retrieve pure dirname if possible */
    char *n = strrchr(dirname, '/');
    if(!n) n = dirname;
    else   n++;

    //    gpx->name = malloc(strlen("<DIR> ")+strlen(n)+1);
    //    strcpy(gpx->name, "<DIR> ");
    //    strcat(gpx->name, n);
    gpx->name = strdup(n);
  }

  gnome_vfs_file_info_unref(finfo);
  gnome_vfs_directory_close(handle);

  return gpx;
}

/* return number of caches in given gpx file */
int gpx_total_caches(gpx_t *gpx) {
  cache_t *cache = gpx->cache;
  int num = 0;
  
  while(cache) {
    num++;
    cache = cache->next;
  }

  return num;
}

int gpx_number_of_waypoints(wpt_t *wpt) {
  int num = 0;
  
  while(wpt) {
    num++;
    wpt = wpt->next;
  }

  return num;
}

int gpx_number_of_logs(log_t *log) {
  int num = 0;
  
  while(log) {
    num++;
    log = log->next;
  }

  return num;
}

int gpx_number_of_tbs(tb_t *tb) {
  int num = 0;
  
  while(tb) {
    num++;
    tb = tb->next;
  }

  return num;
}

/* http://mathforum.org/library/drmath/view/55417.html */
float gpx_pos_get_bearing(pos_t p1, pos_t p2) {
  /* convert to radians */
  p1.lat *= (M_PI/180.0); p1.lon *= (M_PI/180.0);
  p2.lat *= (M_PI/180.0); p2.lon *= (M_PI/180.0);

  return fmodf(360.0 + (180.0/M_PI) * 
	       (atan2( sin(p2.lon - p1.lon) * cos(p2.lat),
		       cos(p1.lat) * sin(p2.lat) - 
		       sin(p1.lat) * cos(p2.lat) * cos(p2.lon - p1.lon))),
	       360.0);
}

/* http://mathforum.org/library/drmath/view/51722.html */
float gpx_pos_get_distance(pos_t p1, pos_t p2, int miles) {
  /* convert to radians */
  p1.lat *= (M_PI/180.0); p1.lon *= (M_PI/180.0);
  p2.lat *= (M_PI/180.0); p2.lon *= (M_PI/180.0);

  float aob = acos(cos(p1.lat) * cos(p2.lat) * cos(p2.lon - p1.lon) +
		   sin(p1.lat) * sin(p2.lat));

  if(miles) 
    return(aob * 3959.0);   /* great circle radius in miles */

  return(aob * 6371.0);     /* great circle radius in kilometers */
}

void gpx_pos_get_distance_str(char *str, int len, 
			      pos_t p1, pos_t p2, int mil) {
  if(isnan(p1.lat) || isnan(p1.lon)) {
    snprintf(str, len, "---");
    return;
  }

  float dist = gpx_pos_get_distance(p1, p2, mil);
  distance_str(str, len, dist, mil);
}

void gpx_sort(gpx_t *gpx, int by, pos_t *refpos) {
  cache_t **new;
  cache_t *cur = gpx->cache;
  int total = gpx_total_caches(gpx);
  float *dist_cache = malloc(total * sizeof(float));

  gpx->cache = NULL;  /* detach old chain */
  while(cur) {
    float cur_dist = -1;
    int cur_cnt = 0;

    if(!isnan(cur->pos.lat) && !isnan(cur->pos.lon)) 
      cur_dist = gpx_pos_get_distance(*refpos, gpx_cache_pos(cur), 0);
    
    new = &(gpx->cache);

    /* search for currect insertion point */
    while(*new && (dist_cache[cur_cnt] < cur_dist)) {
      new = &((*new)->next);
      cur_cnt++;
    }

    /* save distance for further comparisons */
    memmove(dist_cache+cur_cnt+1, dist_cache+cur_cnt, 
	    sizeof(float)*(total-cur_cnt-1));
    dist_cache[cur_cnt++] = cur_dist;
    
    cache_t *next = cur->next;

    /* insert into "new" chain */
    cur->next = *new;
    *new = cur;

    cur = next;
  }

  free(dist_cache);
}

gpx_t *gpx_cache2gpx(gpx_t *gpx, cache_t *search_cache) {
  while(gpx) {
    cache_t *cache = gpx->cache;
    while(cache) {
      if(cache == search_cache)
	return gpx;

      cache = cache->next;
    }
    gpx = gpx->next;
  }

  return NULL;
}

/* since the actual cache position may be overridden, we */
/* always access the position through this function */
pos_t gpx_cache_pos(cache_t *cache) {
  if(cache->notes && cache->notes->override) 
    return cache->notes->pos;

  return cache->pos;
}
