/*
 * Copyright (C) 2008-2009 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"
#include "converter.h"
#include <math.h>    // for isnan

#ifdef ENABLE_OSM_GPS_MAP
#include "osm-gps-map.h"
#endif

#define MAP_SOURCE  OSM_GPS_MAP_SOURCE_OPENSTREETMAP
#define GPS_DEFAULT_ZOOM 13

#define PROXY_KEY  "/system/http_proxy/"

static const char *get_proxy_uri(appdata_t *appdata) {
  static char proxy_buffer[64] = "";
  
  /* use environment settings if preset */
  const char *proxy = g_getenv("http_proxy");
  if(proxy) {
    printf("http_proxy: %s\n", proxy);
    return proxy;
  }

  /* ------------- get proxy settings -------------------- */
  if(gconf_client_get_bool(appdata->gconf_client, 
			   PROXY_KEY "use_http_proxy", NULL)) {

    /* we can savely ignore things like "ignore_hosts" since we */
    /* are pretty sure not inside the net of one of our map renderers */
    /* (unless the user works at google :-) */
      
    /* get basic settings */
    char *host = 
      gconf_client_get_string(appdata->gconf_client, PROXY_KEY "host", NULL);
    if(host) {
      int port =
	gconf_client_get_int(appdata->gconf_client, PROXY_KEY "port", NULL);

      snprintf(proxy_buffer, sizeof(proxy_buffer),
	       "http://%s:%u", host, port);

      g_free(host);
    }
    return proxy_buffer;
  }

  return NULL;
}

static void
cb_map_gps(map_context_t *context) {
  pos_t *refpos = get_pos(context->appdata);
  if(refpos && !isnan(refpos->lat) && !isnan(refpos->lon)) {
    osm_gps_map_set_mapcenter(OSM_GPS_MAP(context->widget),
		      refpos->lat, refpos->lon, GPS_DEFAULT_ZOOM);     
  } else {
    /* no coordinates given: display the entire world */
    osm_gps_map_set_mapcenter(OSM_GPS_MAP(context->widget), 
			      0.0, 0.0, 1);
  }
}

static int dist2pixel(map_context_t *context, float km, float lat) {
  return 1000.0*km/osm_gps_map_get_scale(OSM_GPS_MAP(context->widget));
}

static gboolean map_gps_update(gpointer data) {
  map_context_t *context = (map_context_t*)data;

  /* get reference position ... */
  pos_t *refpos = get_pos(context->appdata);
  gboolean ok = (refpos!= NULL) && !isnan(refpos->lat) && !isnan(refpos->lon);

  /* ... and enable "goto" button if it's valid */
  osm_gps_map_osd_enable_gps (OSM_GPS_MAP(context->widget), 
			      OSM_GPS_MAP_OSD_GPS_CALLBACK(ok?cb_map_gps:NULL), context);

  if(ok) {
    float heading = NAN;
    int radius = 0;

    if(context->appdata->use_gps) {
      heading = gps_get_heading(context->appdata);

      /* get error */
      float eph = gps_get_eph(context->appdata);
      if(!isnan(eph)) 
	radius = dist2pixel(context, eph/1000, refpos->lat);
    }

    g_object_set(context->widget, "gps-track-highlight-radius", radius, NULL);
    osm_gps_map_draw_gps(OSM_GPS_MAP(context->widget), 
			 refpos->lat, refpos->lon, heading);
  } else
    osm_gps_map_clear_gps(OSM_GPS_MAP(context->widget));
  
  return TRUE;
}

static gboolean on_map_configure(GtkWidget *widget,
				 GdkEventConfigure *event,
				 map_context_t *context) {

  /* set default values if they are invalid */
  if(!context->appdata->map.zoom || 
     isnan(context->appdata->map.pos.lat) ||
     isnan(context->appdata->map.pos.lon)) {
    printf("no valid map position found\n");
    
    pos_t *refpos = get_pos(context->appdata);
    if(refpos && !isnan(refpos->lat) && !isnan(refpos->lon)) {
      /* use gps position if present */
      context->appdata->map.pos = *refpos;
      context->appdata->map.zoom = GPS_DEFAULT_ZOOM;
    } else {
      /* use world map otherwise */
      context->appdata->map.pos.lat = 0.0;
      context->appdata->map.pos.lon = 0.0;
      context->appdata->map.zoom = 1;
    }
  }

  /* jump to initial position */
  osm_gps_map_set_mapcenter(OSM_GPS_MAP(context->widget), 
			    context->appdata->map.pos.lat, 
			    context->appdata->map.pos.lon, 
			    context->appdata->map.zoom);
  
  return FALSE;
}

static void map_draw_cachelist(GtkWidget *map, cache_t *cache) {
  while(cache) {
    GdkPixbuf *icon = icon_get(ICON_CACHE_TYPE, cache->type);
    
    osm_gps_map_add_image(OSM_GPS_MAP(map), 
			  cache->pos.lat, cache->pos.lon, icon);

    cache = cache->next;
  }
}

static void
map_cachelist_nearest(cache_t *cache, pos_t *pos, 
		      cache_t **result, float *distance) {
  while(cache) {
    float dist = 
      pow(cache->pos.lat - pos->lat, 2) + 
      pow(cache->pos.lon - pos->lon, 2);

    if(!(dist > *distance)) {
      *result = cache;
      *distance = dist;
    }

    cache = cache->next;
  }
}

static cache_t *map_closest(map_context_t *context, pos_t *pos) {
  cache_t *result = NULL;
  float distance = NAN;
  
#ifdef USE_MAEMO
  if(!context->appdata->cur_gpx) {
#endif
    /* search all geocaches */
    gpx_t *gpx = context->appdata->gpx;
    while(gpx) {
      map_cachelist_nearest(gpx->cache, pos, &result, &distance);
      gpx = gpx->next;
    }
#ifdef USE_MAEMO
  } else {
    map_cachelist_nearest(context->appdata->cur_gpx->cache, 
			  pos, &result, &distance);
  }
#endif

  return result;
}

/* translate between osm-gps-map positions and gpxview ones */
pos_t coord2pos(coord_t coo) {
  pos_t pos;
  pos.lat = rad2deg(coo.rlat);
  pos.lon = rad2deg(coo.rlon);
  return pos;
}

#define CLICK_FUZZ (24)

static gboolean
on_map_button_press_event(GtkWidget *widget, 
			    GdkEventButton *event, map_context_t *context) {
  OsmGpsMap *map = OSM_GPS_MAP(context->widget);

  /* got a press event without release event? eat it! */
  if(context->press_on != NULL) {
    printf("PRESS: already\n");
    return TRUE;
  }

  pos_t pos = 
    coord2pos(osm_gps_map_get_co_ordinates(map, event->x, event->y));

  cache_t *nearest = map_closest(context, &pos);  
  if(nearest) {
    float dist = gpx_pos_get_distance(pos, nearest->pos, FALSE);
    if(dist2pixel(context, dist, nearest->pos.lat) < CLICK_FUZZ) 
      context->press_on = nearest;
  }
  
  return FALSE;
}

static void
cairo_draw_pixbuf(cairo_t *cr, GdkPixbuf *buf, gint x, gint y) {
  /* convert the pixbuf into something cairo can handle */

  // Create a new ImageSurface
  cairo_surface_t *image_surface = 
    cairo_image_surface_create(CAIRO_FORMAT_ARGB32, 
			       gdk_pixbuf_get_width(buf),
			       gdk_pixbuf_get_height(buf));

  // Create the new Context for the ImageSurface
  cairo_t *context = cairo_create(image_surface);

  // Draw the image on the new Context
  gdk_cairo_set_source_pixbuf(context, buf, 0.0, 0.0);
  cairo_paint(context);

  // now draw this onto the original context 
  cairo_set_source_surface(cr, image_surface, x, y);

  cairo_paint(cr);
}

#ifndef BIG_BALLOONS
#define LINE_SKIP  7
#else
#define LINE_SKIP  12
#endif

static void 
balloon_draw_cb(cairo_t *cr, OsmGpsMapRect_t *rect, gpointer data) {
  cache_t *cache = (cache_t*)data;

  //  printf("draw cb for \"%s\"\n", cache->name);

#if 0
  /* draw pink background to check clipping */
  cairo_rectangle (cr, rect->x-20, rect->y-20, rect->w+40, rect->h+40);
  cairo_set_source_rgba (cr, 1, 0, 0, 0.3);
  cairo_fill_preserve (cr);
  cairo_set_line_width (cr, 0);
  cairo_stroke (cr);
#endif

  /* leave a little border top and left */
  gint x = rect->x, y = rect->y;

  /* draw the cache type icon ... */
  GdkPixbuf *icon = icon_get(ICON_CACHE_TYPE, cache->type);
  cairo_draw_pixbuf(cr, icon, x, y);

  /* ... and right of it the waypoint id */
  cairo_text_extents_t extents;

  if(cache->id) {
    cairo_select_font_face (cr, "Sans",
			    CAIRO_FONT_SLANT_NORMAL,
			    CAIRO_FONT_WEIGHT_BOLD);
  
#ifndef BIG_BALLOONS
    cairo_set_font_size (cr, 20.0);
#else
    cairo_set_font_size (cr, 36.0);
#endif

    cairo_text_extents (cr, cache->id, &extents);

    /* display id right of icon vertically centered */
    x += gdk_pixbuf_get_width(icon) + 5;
    y += (gdk_pixbuf_get_height(icon) + extents.height)/2;
    cairo_move_to (cr, x, y);
    cairo_set_source_rgba (cr, 0, 0, 0, 1);
    cairo_show_text (cr, cache->id);
    cairo_stroke (cr);

    y += (gdk_pixbuf_get_height(icon) - extents.height)/2 + LINE_SKIP;
  } else
    y += gdk_pixbuf_get_height(icon);

  /* return to the left border and below icon/text */
  x = rect->x;

  /* everything from here uses the same font */
  cairo_select_font_face (cr, "Sans", CAIRO_FONT_SLANT_NORMAL,
  			  CAIRO_FONT_WEIGHT_NORMAL);
#ifndef BIG_BALLOONS
  cairo_set_font_size (cr, 14.0);
#else
  cairo_set_font_size (cr, 22.0);
#endif

  if(cache->name) {
    /* draw cache name */
    cairo_text_extents (cr, cache->name, &extents);
    y += extents.height;
    cairo_move_to (cr, x, y);
    cairo_set_source_rgba (cr, 0, 0, 0, 1);
    cairo_show_text (cr, cache->name);
    cairo_stroke (cr);
  
    /* return to the left border and below text */
    y += LINE_SKIP;
    x = rect->x;
  }

  if(cache->terrain) {
    /* draw cache rating */
    const char *terrain = "Terrain:"; 
    icon = icon_get(ICON_STARS, (int)(cache->terrain*2-2));
    cairo_text_extents (cr, _(terrain), &extents);
    y += (gdk_pixbuf_get_height(icon) + extents.height)/2;

    /* draw "Terrain:" string */
    cairo_move_to (cr, x, y);
    cairo_set_source_rgba (cr, 0, 0, 0, 1);
    cairo_show_text (cr, _(terrain));
    cairo_stroke (cr);
    x += extents.width + 2;
  
    /* draw terrain stars */
    cairo_draw_pixbuf(cr, icon, x, y - 
		      (gdk_pixbuf_get_height(icon) + extents.height)/2);

    x += gdk_pixbuf_get_width(icon) + LINE_SKIP;
    y -= (gdk_pixbuf_get_height(icon) + extents.height)/2;
  }

  if(cache->difficulty) {
    const char *difficulty = "Difficulty:"; 
    cairo_text_extents (cr, _(difficulty), &extents);
    y += (gdk_pixbuf_get_height(icon) + extents.height)/2;
    
    /* draw "Difficulty:" string */
    cairo_move_to (cr, x, y);
    cairo_set_source_rgba (cr, 0, 0, 0, 1);
    cairo_show_text (cr, _(difficulty));
    cairo_stroke (cr);
    x += extents.width + 2;
    
    icon = icon_get(ICON_STARS, (int)(cache->difficulty*2-2));
    cairo_draw_pixbuf(cr, icon, x, y - 
		      (gdk_pixbuf_get_height(icon) + extents.height)/2);
  }
}

static gboolean
on_map_button_release_event(GtkWidget *widget, 
			    GdkEventButton *event, map_context_t *context) {
  OsmGpsMap *map = OSM_GPS_MAP(context->widget);

  if(context->press_on) {
    coord_t coo;
    coo = osm_gps_map_get_co_ordinates(map, event->x, event->y);

    pos_t pos = 
      coord2pos(osm_gps_map_get_co_ordinates(map, event->x, event->y));

    cache_t *nearest = map_closest(context, &pos);  
    if(nearest && nearest == context->press_on) {
      float dist = gpx_pos_get_distance(pos, nearest->pos, FALSE);
      if(dist2pixel(context, dist, nearest->pos.lat) < CLICK_FUZZ) {
	
	osm_gps_map_draw_balloon(map, nearest->pos.lat, nearest->pos.lon,
				 balloon_draw_cb, nearest);
      }
    }
    context->press_on = NULL;
  } else {
    /* save new map position */
    gfloat lat, lon;
    g_object_get(map, "latitude", &lat, "longitude", &lon, NULL);
    context->appdata->map.pos.lat = lat;
    context->appdata->map.pos.lon = lon;
  }

  return FALSE;
}

static void on_window_destroy(GtkWidget *widget, map_context_t *context) {
  appdata_t *appdata = context->appdata;
  
  printf("destroy map window\n");

  /* save map parameters */
  OsmGpsMap *map = OSM_GPS_MAP(context->widget);
  gint zoom;
  g_object_get(map, "zoom", &zoom, NULL);
  context->appdata->map.zoom = zoom;

  gfloat lat, lon;
  g_object_get(map, "latitude", &lat, "longitude", &lon, NULL);
  context->appdata->map.pos.lat = lat;
  context->appdata->map.pos.lon = lon;

#if MAEMO_VERSION_MAJOR == 5
  /* restore cur_view */
  context->appdata->cur_view = context->old_view;
#endif

  gtk_timeout_remove(context->handler_id);

  g_free(context);
  appdata->map.context = NULL;
}

void map(appdata_t *appdata) {
  map_context_t *context = NULL; 

  /* if the map window already exists, just raise it */
  if(appdata->map.context) {
    gtk_window_present(GTK_WINDOW(appdata->map.context->window));
    return;
  }

  context = appdata->map.context = g_new0(map_context_t, 1);
  context->appdata = appdata;

  char *path = g_strdup_printf("%s/map/", appdata->image_path);
  const char *proxy = get_proxy_uri(appdata);

  context->widget = g_object_new(OSM_TYPE_GPS_MAP,
		 "map-source",               MAP_SOURCE,
                 "tile-cache",               path,
		 "auto-center",              FALSE,
		 "record-trip-history",      FALSE, 
		 "show-trip-history",        FALSE, 
		 proxy?"proxy-uri":NULL,     proxy,
                 NULL);

  g_free(path);

  char *name = NULL;
#ifdef USE_MAEMO
  if(!appdata->cur_gpx) {
#endif
    /* draw all geocaches */
    gpx_t *gpx = appdata->gpx;
    while(gpx) {
      map_draw_cachelist(context->widget, gpx->cache);
      gpx = gpx->next;
    }
    name = g_strdup(_("all geocaches"));
#ifdef USE_MAEMO
  } else {
    map_draw_cachelist(context->widget, appdata->cur_gpx->cache);
    name = g_strdup(appdata->cur_gpx->name);
  }
#endif

  char *title = g_strdup_printf(_("Map - %s"), name);
  g_free(name);

#ifdef USE_MAEMO
#ifdef USE_STACKABLE_WINDOW
  context->window = hildon_stackable_window_new();
#else
  context->window = hildon_window_new();
#endif
#else
  context->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
#endif

  gtk_window_set_title(GTK_WINDOW(context->window), title);

#ifndef USE_MAEMO
  gtk_window_set_default_size(GTK_WINDOW(context->window), 640, 480);
#endif

  g_free(title);

  g_signal_connect(G_OBJECT(context->widget), "configure-event",
		   G_CALLBACK(on_map_configure), context);

  g_signal_connect(G_OBJECT(context->widget), "button-press-event",
                   G_CALLBACK(on_map_button_press_event), context);

  g_signal_connect(G_OBJECT(context->widget), "button-release-event",
                   G_CALLBACK(on_map_button_release_event), context);

  /* install handler for timed updates of the gps button */
  context->handler_id = gtk_timeout_add(1000, map_gps_update, context);

#if MAEMO_VERSION_MAJOR == 5
  /* prevent some of the main screen things */
  context->old_view = appdata->cur_view;
  appdata->cur_view = NULL;
#endif

  g_signal_connect(G_OBJECT(context->window), "destroy", 
		   G_CALLBACK(on_window_destroy), context);

  gtk_container_add(GTK_CONTAINER(context->window), context->widget); 
  gtk_widget_show_all(GTK_WIDGET(context->window));
}
