/*
 * Copyright (C) 2008-2009 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * http://topo.geofabrik.de/relief/${z}/${x}/${y}.png  8-15
 * http://topo.geofabrik.de/trail/${z}/${x}/${y}.png   8-15
 */

#include "gpxview.h"
#include "converter.h"
#include <math.h>    // for isnan

#ifdef ENABLE_OSM_GPS_MAP
#include "osm-gps-map.h"
#include "osm-gps-map-osd-classic.h"
#endif

#if defined(USE_MAEMO) && (MAEMO_VERSION_MAJOR == 5)
#include <gdk/gdkx.h>
#include <X11/Xatom.h>
#endif

/* default values */
#define MAP_SOURCE  OSM_GPS_MAP_SOURCE_OPENCYCLEMAP
#define GPS_DEFAULT_ZOOM 13

#define PROXY_KEY  "/system/http_proxy/"

static const char *get_proxy_uri(appdata_t *appdata) {
  static char proxy_buffer[64] = "";
  
  /* use environment settings if preset */
  const char *proxy = g_getenv("http_proxy");
  if(proxy) {
    printf("http_proxy: %s\n", proxy);
    return proxy;
  }

  /* ------------- get proxy settings -------------------- */
  if(gconf_client_get_bool(appdata->gconf_client, 
			   PROXY_KEY "use_http_proxy", NULL)) {

    /* we can savely ignore things like "ignore_hosts" since we */
    /* are pretty sure not inside the net of one of our map renderers */
    /* (unless the user works at google :-) */
      
    /* get basic settings */
    char *host = 
      gconf_client_get_string(appdata->gconf_client, PROXY_KEY "host", NULL);
    if(host) {
      int port =
	gconf_client_get_int(appdata->gconf_client, PROXY_KEY "port", NULL);

      snprintf(proxy_buffer, sizeof(proxy_buffer),
	       "http://%s:%u", host, port);

      g_free(host);
    }
    return proxy_buffer;
  }

  return NULL;
}

static void
cb_map_gps(osd_button_t but, map_context_t *context) {

  if(but == OSD_GPS) {
    pos_t *refpos = get_pos(context->appdata);
    if(refpos && !isnan(refpos->lat) && !isnan(refpos->lon)) {
      gint zoom;
      g_object_get(OSM_GPS_MAP(context->widget), "zoom", &zoom, NULL);
      if(zoom < 10)
	osm_gps_map_set_mapcenter(OSM_GPS_MAP(context->widget),
				  refpos->lat, refpos->lon, GPS_DEFAULT_ZOOM);
      else
	osm_gps_map_set_center(OSM_GPS_MAP(context->widget),
			       refpos->lat, refpos->lon);

      /* re-enable centering */
      g_object_set(context->widget, "auto-center", TRUE, NULL);
    } else {
      /* no coordinates given: display the entire world */
      osm_gps_map_set_mapcenter(OSM_GPS_MAP(context->widget), 
				0.0, 0.0, 1);
    }
  }
}

static int dist2pixel(map_context_t *context, float km, float lat) {
  return 1000.0*km/osm_gps_map_get_scale(OSM_GPS_MAP(context->widget));
}

static gboolean map_gps_update(gpointer data) {
  map_context_t *context = (map_context_t*)data;
  static gboolean goto_is_enabled = FALSE;

  /* get reference position ... */
  pos_t *refpos = get_pos(context->appdata);
  gboolean ok = (refpos!= NULL) && !isnan(refpos->lat) && !isnan(refpos->lon);

  /* ... and enable "goto" button if it's valid */
  if(ok != goto_is_enabled) {
    osm_gps_map_osd_enable_gps (OSM_GPS_MAP(context->widget), 
		OSM_GPS_MAP_OSD_CALLBACK(ok?cb_map_gps:NULL), context);
    goto_is_enabled = ok;
  }

  if(ok) {
    float heading = NAN;
    int radius = 0;

    if(context->appdata->use_gps) {
      heading = gps_get_heading(context->appdata);

      /* get error */
      float eph = gps_get_eph(context->appdata);
      if(!isnan(eph)) 
	radius = dist2pixel(context, eph/1000, refpos->lat);
    }

    /* TODO: in order to save energy: only draw if state actually changed */

    g_object_set(context->widget, "gps-track-highlight-radius", radius, NULL);
    osm_gps_map_draw_gps(OSM_GPS_MAP(context->widget), 
			 refpos->lat, refpos->lon, heading);
  } else
    osm_gps_map_clear_gps(OSM_GPS_MAP(context->widget));
  
  return TRUE;
}

static void map_draw_cache(GtkWidget *map, cache_t *cache) {
  GdkPixbuf *icon = icon_get(ICON_CACHE_TYPE, cache->type);

  /* check if there's also an overwritten coordinate */
  if(cache->notes && cache->notes->override) {
    GdkPixbuf *over = icon_get(ICON_MISC, 1);

    osm_gps_map_add_image(OSM_GPS_MAP(map), 
	  cache->notes->pos.lat, cache->notes->pos.lon, icon);

    osm_gps_map_add_image(OSM_GPS_MAP(map),
	  cache->notes->pos.lat, cache->notes->pos.lon, over);
  } else {
    if(!isnan(cache->pos.lat) && !isnan(cache->pos.lon))
      osm_gps_map_add_image(OSM_GPS_MAP(map), 
			    cache->pos.lat, cache->pos.lon, icon);
  }
}

static void map_draw_gpx(appdata_t *appdata, GtkWidget *map, gpx_t *gpx) {
  if(!gpx->notes_loaded) {
    notes_load_all(appdata, gpx);
    gpx->notes_loaded = TRUE;
  }

  cache_t *cache = gpx->cache;
  while(cache) {
    map_draw_cache(map, cache);
    cache = cache->next;
  }
}

/* draw geocaches and set window title */
static void map_setup(map_context_t *context) {
  char *name = NULL;

  if(!context->appdata->cur_gpx && !context->appdata->cur_cache) {
    if(context->state != MAP_ALL) {
      printf("map_setup(ALL)\n");

#ifdef OSD_NAV
      /* no navigation in this mode */
      osm_gps_map_osd_clear_nav (OSM_GPS_MAP(context->widget));
#endif

      /* clear all existing ccahe images */
      osm_gps_map_clear_images (OSM_GPS_MAP(context->widget));

      /* draw all geocaches */
      gpx_t *gpx = context->appdata->gpx;
      while(gpx) {
	map_draw_gpx(context->appdata, context->widget, gpx);
	gpx = gpx->next;
      }
      name = g_strdup(_("all"));
      context->state = MAP_ALL;
    }
  } else if(!context->appdata->cur_cache) {
    if(context->state != MAP_GPX) {
      printf("map_setup(GPX)\n");

#ifdef OSD_NAV
      /* no navigation in this mode */
      osm_gps_map_osd_clear_nav (OSM_GPS_MAP(context->widget));
#endif

      /* clear all existing ccahe images */
      osm_gps_map_clear_images (OSM_GPS_MAP(context->widget));

      map_draw_gpx(context->appdata, context->widget, 
		   context->appdata->cur_gpx);
      name = g_strdup(context->appdata->cur_gpx->name);
      context->state = MAP_GPX;
    }
  } else {
    cache_t *cache = context->appdata->cur_cache;
    
    printf("map_setp(CACHE)\n");
    
    /* no balloons in this mode */
    context->balloon = NULL;
    osm_gps_map_osd_clear_balloon (OSM_GPS_MAP(context->widget));
    
    /* clear all existing ccahe images */
    osm_gps_map_clear_images (OSM_GPS_MAP(context->widget));
    
    map_draw_cache(context->widget, cache);
    name = g_strdup(cache->name);
    context->state = MAP_CACHE;
    
    /* navigation in this mode! */
    pos_t cpos = gpx_cache_pos(cache);

#ifdef OSD_NAV
    osm_gps_map_osd_draw_nav (OSM_GPS_MAP(context->widget), 
			      context->appdata->imperial,
			      cpos.lat, cpos.lon, cache->name);
#else
#warning OSD_NAV not defined!
#endif
  }

  if(name) {
    char *title = g_strdup_printf(_("Map - %s"), name);
    g_free(name);

    gtk_window_set_title(GTK_WINDOW(context->window), title);
    
    g_free(title);
  } else
    printf("map_setup(keep)\n");
}

static gboolean on_map_configure(GtkWidget *widget,
				 GdkEventConfigure *event,
				 map_context_t *context) {

  /* for some reason there's a configure event with 1/1 */
  /* on diablo. We just ignore this! */

  printf("on_map_configure %d %d\n", 
	 widget->allocation.width, 
	 widget->allocation.height);

  if(!context->map_complete && 
     (widget->allocation.width > 100) &&
     (widget->allocation.height > 100)) {

    /* setup cache state */
    map_setup(context);

    /* set default values if they are invalid */
    if(!context->appdata->map.zoom || 
       isnan(context->appdata->map.pos.lat) ||
       isnan(context->appdata->map.pos.lon)) {
      printf("no valid map position found\n");
      
      pos_t *refpos = get_pos(context->appdata);
      if(refpos && !isnan(refpos->lat) && !isnan(refpos->lon)) {
	printf("use refpos\n");

	/* use gps position if present */
	context->appdata->map.pos = *refpos;
	context->appdata->map.zoom = GPS_DEFAULT_ZOOM;
      } else {
	printf("use zero pos\n");

	/* use world map otherwise */
	context->appdata->map.pos.lat = 0.0;
	context->appdata->map.pos.lon = 0.0;
	context->appdata->map.zoom = 1;
      }
    }
    
    /* jump to initial position */
    printf("osm_gps_map_set_mapcenter(%f,%f,%d)\n", 
	   context->appdata->map.pos.lat,
	   context->appdata->map.pos.lon,
	   context->appdata->map.zoom);
	   
    osm_gps_map_set_mapcenter(OSM_GPS_MAP(context->widget), 
			      context->appdata->map.pos.lat, 
			      context->appdata->map.pos.lon, 
			      context->appdata->map.zoom);
    context->map_complete = TRUE;
  }

  return FALSE;
}

static void
map_cachelist_nearest(cache_t *cache, pos_t *pos, 
		      cache_t **result, float *distance) {

  while(cache) {
    pos_t cpos = gpx_cache_pos(cache);

    float dist = 
      pow(cpos.lat - pos->lat, 2) + 
      pow(cpos.lon - pos->lon, 2);

    if(!(dist > *distance)) {
      *result = cache;
      *distance = dist;
    }

    cache = cache->next;
  }
}

static cache_t *map_closest(map_context_t *context, pos_t *pos) {
  cache_t *result = NULL;
  float distance = NAN;
  
  if(!context->appdata->cur_gpx && !context->appdata->cur_cache) {
    /* search all geocaches */
    gpx_t *gpx = context->appdata->gpx;
    while(gpx) {
      map_cachelist_nearest(gpx->cache, pos, &result, &distance);
      gpx = gpx->next;
    }
  } else if(context->appdata->cur_gpx) {
    map_cachelist_nearest(context->appdata->cur_gpx->cache, 
			  pos, &result, &distance);
  } else
    result = context->appdata->cur_gpx->cache;

  return result;
}

/* translate between osm-gps-map positions and gpxview ones */
pos_t coord2pos(coord_t coo) {
  pos_t pos;
  pos.lat = rad2deg(coo.rlat);
  pos.lon = rad2deg(coo.rlon);
  return pos;
}

#define CLICK_FUZZ (24)

static gboolean
on_map_button_press_event(GtkWidget *widget, 
			    GdkEventButton *event, map_context_t *context) {

  OsmGpsMap *map = OSM_GPS_MAP(context->widget);

  /* check if we actually clicked parts of the OSD */
  if(osm_gps_map_osd_check(map, event->x, event->y) != OSD_NONE) 
    return FALSE;

  /* got a press event without release event? eat it! */
  if(context->press_on != NULL) {
    printf("PRESS: already\n");
    return FALSE;
  }

  pos_t pos = 
    coord2pos(osm_gps_map_get_co_ordinates(map, event->x, event->y));

  cache_t *nearest = map_closest(context, &pos);  
  if(nearest) {
    pos_t cpos = gpx_cache_pos(nearest);

    float dist = gpx_pos_get_distance(pos, cpos, FALSE);
    if(dist2pixel(context, dist, cpos.lat) < CLICK_FUZZ) 
      context->press_on = nearest;
  }
  
  return FALSE;
}

static void
cairo_draw_pixbuf(cairo_t *cr, GdkPixbuf *buf, gint x, gint y) {
  /* convert the pixbuf into something cairo can handle */

  // Create a new ImageSurface
  cairo_surface_t *image_surface = 
    cairo_image_surface_create(CAIRO_FORMAT_ARGB32, 
			       gdk_pixbuf_get_width(buf),
			       gdk_pixbuf_get_height(buf));

  // Create the new Context for the ImageSurface
  cairo_t *context = cairo_create(image_surface);

  // Draw the image on the new Context
  gdk_cairo_set_source_pixbuf(context, buf, 0.0, 0.0);
  cairo_paint(context);

  // now draw this onto the original context 
  cairo_set_source_surface(cr, image_surface, x, y);

  cairo_paint(cr);
}

#ifndef BIG_BALLOONS
#define LINE_SKIP  7
#else
#define LINE_SKIP  12
#endif

static void 
balloon_cb(osm_gps_map_balloon_event_t *event, gpointer data) {
  printf("balloon event: ");

  map_context_t *context = (map_context_t*)data;
  cache_t *cache = context->balloon;

  if(event->type == OSM_GPS_MAP_BALLOON_EVENT_TYPE_DRAW) {
    printf("draw\n");

#if 0
    /* draw pink background to check clipping */
    cairo_rectangle (event->data.draw.cr, 
		     event->data.draw.rect->x-20, event->data.draw.rect->y-20, 
		     event->data.draw.rect->w+40, event->data.draw.rect->h+40);
    cairo_set_source_rgba (event->data.draw.cr, 1, 0, 0, 0.3);
    cairo_fill_preserve (event->data.draw.cr);
    cairo_set_line_width (event->data.draw.cr, 0);
    cairo_stroke (event->data.draw.cr);
#endif
    
    /* leave a little border top and left */
    gint x = event->data.draw.rect->x, y = event->data.draw.rect->y;
    
    /* draw the cache type icon ... */
    GdkPixbuf *icon = icon_get(ICON_CACHE_TYPE, cache->type);
    cairo_draw_pixbuf(event->data.draw.cr, icon, x, y);
    
    if(cache->notes && cache->notes->override) {
      GdkPixbuf *over = icon_get(ICON_MISC, 1);
      cairo_draw_pixbuf(event->data.draw.cr, over, x, y);
    }
    
    /* ... and right of it the waypoint id */
    cairo_text_extents_t extents;
    
    if(cache->id) {
      cairo_select_font_face (event->data.draw.cr, "Sans",
			      CAIRO_FONT_SLANT_NORMAL,
			      CAIRO_FONT_WEIGHT_BOLD);
      
#ifndef BIG_BALLOONS
      cairo_set_font_size (event->data.draw.cr, 20.0);
#else
      cairo_set_font_size (event->data.draw.cr, 36.0);
#endif
      
      cairo_text_extents (event->data.draw.cr, cache->id, &extents);
      
      /* display id right of icon vertically centered */
      x += gdk_pixbuf_get_width(icon) + 5;
      y += (gdk_pixbuf_get_height(icon) + extents.height)/2;
      cairo_move_to (event->data.draw.cr, x, y);
      cairo_set_source_rgba (event->data.draw.cr, 0, 0, 0, 1);
      cairo_show_text (event->data.draw.cr, cache->id);
      cairo_stroke (event->data.draw.cr);
      
      y += (gdk_pixbuf_get_height(icon) - extents.height)/2 + LINE_SKIP;
    } else
      y += gdk_pixbuf_get_height(icon);
    
    /* return to the left border and below icon/text */
    x = event->data.draw.rect->x;
    
    /* everything from here uses the same font */
    cairo_select_font_face (event->data.draw.cr, "Sans", CAIRO_FONT_SLANT_NORMAL,
			    CAIRO_FONT_WEIGHT_NORMAL);
#ifndef BIG_BALLOONS
    cairo_set_font_size (event->data.draw.cr, 14.0);
#else
    cairo_set_font_size (event->data.draw.cr, 22.0);
#endif
    
    if(cache->name) {
      /* draw cache name */
      cairo_text_extents (event->data.draw.cr, cache->name, &extents);
      y += extents.height;
      cairo_move_to (event->data.draw.cr, x, y);
      cairo_set_source_rgba (event->data.draw.cr, 0, 0, 0, 1);
      cairo_show_text (event->data.draw.cr, cache->name);
      cairo_stroke (event->data.draw.cr);
      
      /* return to the left border and below text */
      y += LINE_SKIP;
      x = event->data.draw.rect->x;
    }
    
    if(cache->terrain) {
      /* draw cache rating */
      const char *terrain = "Terrain:"; 
      icon = icon_get(ICON_STARS, (int)(cache->terrain*2-2));
      cairo_text_extents (event->data.draw.cr, _(terrain), &extents);
      y += (gdk_pixbuf_get_height(icon) + extents.height)/2;
      
      /* draw "Terrain:" string */
      cairo_move_to (event->data.draw.cr, x, y);
      cairo_set_source_rgba (event->data.draw.cr, 0, 0, 0, 1);
      cairo_show_text (event->data.draw.cr, _(terrain));
      cairo_stroke (event->data.draw.cr);
      x += extents.width + 2;
      
      /* draw terrain stars */
      cairo_draw_pixbuf(event->data.draw.cr, icon, x, y - 
			(gdk_pixbuf_get_height(icon) + extents.height)/2);
      
      x += gdk_pixbuf_get_width(icon) + LINE_SKIP;
    y -= (gdk_pixbuf_get_height(icon) + extents.height)/2;
    }
    
    if(cache->difficulty) {
      const char *difficulty = "Difficulty:"; 
      cairo_text_extents (event->data.draw.cr, _(difficulty), &extents);
      y += (gdk_pixbuf_get_height(icon) + extents.height)/2;
      
      /* draw "Difficulty:" string */
      cairo_move_to (event->data.draw.cr, x, y);
      cairo_set_source_rgba (event->data.draw.cr, 0, 0, 0, 1);
      cairo_show_text (event->data.draw.cr, _(difficulty));
      cairo_stroke (event->data.draw.cr);
      x += extents.width + 2;
      
      icon = icon_get(ICON_STARS, (int)(cache->difficulty*2-2));
      cairo_draw_pixbuf(event->data.draw.cr, icon, x, y - 
		      (gdk_pixbuf_get_height(icon) + extents.height)/2);
    }
  } else if(event->type == OSM_GPS_MAP_BALLOON_EVENT_TYPE_CLICK) {
    printf("click %s event at %d %d\n", 
	   event->data.click.down?"down":"up",
	   event->data.click.x, event->data.click.y);

    /* make the main screen jump to that cache */
    if(!event->data.click.down) {
      if(context->appdata->cur_cache) {
	printf("ERROR: no current cache should be visible!\n");
      } else {
	gpx_t *is_in = NULL;

	if(!context->appdata->cur_gpx) {
	  printf("click while in \"all\" view\n");

	  /* we first need to figure out which gpx file this cache */
	  /* is in so we can open it first */
	  gpx_t *gpx = context->appdata->gpx;
	  while(gpx && !is_in) {
	    cache_t *cur = gpx->cache;
	    while(cur && !is_in) {
	      if(cur == cache)
		is_in = gpx;
	      cur = cur->next;
	    }
	    gpx = gpx->next;
	  }
	  
	  if(is_in)
	    gpxlist_goto_cachelist(context->appdata, is_in);

	} else 
	  /* the simple case: there already is an open gpx file and */
	  /* we just jump into the "cache" view */
	  is_in = context->appdata->cur_gpx;

	if(is_in) {
	  printf("selecting %s in %s\n", 
		 cache->id,
		 context->appdata->cur_gpx->name);

	  cachelist_goto_cache(context->appdata, cache);

	  /* give focus to main screen (important for maemo) */
	  printf("raising main window\n");
	  gtk_window_present(GTK_WINDOW(context->appdata->window));
	}
      }
    }
  } else if(event->type == OSM_GPS_MAP_BALLOON_EVENT_TYPE_REMOVED) {
    printf("removed\n");
    context->balloon = NULL;
  }
}

static gboolean
on_map_button_release_event(GtkWidget *widget, 
			    GdkEventButton *event, map_context_t *context) {
  OsmGpsMap *map = OSM_GPS_MAP(context->widget);

  /* in "MAP_CACHE" state only one cache is visible */
  /* and the map is in navigation mode. the balloon is */
  /* pretty useless there */
  if(context->press_on && (context->state != MAP_CACHE)) {

    coord_t coo;
    coo = osm_gps_map_get_co_ordinates(map, event->x, event->y);

    pos_t pos = 
      coord2pos(osm_gps_map_get_co_ordinates(map, event->x, event->y));

    cache_t *nearest = map_closest(context, &pos);  
    if(nearest && nearest == context->press_on) {
      pos_t cpos = gpx_cache_pos(nearest);

      float dist = gpx_pos_get_distance(pos, cpos, FALSE);
      if(dist2pixel(context, dist, cpos.lat) < CLICK_FUZZ) {
	
	context->balloon = nearest;
	osm_gps_map_osd_draw_balloon(map, cpos.lat, cpos.lon,
				     balloon_cb, context);
      }
    }
    context->press_on = NULL;
  } else {
    /* save new map position */
    gfloat lat, lon;
    g_object_get(map, "latitude", &lat, "longitude", &lon, NULL);
    context->appdata->map.pos.lat = lat;
    context->appdata->map.pos.lon = lon;
  }

  return FALSE;
}

static void on_window_destroy(GtkWidget *widget, map_context_t *context) {
  appdata_t *appdata = context->appdata;
  
  /* save map parameters */
  OsmGpsMap *map = OSM_GPS_MAP(context->widget);
  gint zoom;
  g_object_get(map, "zoom", &zoom, NULL);
  context->appdata->map.zoom = zoom;

  gfloat lat, lon;
  g_object_get(map, "latitude", &lat, "longitude", &lon, NULL);
  context->appdata->map.pos.lat = lat;
  context->appdata->map.pos.lon = lon;

  gint source;
  g_object_get(map, "map-source", &source, NULL);
  context->appdata->map.source = source;

#if MAEMO_VERSION_MAJOR == 5
  /* restore cur_view */
  context->appdata->cur_view = context->old_view;
#endif

  gtk_timeout_remove(context->handler_id);
  gps_unregister_callback(appdata, context->cb_id);

  g_free(context);
  appdata->map.context = NULL;
}

#if (MAEMO_VERSION_MAJOR == 5) && !defined(__i386__)
/* get access to zoom buttons */
static void
on_window_realize(GtkWidget *widget, gpointer data) {
  if (widget->window) {
    unsigned char value = 1;
    Atom hildon_zoom_key_atom = 
      gdk_x11_get_xatom_by_name("_HILDON_ZOOM_KEY_ATOM"),
      integer_atom = gdk_x11_get_xatom_by_name("INTEGER");
    Display *dpy = 
      GDK_DISPLAY_XDISPLAY(gdk_drawable_get_display(widget->window));
    Window w = GDK_WINDOW_XID(widget->window);

    XChangeProperty(dpy, w, hildon_zoom_key_atom, 
		    integer_atom, 8, PropModeReplace, &value, 1);
  }
}
#endif

/* on maemo a window is either on top or completely invisible. this */
/* means that we only need to update the map window if its raised.  */
/* on ordinary desktops this is different and we always update */

static gboolean on_focus_in(GtkWidget *widget, GdkEventFocus *event,
			 gpointer data) {
  printf("map got focus\n");
  map_setup((map_context_t*)data);
  return FALSE;
}  

void map_update(appdata_t *appdata) {
  printf("map_update\n");
#ifndef USE_MAEMO
  if(appdata->map.context) 
    map_setup(appdata->map.context);
#endif
}

/* callback called by the gps layer whenever gps state changes */
static void 
gps_callback(struct gps_state *state, gpointer data) {
  printf("map: gps callback\n");
}

void map(appdata_t *appdata) {
  map_context_t *context = NULL; 

  /* if the map window already exists, just raise it */
  if(appdata->map.context) {
    printf("using existing map!\n");
    gtk_window_present(GTK_WINDOW(appdata->map.context->window));
    map_setup(appdata->map.context);
    return;
  }

  context = appdata->map.context = g_new0(map_context_t, 1);
  context->appdata = appdata;
  context->map_complete = FALSE;
  context->state = MAP_NONE;

  /* cleanup old (pre 0.8.7) path if it exists */
  char *old_path = g_strdup_printf("%s/map/", appdata->image_path);
  if(g_file_test(old_path, G_FILE_TEST_IS_DIR)) {
    printf("old file path %s exists\n", old_path);
    rmdir_recursive(old_path);
  }

  /* It is recommanded that all applications share these same */
  /* map path, so data is only cached once. The path should be: */
  /* ~/.osm-gps-map on standard PC     (users home) */
  /* /home/user/.osm-gps-map on Maemo5 (ext3 on internal card) */
  /* /media/mmc2/osm-gps-map on Maemo4 (vfat on internal card) */
#if !defined(USE_MAEMO)
  char *p = getenv("HOME");
  if(!p) p = "/tmp"; 
  char *path = g_strdup_printf("%s/.osm-gps-map", p);
#else
#if MAEMO_VERSION_MAJOR == 5
  char *path = g_strdup("/home/user/.osm-gps-map");
#else
  char *path = g_strdup("/media/mmc2/osm-gps-map");
#endif
#endif

  const char *proxy = get_proxy_uri(appdata);

  gint source = context->appdata->map.source;
  if(!source) source = MAP_SOURCE;

  context->widget = g_object_new(OSM_TYPE_GPS_MAP,
		 "map-source",               source,
                 "tile-cache",               path,
		 "auto-center",              FALSE,
		 "record-trip-history",      FALSE, 
		 "show-trip-history",        FALSE, 
		 proxy?"proxy-uri":NULL,     proxy,
                 NULL);

  g_free(path);

  osm_gps_map_osd_classic_init(OSM_GPS_MAP(context->widget));

#ifdef USE_MAEMO
  /* we don't use a stackable window here on fremantle, since */
  /* this leaves the main window independent from the map and */
  /* the user can e.g. still navigate the main menu */
  context->window = hildon_window_new();

#if (MAEMO_VERSION_MAJOR == 5) && !defined(__i386__)
  g_signal_connect(G_OBJECT(context->window), "realize", 
		   G_CALLBACK(on_window_realize), NULL);
#endif // MAEMO_VERSION
#else
  context->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
#endif

#ifndef USE_MAEMO
  gtk_window_set_default_size(GTK_WINDOW(context->window), 640, 480);
#endif

  g_signal_connect(G_OBJECT(context->widget), "focus-in-event", 
		   G_CALLBACK(on_focus_in), context);

  g_signal_connect(G_OBJECT(context->widget), "configure-event",
		   G_CALLBACK(on_map_configure), context);

  g_signal_connect(G_OBJECT(context->widget), "button-press-event",
                   G_CALLBACK(on_map_button_press_event), context);

  g_signal_connect(G_OBJECT(context->widget), "button-release-event",
                   G_CALLBACK(on_map_button_release_event), context);

  /* install handler for timed updates of the gps button */
  context->handler_id = gtk_timeout_add(1000, map_gps_update, context);
  context->cb_id = gps_register_callback(appdata, gps_callback, context);

#if MAEMO_VERSION_MAJOR == 5
  /* prevent some of the main screen things */
  context->old_view = appdata->cur_view;
  appdata->cur_view = NULL;
#endif

  g_signal_connect(G_OBJECT(context->window), "destroy", 
		   G_CALLBACK(on_window_destroy), context);

  gtk_container_add(GTK_CONTAINER(context->window), context->widget); 
  gtk_widget_show_all(GTK_WIDGET(context->window));
}
