/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"
#include <math.h>

static GtkWidget *cache_description(appdata_t *appdata, cache_t *cache) {
  return html_view(appdata, cache->long_description, 
		   cache->long_is_html, TRUE, cache, NULL);
}

#ifndef USE_MAEMO  // maemos touchscreen doesn't support tooltips
static const char *cache_type_tip[] = {
  "Traditional Cache", "Multicache",    "Mystery/Unknown Cache", 
  "Virtual Cache",     "Webcam Cache",  "Event Cache", 
  "Letterbox Hybrid",  "Earthcache",    "Wherigo Cache", 
  "Mega-Event Cache",  "Cache-In-Trash-Out Cache"
};

static const char *cache_size_tip[] = {
  "Regular Container", "Small Container", "Micro", 
  "Other Container", "Container not chosen", "Large Container", 
  "Virtual Container" 
};
#endif

static const char *cache_size_name[] = {
  "Regular", "Small", "Micro", "Other", 
  "Not chosen", "Large", "Virtual" 
};

void bearing_fill_hbox(GtkWidget *hbox, 
		       appdata_t *appdata, pos_t refpos, pos_t pos) {
  char str[32];

  if(!isnan(pos.lat) && !isnan(pos.lon)) {
    gtk_box_pack_start(GTK_BOX(hbox), gtk_image_new_from_pixbuf(
			       		icon_bearing(refpos, pos)),1,0,0);

    if(refpos.lat && refpos.lon) {
      gtk_box_pack_start_defaults(GTK_BOX(hbox), 
		   GTK_LABEL_SMALL((char*)pos_get_bearing_str(refpos, pos)));
      snprintf(str, sizeof(str), _("%.1f°"), 
	       gpx_pos_get_bearing(refpos, pos));
      gtk_box_pack_start_defaults(GTK_BOX(hbox), GTK_LABEL_SMALL(str));
      gpx_pos_get_distance_str(str, sizeof(str), 
			     refpos, pos, appdata->imperial);
      gtk_box_pack_start(GTK_BOX(hbox), 
	     gtk_label_attrib(str, SIZE_SMALL, STRIKETHROUGH_NONE),1,0,0);
    } else
      gtk_box_pack_start(GTK_BOX(hbox), 
			 gtk_label_attrib(_("(no position)"), 
			 SIZE_SMALL, STRIKETHROUGH_NONE),1,0,0);
  } else
    gtk_box_pack_start(GTK_BOX(hbox), 
	   gtk_label_attrib(_("(invalid position in notes)"), 
			    SIZE_SMALL, STRIKETHROUGH_NONE),1,0,0);
}

/* this function sets everthing related to the coordinate. used to */
/* cope with the user setting a new "note coordinate" */
void overview_coordinate_update(cache_context_t *context) {
  if(!context->notes.modified)
    return;

  /* update position labels */
  int strike = notes_get_override(context)?STRIKETHROUGH:STRIKETHROUGH_NONE;
  char str[32];
  pos_lat_str(str, sizeof(str), context->cache->pos.lat);
  gtk_label_attrib_set(context->pos_lat_label, str, SIZE_BIG, strike);
  pos_lon_str(str, sizeof(str), context->cache->pos.lon);
  gtk_label_attrib_set(context->pos_lon_label, str, SIZE_BIG, strike);

  /* remove enire hbox and build a new one */
  gtk_container_foreach(GTK_CONTAINER(context->bearing_hbox), 
			(GtkCallback)gtk_widget_destroy, NULL);

  /* update distance/etc */
  if(!isnan(context->cache->pos.lat) && 
     !isnan(context->cache->pos.lon)) 
    bearing_fill_hbox(context->bearing_hbox, context->appdata, 
		      *get_pos(context->appdata), notes_get_pos(context)); 

  gtk_widget_show_all(context->bearing_hbox);
}

#ifdef ENABLE_BROWSER_INTERFACE
static void on_www_clicked(GtkButton *button, gpointer data) {
  cache_context_t *context = (cache_context_t*)data;
  browser_url(context->appdata, context->cache->url);
}
#endif

static GtkWidget *cache_overview(cache_context_t *context) {
  GtkWidget *vbox, *ivbox;
  GtkWidget *table, *tip;
  char str[64];
#ifndef USE_MAEMO
  GtkTooltips *tips = gtk_tooltips_new();
#endif
  appdata_t *appdata = context->appdata;
  cache_t *cache = context->cache;

  vbox = gtk_vbox_new(FALSE, 0);

  table =  gtk_table_new(3,4, FALSE);

  if(cache->type != CACHE_TYPE_UNKNOWN) {
    gtk_table_attach_defaults(GTK_TABLE(table), 
      tip = icon_get_widget(ICON_CACHE_TYPE, cache->type), 0,1,0,1);
#ifndef USE_MAEMO
    gtk_tooltips_set_tip(tips, tip, _(cache_type_tip[cache->type]), NULL);
#endif
  }

  /* ------------ box containing container info ------------ */
  if(cache->container != CACHE_CONT_UNKNOWN) {
    ivbox = gtk_vbox_new(FALSE, 0);
    sprintf(str, _("Size: %s"), _(cache_size_name[cache->container]));
    gtk_box_pack_start_defaults(GTK_BOX(ivbox), GTK_LABEL_SMALL(str));
    gtk_box_pack_start_defaults(GTK_BOX(ivbox), 
      	icon_get_widget(ICON_CACHE_SIZE, cache->container));
    gtk_table_attach_defaults(GTK_TABLE(table), ivbox, 0,1,1,2);
#ifndef USE_MAEMO
    gtk_tooltips_set_tip(tips, ivbox, _(cache_size_tip[cache->container]), NULL);
#endif
  }

  /* ----------------------- id ---------------------------- */
  if(cache->id) {
    int strike = cache->archived?STRIKETHROUGH_RED:
      (!cache->available?STRIKETHROUGH:STRIKETHROUGH_NONE);
    GtkWidget *lbl = NULL;
    
#ifdef ENABLE_BROWSER_INTERFACE
    if(!cache->url) 
#endif
      lbl = gtk_label_attrib(cache->id, SIZE_BIG, strike);
#ifdef ENABLE_BROWSER_INTERFACE
    else {
      /* add Go button */
      lbl = gtk_button_attrib(cache->id, SIZE_BIG, strike);
      //      gtk_button_set_relief(GTK_BUTTON(button),GTK_RELIEF_NONE);
      gtk_signal_connect(GTK_OBJECT(lbl), "clicked",
			 (GtkSignalFunc)on_www_clicked, context);
    }
#endif

    gtk_table_attach(GTK_TABLE(table), lbl, 1,2,0,1, FALSE, FALSE, 0, 0);
  }

  /* --------------- box containing owner info ------------- */
  if(cache->owner) {
    ivbox = gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start_defaults(GTK_BOX(ivbox), GTK_LABEL_SMALL(_("by")));
    gtk_box_pack_start_defaults(GTK_BOX(ivbox), GTK_LABEL_SMALL(cache->owner));
    gtk_table_attach_defaults(GTK_TABLE(table), ivbox, 1,2,1,2);
  }

  /* ----------- box containing difficulty rating ---------- */
  if(cache->difficulty != 0) {
    ivbox = gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start_defaults(GTK_BOX(ivbox), 
				GTK_LABEL_SMALL(_("Difficulty:")));
    gtk_box_pack_start_defaults(GTK_BOX(ivbox), 
       icon_get_widget(ICON_STARS, (int)(cache->difficulty*2-2)));
    gtk_table_attach_defaults(GTK_TABLE(table), ivbox, 2,3,0,1);
#ifndef USE_MAEMO
    sprintf(str, _("Difficulty: %.1f"), cache->difficulty);
    gtk_tooltips_set_tip(tips, ivbox, str, NULL);
#endif
  }
    
  /* ------------ box containing terrain rating ------------ */
  if(cache->terrain != 0) {
    ivbox = gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start_defaults(GTK_BOX(ivbox), GTK_LABEL_SMALL(_("Terrain:")));
    gtk_box_pack_start_defaults(GTK_BOX(ivbox), 
      icon_get_widget(ICON_STARS, (int)(cache->terrain*2-2)));
    gtk_table_attach_defaults(GTK_TABLE(table), ivbox, 2,3,1,2);
#ifndef USE_MAEMO
    sprintf(str, _("Terrain: %.1f"), cache->terrain);
    gtk_tooltips_set_tip(tips, ivbox, str, NULL);
#endif
  }

  /* ----------------- the two coordinates ----------------- */
  /* ----------------- and the heading/distance ------------ */
  pos_t *refpos = get_pos(appdata);

  ivbox = gtk_vbox_new(FALSE, 0);
  int strike = (cache->notes && cache->notes->override)?
                  STRIKETHROUGH:STRIKETHROUGH_NONE;

  /* the original coordinate is being displayed */
  gtk_box_pack_start_defaults(GTK_BOX(ivbox), 
       context->pos_lat_label = pos_lat(cache->pos.lat, SIZE_BIG, strike));
  gtk_box_pack_start_defaults(GTK_BOX(ivbox), 
       context->pos_lon_label = pos_lon(cache->pos.lon, SIZE_BIG, strike));

  /* but calculations may be done with respect to the overriden one */
  if(!isnan(cache->pos.lat) && !isnan(cache->pos.lon)) {
    context->bearing_hbox = gtk_hbox_new(FALSE, 0);
    bearing_fill_hbox(context->bearing_hbox, appdata, *refpos, 
		      gpx_cache_pos(cache));
    gtk_box_pack_start_defaults(GTK_BOX(ivbox), context->bearing_hbox);
  }

  gtk_table_attach_defaults(GTK_TABLE(table), ivbox, 3,4,0,2);
 
  /* ----------------------------------------------------- */

  gtk_box_pack_start(GTK_BOX(vbox), table, 0,0,0);
  gtk_box_pack_start(GTK_BOX(vbox), gtk_hseparator_new(),FALSE,FALSE,0);

  if(cache->short_description)
    gtk_box_pack_start_defaults(GTK_BOX(vbox), 
	html_view(appdata, cache->short_description, 
		  cache->short_is_html, TRUE, cache, NULL));

  return vbox;
}

/* slow but short, we don't need performance here ... */
static void rot13(char *t) {
  while(*t) {
    if(((*t >= 'a') && (*t < 'n')) ||
       ((*t >= 'A') && (*t < 'N'))) *t += 13;
    else if(((*t >= 'n') && (*t <= 'z')) ||
	    ((*t >= 'N') && (*t <= 'Z'))) *t -= 13;

    t++;
  }
}

static void on_decrypt(GtkWidget *widget, gpointer data) {
  /* data is a link to the textview */
  g_assert(GTK_IS_TEXT_VIEW(data));

  GtkTextIter start, end;
  GtkTextBuffer *buffer = gtk_text_view_get_buffer((GtkTextView*)data);

  gtk_text_buffer_get_start_iter(buffer, &start);
  gtk_text_buffer_get_end_iter(buffer, &end);
  char *text = gtk_text_buffer_get_text(buffer, &start, &end, FALSE);

  rot13(text);
  gtk_text_buffer_set_text(buffer, text, -1);

  free(text);
}

static GtkWidget *cache_hint(appdata_t *appdata, cache_t *cache) {
  /* encrypting/decrypting html is nothing we want to do */
  if(cache->hint_is_html) 
    return html_view(appdata, cache->hint, TRUE, TRUE, NULL, NULL);

  /* we can now be sure that we are talking about pain text */
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);

  char *hint = strdup(cache->hint);
  rot13(hint);
  GtkWidget *view = html_view(appdata, hint, FALSE, TRUE, NULL, NULL);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), view);
  free(hint);

  GtkWidget *button = gtk_button_new_with_label(_("Encrypt/Decrypt"));
  
  gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
	   GTK_SIGNAL_FUNC(on_decrypt), gtk_bin_get_child(GTK_BIN(view)));

  return vbox;
}

static GtkWidget *cache_wpts(appdata_t *appdata, wpt_t *wpt) {
  pos_t *refpos = NULL;

#ifndef USE_PANNABLE_AREA
  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
  				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
#endif

  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);

  /* four rows per waypoint */
  GtkWidget *table =  gtk_table_new(4*gpx_number_of_waypoints(wpt)-1,4, FALSE);

  refpos = get_pos(appdata);

  int wpt_row=0;
  while(wpt) {
    GtkWidget *ihbox, *tip;
    char str[32];

    /* ----------------------- icon/id ---------------------------- */
    ihbox = gtk_hbox_new(FALSE, 0);

    if(wpt->sym != WPT_SYM_UNKNOWN) {
      gtk_box_pack_start(GTK_BOX(ihbox),
	  tip = icon_get_widget(ICON_WPT, wpt->sym), 1,0,0);
    }

    if(wpt->id)
      gtk_box_pack_start(GTK_BOX(ihbox), GTK_LABEL_BIG(wpt->id), 1,0,0);

    gtk_table_attach_defaults(GTK_TABLE(table), ihbox, 0,1,wpt_row, wpt_row+1);

    /* ----------------- the two coordinates ----------------- */
    /* ----------------- and the heading/distance ------------ */
    gtk_table_attach_defaults(GTK_TABLE(table), 
		      pos_lat(wpt->pos.lat, SIZE_BIG, STRIKETHROUGH_NONE), 
		      1,2, wpt_row, wpt_row+1);
    gtk_table_attach_defaults(GTK_TABLE(table), 
		      pos_lon(wpt->pos.lon, SIZE_BIG, STRIKETHROUGH_NONE), 
		      2,3, wpt_row, wpt_row+1);

    ihbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(ihbox), gtk_image_new_from_pixbuf(
			   icon_bearing(*refpos, wpt->pos)),1,0,0);
    gtk_box_pack_start_defaults(GTK_BOX(ihbox), 
          GTK_LABEL_SMALL((char*)pos_get_bearing_str(*refpos, wpt->pos)));
    snprintf(str, sizeof(str), _("%.1f°"), 
	   gpx_pos_get_bearing(*refpos, wpt->pos));
    gtk_box_pack_start_defaults(GTK_BOX(ihbox), GTK_LABEL_SMALL(str));
    gpx_pos_get_distance_str(str, sizeof(str), 
			     *refpos, wpt->pos, appdata->imperial);
    gtk_box_pack_start(GTK_BOX(ihbox), GTK_LABEL_SMALL(str),1,0,0);

    gtk_table_attach_defaults(GTK_TABLE(table), ihbox, 3,4, 
    			      wpt_row+0, wpt_row+1);
    
    /* ------------------ description ------------------------- */
    if(wpt->desc) {
      GtkTextBuffer *buffer = gtk_text_buffer_new(NULL);
      gtk_text_buffer_set_text(buffer, wpt->desc, strlen(wpt->desc));

#ifndef USE_HILDON_TEXT_VIEW
      GtkWidget *textview = gtk_text_view_new_with_buffer(buffer);
#else
      GtkWidget *textview = hildon_text_view_new();
      hildon_text_view_set_buffer(HILDON_TEXT_VIEW(textview), buffer);
#endif

      gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(textview), GTK_WRAP_WORD);
      gtk_text_view_set_editable(GTK_TEXT_VIEW(textview), FALSE);
      gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(textview), FALSE);

      gtk_table_attach_defaults(GTK_TABLE(table), textview, 0,4, 
				wpt_row+1, wpt_row+2);
    }

    /* ------------------ comment ------------------------- */
    if(wpt->cmt) {
      GtkTextBuffer *buffer = gtk_text_buffer_new(NULL);
      gtk_text_buffer_set_text(buffer, wpt->cmt, strlen(wpt->cmt));
#ifndef USE_HILDON_TEXT_VIEW
      GtkWidget *textview = gtk_text_view_new_with_buffer(buffer);
#else
      GtkWidget *textview = hildon_text_view_new();
      hildon_text_view_set_buffer(HILDON_TEXT_VIEW(textview), buffer);
#endif
      gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(textview), GTK_WRAP_WORD);
      gtk_text_view_set_editable(GTK_TEXT_VIEW(textview), FALSE);
      gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(textview), FALSE);

      gtk_table_attach_defaults(GTK_TABLE(table), textview, 0,4, 
				wpt_row+2, wpt_row+3);
    }

    /* --------------------- seperator -------------------------*/
    if(wpt->next)
      gtk_table_attach_defaults(GTK_TABLE(table), gtk_hseparator_new(), 0,4, 
				wpt_row+3, wpt_row+4);


    wpt_row+=4;
    wpt = wpt->next;
  }

  gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

#ifndef  USE_PANNABLE_AREA
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), 
					vbox);
  return scrolled_window;
#else
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(pannable_area),
					 vbox);
  return pannable_area;
#endif
}

static GtkWidget *cache_tbs(appdata_t *appdata, tb_t *tb) {
  pos_t *refpos = NULL;

#ifndef USE_PANNABLE_AREA
  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
  				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
#endif

  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);

  /* four rows per waypoint */
  GtkWidget *table =  gtk_table_new(2*gpx_number_of_tbs(tb)-1,3, FALSE);

  refpos = get_pos(appdata);

  int tb_row=0;
  while(tb) {
    /* --------------------- icon/ref/name -------------------------*/
    gtk_table_attach_defaults(GTK_TABLE(table), icon_get_widget(ICON_TB, 0), 
			      0, 1, tb_row+0, tb_row+1);
    if(tb->ref)
      gtk_table_attach_defaults(GTK_TABLE(table), GTK_LABEL_BIG(tb->ref),
				1, 2, tb_row+0, tb_row+1);
    if(tb->name)
      gtk_table_attach_defaults(GTK_TABLE(table), GTK_LABEL_BIG(tb->name),
				2, 3, tb_row+0, tb_row+1);

    /* --------------------- seperator -------------------------*/
    if(tb->next)
      gtk_table_attach_defaults(GTK_TABLE(table), gtk_hseparator_new(), 0, 3, 
				tb_row+1, tb_row+2);
    tb_row+=2;
    tb = tb->next;
  }

  gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

#ifndef  USE_PANNABLE_AREA
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), 
					vbox);
  return scrolled_window;
#else
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(pannable_area),
					 vbox);
  return pannable_area;
#endif
}

static GtkWidget *cache_logs(appdata_t *appdata, log_t *log, int is_html) {
#ifndef  USE_PANNABLE_AREA
  /* put this inside a scrolled view */
  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
  				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
#else
  GtkWidget *pannable_area = hildon_pannable_area_new();
#endif

  GtkWidget *table = gtk_table_new(4*gpx_number_of_logs(log), 3, FALSE);
  int cnt = 0;
  
  /* add all logs to the vbox */
  while(log) {
    gtk_table_attach_defaults(GTK_TABLE(table), gtk_hseparator_new(), 
			      0, 3, cnt+0, cnt+1);
    gtk_table_attach_defaults(GTK_TABLE(table), 
			      icon_get_widget(ICON_LOG, log->type), 
			      0, 1, cnt+1, cnt+2);

    char date_str[32];
    if(log->day && log->month && log->year) {
      GDate *date = g_date_new_dmy(log->day, log->month, log->year);
      g_date_strftime(date_str, sizeof(date_str), "%x", date);
      g_date_free(date);
    } else
      strcpy(date_str, "---");

    gtk_table_attach_defaults(GTK_TABLE(table), gtk_label_new(date_str),
			      1, 2, cnt+1, cnt+2);

    gtk_table_attach_defaults(GTK_TABLE(table), gtk_label_new(log->finder),
			      2, 3, cnt+1, cnt+2);
    gtk_table_attach_defaults(GTK_TABLE(table), gtk_hseparator_new(), 
			      0, 3, cnt+2, cnt+3);

    if(log->text) {
      gtk_table_attach_defaults(GTK_TABLE(table), 
		html_view(appdata, log->text, is_html, FALSE, NULL, NULL),
		0, 3, cnt+3, cnt+4);
    }

    log = log->next;
    cnt+=4;
  }

#ifndef  USE_PANNABLE_AREA
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), 
					table);
  return scrolled_window;
#else
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(pannable_area),
					 table);
  return pannable_area;
#endif
}

#ifdef USE_MAEMO
/* this routine is called once a second as long as the "goto" tab is visible */
static gboolean screensaver_update(gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  if(appdata->goto_disable_screensaver)
    if (osso_display_blanking_pause(appdata->osso_context) != OSSO_OK) 
      fprintf(stderr, "error with display blank\n");

  return TRUE;  // fire again
}
#endif

static void on_notebook_page_change(GtkNotebook     *notebook,
				    GtkNotebookPage *page,
				    guint            page_num,
				    gpointer         user_data) {

  cache_context_t *context = (cache_context_t*)user_data;
  GtkWidget *w = gtk_notebook_get_nth_page(notebook, page_num);
  const char *name = gtk_notebook_get_tab_label_text(notebook, w);

#ifdef USE_MAEMO  
  if(context->handler_id)
    gtk_timeout_remove(context->handler_id);
#endif

  /* this is a workaround, around some bug in the gtktextwidget or so ... */
  /* i tried to get info on this and everybody agreed that this is a bug */
  /* in gtk but noone had a read fix. so i came up with this. */
  /* seems to work ... */
  if(strcasecmp(name, _("Logs")) == 0) {
    gtk_widget_queue_resize(w);
  }

  if(strcasecmp(name, _("Goto")) == 0) {
#ifdef USE_MAEMO  
    context->handler_id = gtk_timeout_add(1000, screensaver_update, 
					  context->appdata);
#endif
    goto_coordinate_update(context);
  }

  if(strcasecmp(name, _("Main")) == 0) {
    /* the notes page may have changed its "override" setting, thus the */
    /* striked out coordinate may need update */
    overview_coordinate_update(context);
  }
}

static void on_notebook_destroy(GtkWidget *widget, gpointer user_data ) {
  cache_context_t *context = (cache_context_t*)user_data;

  printf("destroying notebook\n");

  notes_destroy_event(NULL, context);
  goto_destroy_event(NULL, context);

#ifdef USE_MAEMO
  if(context->handler_id)
    gtk_timeout_remove(context->handler_id);
#endif

  free(user_data);
}

GtkWidget *cache_view(appdata_t *appdata, cache_t *cache) {
  GtkWidget *notebook;

  cache_context_t *cache_context = malloc(sizeof(cache_context_t));
  memset(cache_context, 0, sizeof(cache_context_t));
  cache_context->appdata = appdata;
  cache_context->cache = cache;

#ifdef USE_MAEMO
#define TAB_DESC   _("Desc.")
#define TAB_WPTS   _("Wpts")
#else
#define TAB_DESC   _("Description")
#define TAB_WPTS   _("Waypoints")
#endif

  notebook =  gtk_notebook_new();
#ifdef USE_MAEMO
#if MAEMO_VERSION_MAJOR >= 5
  /* prevents user from accidentially touching the breadcrumb trail */
  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_BOTTOM);
#endif
#endif

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), 
       cache_overview(cache_context), gtk_label_new(_("Main")));

  if(cache->long_description)
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), 
	   cache_description(appdata, cache), gtk_label_new(TAB_DESC));

  if(cache->hint)
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), 
	   cache_hint(appdata, cache), gtk_label_new(_("Hint")));

  if(cache->log)
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), 
           cache_logs(appdata, cache->log, cache->logs_are_html), 
	   gtk_label_new(_("Logs")));

  if(cache->wpt)
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), 
	     cache_wpts(appdata, cache->wpt), gtk_label_new(TAB_WPTS));

  if(cache->tb)
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), 
	     cache_tbs(appdata, cache->tb), gtk_label_new(_("TBs")));

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), 
	   cache_notes(cache_context), gtk_label_new(_("Notes")));

  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), 
	   goto_cache(cache_context), gtk_label_new(_("Goto")));

  g_signal_connect(G_OBJECT(notebook), "switch-page", 
	   G_CALLBACK(on_notebook_page_change), cache_context);

  g_signal_connect(G_OBJECT(notebook), "destroy", 
	   G_CALLBACK(on_notebook_destroy), cache_context);

  return notebook;
}

#ifndef USE_MAEMO
void cache_dialog(appdata_t *appdata, cache_t *cache) {
  GtkWidget *dialog = gtk_dialog_new_with_buttons(cache->name, 
			  GTK_WINDOW(appdata->window),
		          GTK_DIALOG_NO_SEPARATOR | GTK_DIALOG_MODAL | 
				       GTK_DIALOG_DESTROY_WITH_PARENT,
		          GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
                          NULL);

  gtk_window_set_default_size(GTK_WINDOW(dialog), DIALOG_WIDTH, DIALOG_HEIGHT);

  /* create cache visualization widget */
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), 
		    cache_view(appdata, cache));

  gtk_widget_show_all(dialog);
  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
}

#else
#ifdef USE_STACKABLE_WINDOW
static void on_cache_destroy (GtkWidget *widget, appdata_t *appdata) {
  appdata->cur_cache = NULL;

  /* restore cur_view */
  appdata->cur_view = g_object_get_data(G_OBJECT(widget), "cur_view");
}

void cache_dialog(appdata_t *appdata, cache_t *cache) {
  GtkWidget *window = hildon_stackable_window_new();

  /* store last "cur_view" in window */
  g_object_set_data(G_OBJECT(window), "cur_view", appdata->cur_view);

  appdata->cur_cache = cache;
  char *title = g_strdup_printf("GPXView - %s", cache->name);
  gtk_window_set_title(GTK_WINDOW(window), title);
  g_free(title);

  /* create cache visualization widget */
  appdata->cur_view = cache_view(appdata, cache);
  gtk_container_add(GTK_CONTAINER(window), appdata->cur_view);

  hildon_window_set_app_menu(HILDON_WINDOW(window), 
			     menu_create(appdata, MENU_CACHE));

  g_signal_connect(G_OBJECT(window), "destroy", 
		   G_CALLBACK(on_cache_destroy), appdata);

  gtk_widget_show_all(window);
}
#endif // USE_STACKABLE_WINDOW

#endif // USE_MAEMO
