/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"

#include <math.h>
#include <sqlite3.h>

static const char *cache_type_icons[] = { 
  "traditional",  "multi",  "mystery",  "virtual",
  "webcam",  "event",  "letterbox",  "earthcache",
  "wherigo",  "megaevent",  "cito",  "" };

static const char *cache_type_str[] = { 
  "Traditional", "Multi", "Mystery","Virtual", "Webcam", "Event",
  "Letterbox", "Earthcache", "Wherigo","Megaevent", "CITO",""};

static const char *cache_type_desc[] = {
  "Traditional Caches", "Multi-Caches", "Unknown Caches",
  "Virtual Caches", "Webcam Caches", "Event Caches",
  "Letterbox Hybrids", "Earthcaches", "Wherigo Caches",
  "Mega-Event Caches", "Cache In Trash Out Events",""};

static const char *mode_str[] = { 
  "???", "the cache listing", "the list of caches", "all GPX files" };

static int callback(void *NotUsed, int argc, char **argv, char **azColName){
  int i;
  NotUsed=0;

  for(i=0; i<argc; i++){
    printf("%s = %s\n", azColName[i], argv[i] ? argv[i] : "NULL");
  }
  printf("\n");
  return 0;
}

char *escape(char *str) {
  if(!str) return NULL;

  char *result = malloc(strlen(str)+1);
  char *p = result;

  while(*str) {
    if(*str != '\'') *p++=*str++;
    else             {  *p++='`'; str++; }
  }
  *p = 0;

  return result;
}

/* export a single cache to the database */
static void cache_export(sqlite3 *db, int id, cache_t *cache) {
  char *zErrMsg = 0;
  char cmd[256];

  //  printf("exporting %s: %s\n", cache->id, cache->name);

  char *desc = escape(cache->id);  // short_description
  if(!desc) desc = strdup(_("<none>"));

  char *name = escape(cache->name);
  if(!name) desc = strdup(_("<none>"));

  pos_t pos = gpx_cache_pos(cache);
  char strlon[32], strlat[32];
  g_ascii_dtostr(strlat, sizeof(strlat), pos.lat);
  g_ascii_dtostr(strlon, sizeof(strlon), pos.lon);

  snprintf(cmd, sizeof(cmd), 
	   "insert into poi values (%d,%s,%s,'%s','%s',%d)",
	   id, strlat, strlon, name, desc, cache->type+12);

  if(sqlite3_exec(db,cmd, callback, 0, &zErrMsg) != SQLITE_OK)
    errorf(_("Creating POI entry:\n\n%s"), zErrMsg);

  if(desc) free(desc);
  if(name) free(name);
}

/* add a cache to the chain of caches to be exported */
static void chain_cache(gpx_t *gpx, cache_t *cache) {
  cache_t **cur = &gpx->cache;

  /* search end of chain */
  while(*cur) {
    if(strcmp(cache->id, (*cur)->id) == 0) 
      return;

    cur = &(*cur)->next;
  }

  *cur = malloc(sizeof(cache_t));
  if(!(*cur)) return;

  memcpy(*cur, cache, sizeof(cache_t));
  (*cur)->next = NULL;
}

static gpx_t *export_list_create(appdata_t *appdata, int *mode) {
  gpx_t *gpx = g_new0(gpx_t,1);
  *mode = 0;

  if(!gpx) return gpx;

#ifdef USE_MAEMO
  if(appdata->cur_cache) {
    printf("cache display, exporting cache only!\n");
    /* appdata->cur_cache */
    chain_cache(gpx, appdata->cur_cache);
    *mode = 1;
  } else if(appdata->search_results) {
    printf("search result display, exporting entire search result\n");
    cache_t *cache = appdata->search_results->cache;
    while(cache) { chain_cache(gpx, cache); cache = cache->next; }
    *mode = 2;
  } else if(appdata->cur_gpx) {
    printf("cachelist display, exporting entire cachelist\n");
    cache_t *cache = appdata->cur_gpx->cache;
    while(cache) { chain_cache(gpx, cache); cache = cache->next; }
    *mode = 2;
  } else
#endif
  {
    printf("gpxlist display, exporting all caches\n");
    gpx_t *lgpx = appdata->gpx;
    while(lgpx) {
      /* make sure notes are imported */
      if(!lgpx->notes_loaded) {
	notes_load_all(appdata, lgpx);
	lgpx->notes_loaded = TRUE;
      }

      cache_t *cache = lgpx->cache;
      while(cache) { chain_cache(gpx, cache); cache = cache->next; }
      lgpx = lgpx->next;
    }
    *mode = 3;
  }

  return gpx;
}

static void export_list_free(gpx_t *gpx) {
  printf("freeing export list\n");

  cache_t *cache = gpx->cache;
  while(cache) {
    cache_t *tmp = cache;
    cache = cache->next;
    free(tmp);
  }
  free(gpx);
}

static int export_list_count(appdata_t *appdata, gpx_t *gpx) {
  pos_t *refpos = get_pos(appdata);
  int cnt = 0;

  cache_t *cache = gpx->cache;
  while(cache) {
    /* no disabled/archived if requested */
    if(!(appdata->mmpoi_dont_export_disabled && 
	 (!cache->available || cache->archived))) {

      /* no found if requested */
      if(!(appdata->mmpoi_dont_export_found && 
	   cache->notes && cache->notes->found)) {
	   
	if(appdata->mmpoi_use_radius && !isnan(appdata->mmpoi_radius)) {
	  /* count all caches within given radius around current position */
	  if(gpx_pos_get_distance(*refpos, gpx_cache_pos(cache), 0) < 
	     appdata->mmpoi_radius)
	    cnt++;
	} else
	  cnt++;
      }
      
    }

    cache = cache->next;
  }
  return cnt;
}

typedef struct {
  GtkWidget *use_radius;
  GtkWidget *no_found;
  GtkWidget *no_disabled;
  GtkWidget *entry;
  GtkWidget *info_label;
  GtkWidget *dialog;
  appdata_t *appdata;
  guint handler_id;
  int mode;
  gpx_t *gpx;
  GtkWidget *path_label;
#ifdef USE_MAEMO
  GtkWidget *launch;
#endif
} export_context_t;

static float distance_eval(GtkWidget *widget, export_context_t *context) {
  char *p = (char*)gtk_entry_get_text(GTK_ENTRY(widget));
  float val = distance_parse(p, context->appdata->imperial);
  return val;
}

static gboolean export_list_update(gpointer data) {
  char str[256];
  export_context_t *context = (export_context_t*)data;

  printf("updating list ...\n");

  int cnum = export_list_count(context->appdata, context->gpx);
  printf("About to export %d caches\n", cnum);

  snprintf(str, sizeof(str), 
	   _("This will export %d caches from %s."),
	   cnum, _(mode_str[context->mode]));

  gtk_label_set_text(GTK_LABEL(context->info_label), str);

  return FALSE;
}

/* Our usual callback function */
static void export_update(GtkWidget *widget, gpointer data) {
  export_context_t *context = (export_context_t *)data;

  if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context->use_radius))) {
    textbox_enable(context->entry);
    context->appdata->mmpoi_use_radius = TRUE;
  } else { 
    textbox_disable(context->entry);
    context->appdata->mmpoi_use_radius = FALSE;
  }

  context->appdata->mmpoi_dont_export_found = 
      gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context->no_found));

  context->appdata->mmpoi_dont_export_disabled = 
      gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context->no_disabled));

  export_list_update(context);
}

static void export_radius_modified(GtkWidget *widget, gpointer data ) {
  export_context_t *context = (export_context_t*)data;

  context->appdata->mmpoi_radius = distance_eval(context->entry, context);

  printf("Distance is %.2f km\n", context->appdata->mmpoi_radius);

  if(context->handler_id) {
    printf("resetting running timer\n");
    gtk_timeout_remove(context->handler_id);
    context->handler_id = 0;
  }

  if(!isnan(context->appdata->mmpoi_radius))
    context->handler_id = 
      gtk_timeout_add(1000, export_list_update, context);  
}

/* Our usual callback function */
static void on_cancel(GtkWidget *widget, gpointer data) {
  int *flag = (int*)data;
  *flag = TRUE;
}

static void on_browse(GtkWidget *widget, gpointer data) {
  GtkWidget *dialog;

  export_context_t *context = (export_context_t*)data;

  printf("Browse\n");

#ifdef USE_MAEMO
  dialog = hildon_file_chooser_dialog_new(GTK_WINDOW(context->dialog), 
					  GTK_FILE_CHOOSER_ACTION_SAVE);
#else
  dialog = gtk_file_chooser_dialog_new(_("Save POI database"),
				       GTK_WINDOW(context->dialog),
				       GTK_FILE_CHOOSER_ACTION_SAVE,
				       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				       GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
				       NULL);
#endif

  printf("set filename <%s>\n", context->appdata->mmpoi_path);

  if(!g_file_test(context->appdata->mmpoi_path, G_FILE_TEST_EXISTS)) {
    char *last_sep = strrchr(context->appdata->mmpoi_path, '/');
    if(last_sep) {
      *last_sep = 0;  // seperate path from file 

      /* the user just created a new document */
      gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), 
					  context->appdata->mmpoi_path);
      gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), last_sep+1);

      /* restore full filename */
      *last_sep = '/';
    }
  } else 
    gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(dialog), 
				  context->appdata->mmpoi_path);

  if (gtk_dialog_run (GTK_DIALOG(dialog)) == GTK_FM_OK) {
    gchar *name = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
    if(name) {
      free(context->appdata->mmpoi_path);
      context->appdata->mmpoi_path = strdup(name);
      gtk_label_set_text(GTK_LABEL(context->path_label), 
			 context->appdata->mmpoi_path);
    }
  }
  
  gtk_widget_destroy (dialog);
}

void mmpoi_export(appdata_t *appdata) {
  sqlite3 *db;
  char *zErrMsg = 0;
  char cmd[256];
  int rc, i;
  char *old_mmpoi_path = strdup(appdata->mmpoi_path);

  export_context_t context;
  memset(&context, 0, sizeof(export_context_t));
  context.appdata = appdata;

  printf("export poi\n");

  /* first create a new faked gpx file containing all caches to */
  /* be exported */
  context.gpx = export_list_create(appdata, &context.mode);
  if(!context.gpx) {
    errorf(_("Out of memory"));
    return;
  }

  /* --------- do confirmation dialog ----------- */
  context.dialog = gtk_dialog_new_with_buttons(_("Maemo Mapper POI export"),
	  GTK_WINDOW(appdata->window), GTK_DIALOG_MODAL,
	  GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
          GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
          NULL);

#if defined(USE_MAEMO) && defined(HILDON_HELP)
  hildon_help_dialog_help_enable(GTK_DIALOG(context.dialog), 
		 HELP_ID_EXPORT, appdata->osso_context);
#endif 

  GtkWidget *vbox = gtk_vbox_new(FALSE,0);

  /* ------------------ radius limit gui ------------------ */
			      
  GtkWidget *label, *hbox = gtk_hbox_new(FALSE,2);
  context.use_radius = gtk_check_button_new_with_label(
	       _("Limit export radius to:"));
  context.entry = dist_entry_new(appdata->mmpoi_radius, appdata->imperial);

  gtk_box_pack_start_defaults(GTK_BOX(hbox), context.use_radius);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context.entry);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), hbox);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context.use_radius), 
			       appdata->mmpoi_use_radius);

  /* Connect the "clicked" signal of the button to our callback */
  gtk_signal_connect (GTK_OBJECT(context.use_radius), "clicked",
		      GTK_SIGNAL_FUNC(export_update), (gpointer)&context);

  g_signal_connect(G_OBJECT(context.entry), "changed",
		   G_CALLBACK(export_radius_modified), (gpointer)&context);

  /* ------------------ don't export found/disabled/archived -------------- */

  hbox = gtk_hbox_new(FALSE,2); 

  label = gtk_label_new(_("Don't export"));
  gtk_misc_set_alignment(GTK_MISC(label), 0.f, 0.5f);
  context.no_found = gtk_check_button_new_with_label(_("found"));
  context.no_disabled = gtk_check_button_new_with_label(_("disabled/archived"));

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context.no_found), 
			       appdata->mmpoi_dont_export_found);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context.no_disabled), 
			       appdata->mmpoi_dont_export_disabled);

  gtk_box_pack_start_defaults(GTK_BOX(hbox), label);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context.no_found);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context.no_disabled);

  gtk_signal_connect (GTK_OBJECT(context.no_found), "clicked",
		      GTK_SIGNAL_FUNC(export_update), (gpointer)&context);
  gtk_signal_connect (GTK_OBJECT(context.no_disabled), "clicked",
		      GTK_SIGNAL_FUNC(export_update), (gpointer)&context);

  gtk_box_pack_start_defaults(GTK_BOX(vbox),hbox);

  /* ------------------ info text -------------- */

  context.info_label = gtk_label_new("");
  gtk_label_set_line_wrap_mode(GTK_LABEL(context.info_label), PANGO_WRAP_WORD);
  gtk_label_set_line_wrap(GTK_LABEL(context.info_label), TRUE);
  gtk_misc_set_alignment(GTK_MISC(context.info_label), 0.f, 0.5f);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), context.info_label);

  export_update(NULL, &context);

#ifdef USE_MAEMO
  gtk_box_pack_start_defaults(GTK_BOX(vbox),
		      context.launch = gtk_check_button_new_with_label(
	              _("Launch Maemo Mapper after export")));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context.launch), 
			       !appdata->mmpoi_dontlaunch);
#endif

  /* ------------------ path/file ------------------ */
  gtk_box_pack_start_defaults(GTK_BOX(vbox), gtk_hseparator_new());

  hbox = gtk_hbox_new(FALSE, 0);
  label = gtk_label_new(_("Export to:"));
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE,0);
  gtk_misc_set_alignment(GTK_MISC(label), 0.f, 0.5f);
  GtkWidget *button = gtk_button_new_with_label(_("Browse"));
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(on_browse), (gpointer)&context);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE,0);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), hbox);

  context.path_label = gtk_label_new(appdata->mmpoi_path);
  gtk_misc_set_alignment(GTK_MISC(context.path_label), 0.f, 0.5f);
  gtk_label_set_ellipsize(GTK_LABEL(context.path_label), PANGO_ELLIPSIZE_MIDDLE);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), context.path_label);

  /* ------------------ info ------------------ */

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(context.dialog)->vbox), vbox);

  gtk_widget_show_all(context.dialog);

  if(GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(context.dialog))) {
#ifdef USE_MAEMO
    appdata->mmpoi_dontlaunch = 
      !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context.launch));
#endif

    /* remove existing database */
    remove(appdata->mmpoi_path);

    if(checkdir(appdata->mmpoi_path) != 0) 
      errorf(_("Unable to access or create output directory!"));
    else {

      /* build a progress dialog which can even be stopped */
      
      GtkWidget *pbar, *wait_dialog = gtk_dialog_new();
      gtk_dialog_set_has_separator(GTK_DIALOG(wait_dialog), FALSE);
      gtk_window_set_default_size(GTK_WINDOW(wait_dialog), 250, 100);
      gtk_window_set_title(GTK_WINDOW(wait_dialog), _("Exporting"));
      gtk_window_set_modal(GTK_WINDOW(wait_dialog), TRUE);
      gtk_window_set_transient_for(GTK_WINDOW(wait_dialog), 
				   GTK_WINDOW(context.dialog));
      gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(wait_dialog)->vbox),
				  pbar = gtk_progress_bar_new());
      
      GtkWidget *button = gtk_button_new_with_label(_("Cancel"));
      int cancelled = 0;
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
			 GTK_SIGNAL_FUNC(on_cancel), (gpointer)&cancelled);
      gtk_container_add(GTK_CONTAINER(GTK_DIALOG(wait_dialog)->action_area),
			button);
      
      gtk_widget_show_all(wait_dialog);
      
      /* ---------------------- copy icons --------------------- */
      for(i=0;cache_type_icons[i][0];i++) {
	char *src = malloc(strlen(ICONPATH)+strlen(cache_type_icons[i])+
			   strlen("cache_type_.gif")+1);
	sprintf(src, ICONPATH "cache_type_%s.gif", cache_type_icons[i]);
	if(!g_file_test(src, G_FILE_TEST_EXISTS)) 
	  sprintf(src, "./icons/cache_type_%s.gif", cache_type_icons[i]);
	
	if(g_file_test(src, G_FILE_TEST_EXISTS)) {
	  char *dest = malloc(strlen(appdata->mmpoi_path)+
			      strlen(cache_type_icons[i])+strlen(".jpg")+1);
	  strcpy(dest, appdata->mmpoi_path);
	  *(strrchr(dest, '/')+1) = 0;
	  sprintf(dest+strlen(dest), "%s.jpg", cache_type_icons[i]);
	  
	  /* read source file */
	  FILE *file = fopen(src, "rb");
	  fseek(file, 0, SEEK_END);
	  int len = ftell(file);
	  fseek(file, 0, SEEK_SET);
	  char *buffer = malloc(len);
	  fread(buffer, 1l, len, file);
	  fclose(file);
	
	  /* write destination file */
	  file = fopen(dest, "wb");
	  fwrite(buffer, 1l, len, file);
	  fclose(file);
	  
	  free(buffer);
	  free(dest);
	}
	
	free(src);
      }
    
      /* ---------------------- database export --------------------- */
      
      printf("exporting to %s\n", appdata->mmpoi_path);
      rc = sqlite3_open(appdata->mmpoi_path, &db);
      if( rc ){
	errorf(_("Can't open SQL database:\n\n%s"), sqlite3_errmsg(db));
	sqlite3_close(db);
	export_list_free(context.gpx);
	return;
      }
      
      /* create the table */
      rc = sqlite3_exec(db, "create table category (cat_id integer PRIMARY "
			"KEY, label text, desc text, enabled integer)", 
			callback, 0, &zErrMsg);
      if( rc!=SQLITE_OK ) {
	errorf(_("Creating cathegory table:\n\n%s"), zErrMsg);
	sqlite3_close(db);
	export_list_free(context.gpx);
	return;
      }
      
      /* create all types */
      i = 0;
      while(cache_type_str[i][0]) {
	snprintf(cmd, sizeof(cmd), 
		 "insert into category values (%d, '%s', '%s', 1)", 
		 12+i, cache_type_str[i], cache_type_desc[i]);
	
	rc = sqlite3_exec(db, cmd, callback, 0, &zErrMsg);
	if( rc != SQLITE_OK ){
	  errorf(_("Creating cathegory:\n\n%s"), zErrMsg);
	  sqlite3_close(db);
	  export_list_free(context.gpx);
	  return;
	}
	i++;
      }
      
      rc = sqlite3_exec(db, "create table poi (poi_id integer PRIMARY KEY, "
			"lat real, lon real, label text, desc text, "
			"cat_id integer)", callback, 0, &zErrMsg);
      if( rc!=SQLITE_OK ){
	errorf(_("Creating POI table:\n\n%s"), zErrMsg);
	sqlite3_close(db);
	export_list_free(context.gpx);
	return;
      }

      pos_t *refpos = get_pos(appdata);
      
      /* export the caches */
      int id = 1;
      cache_t *cache = context.gpx->cache;
      int total = export_list_count(appdata, context.gpx);
      
      while(cache && !cancelled) {
	
	/* no disabled/archived if requested */
	if(!(appdata->mmpoi_dont_export_disabled && 
	     (!cache->available || cache->archived))) {
	  
	  /* no found if requested */
	  if(!(appdata->mmpoi_dont_export_found && 
	       cache->notes && cache->notes->found)) {

	    if(appdata->mmpoi_use_radius && !isnan(appdata->mmpoi_radius)) {
	      /* count all caches within given radius around current position */
	      if(gpx_pos_get_distance(*refpos, gpx_cache_pos(cache), 0) < 
		 appdata->mmpoi_radius) {
		printf("%d exporting %s\n", id, cache->id);
		cache_export(db, id++, cache);
	      }
	    } else {
	      printf("%d exporting %s\n", id, cache->id);
	      cache_export(db, id++, cache);
	    }
	  }
	}
	cache = cache->next;
	
	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(pbar), 
				      (float)(id-2)/(float)total);
	
	while(gtk_events_pending ())
	  gtk_main_iteration ();
      }
      sqlite3_close(db);

      gtk_widget_destroy(wait_dialog);
      
#ifdef USE_MAEMO
      if(!cancelled && !appdata->mmpoi_dontlaunch)
	dbus_mm_set_position(appdata, NULL);
#endif
    }
  } else {
    /* restore old mmpoi_path, in case it has been altered but not been used */
    free(appdata->mmpoi_path);
    appdata->mmpoi_path = strdup(old_mmpoi_path);
  }

  gtk_widget_destroy(context.dialog);
  
  if(context.handler_id) {
    printf("removing timer\n");
    gtk_timeout_remove(context.handler_id);
  }

  export_list_free(context.gpx);

  free(old_mmpoi_path);
}
