#include "settingsdialog.h"

SettingsDialog::SettingsDialog(QWidget *parent) :
    QDialog(parent),
    ui(new Ui::SettingsDialog)
{
    ui->setupUi(this);

    connect(ui->dirButton, SIGNAL(clicked()), this, SLOT(selectDir()));
    connect(ui->biosButton, SIGNAL(clicked()), this, SLOT(selectBios()));
    connect(ui->biosClearButton, SIGNAL(clicked()), this, SLOT(clearBios()));

    load();
}

SettingsDialog::~SettingsDialog()
{
    delete ui;
}

// Populate the dialog with fresh data
void SettingsDialog::load()
{
    ui->dirButton->setValueText(Path::romDir());

    QString biosFile;
    if (QFile::exists(biosFile = Path::emuDir() + "gba_bios.bin")) {
        ui->biosClearButton->show();
        ui->biosButton->setValueText(checkBios(biosFile) ? "Imported, looks good"
                                                         : "Imported, but seems incorrect");
    } else if (QFile::exists(biosFile = Path::emuRoot() + "gba_bios.bin")) {
        ui->biosClearButton->hide();
        ui->biosButton->setValueText(checkBios(biosFile) ? "Installed, looks good"
                                                         : "Installed, but seems incorrect");
    } else {
        ui->biosClearButton->hide();
        ui->biosButton->setValueText("Not found");
    }
}

// Prompt the user to select a directory wqith ROMs
void SettingsDialog::selectDir()
{
    // File picker can take a while to load, prevent multiple clicks
    this->setEnabled(false);

    QString romDir = QFileDialog::getExistingDirectory(this, "Select ROMs directory", Path::romDir());

    if (!romDir.isNull()) {
        Path::setRomDir(romDir);
        load();
    }

    this->setEnabled(true);
}

// Prompt the user to select a BIOS file
void SettingsDialog::selectBios()
{
    // File picker can take a while to load, prevent multiple clicks
    this->setEnabled(false);

    QString biosFile = QFileDialog::getOpenFileName(this, tr("Select BIOS"), Path::romDir(), tr("GBA BIOS (gba_bios.bin)"));

    if (!biosFile.isNull()) {
        QDir().mkpath(Path::emuDir());
        QFile::remove(Path::emuDir() + "gba_bios.bin");
        if (!QFile::copy(biosFile, Path::emuDir() + "gba_bios.bin"))
            QMaemo5InformationBox::information(this, "Failed to import the selected BIOS");

        load();
    }

    this->setEnabled(true);
}

// Remove the imported BIOS file
void SettingsDialog::clearBios()
{
    if (QMessageBox::warning(this, "Warning",
                             "The imported BIOS will be dropped.",
                             QMessageBox::Apply | QMessageBox::Cancel)
        != QMessageBox::Apply) return;

    if (!QFile::remove(Path::emuDir() + "gba_bios.bin"))
        QMaemo5InformationBox::information(this, "Failed to remove the imported BIOS");

    load();
}


bool SettingsDialog::checkBios(QString biosPath)
{
    char biosFirstByte = 0x0;

    QFile biosFile(biosPath);
    biosFile.open(QIODevice::ReadOnly);
    biosFile.getChar(&biosFirstByte);
    // TODO: Add size and checksum?

    return biosFirstByte == 0x18;
}
