/*****************************************************************************
 * Copyright: 2010-2011 Christian Fetzer <fetzer.ch@googlemail.com>          *
 * Copyright: 2010-2011 Michael Zanetti <mzanetti@kde.org>                   *
 *                                                                           *
 * This program is free software: you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation, either version 3 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.     *
 *                                                                           *
 ****************************************************************************/

#include "mobilitymap.h"

#include "geomap.h"
#include "gmwmarker.h"
#include "data/gmwvehicle.h"

#include <QtCore/QDebug>
#include <QtCore/QPropertyAnimation>
#include <QtCore/QParallelAnimationGroup>

#include <QtGui/QVBoxLayout>
#include <QApplication>

#include <qgeoboundingbox.h>
#include <qgeomappolygonobject.h>
#include <qgeomapgroupobject.h>
#include <qgeoroutingmanager.h>
#include <qgeomappolylineobject.h>
#include <QGeoRouteSegment>
#include <QGeoMapCircleObject>

/**
 * Private class ensures that map always fills view.
 */
class MapView : public QGraphicsView
{
public:
    MapView(QGraphicsScene *scene, QWidget *parent = 0) : QGraphicsView(scene, parent), m_map(0) {}
    void setMap(GeoMap* map) {
        if(m_map) {
            scene()->removeItem(m_map);
        }
        m_map = map;
        scene()->addItem(m_map);
        m_map->resize(viewport()->size());
        qDebug() << "setting map size to viewport size:" << viewport()->size();
    }

private:
    GeoMap* m_map;
    void keyPressEvent(QKeyEvent *event) {
        m_map->keyPressEvent(event);
    }
    void keyReleaseEvent(QKeyEvent *event) {
        m_map->keyReleaseEvent(event);
    }

protected:
    void resizeEvent(QResizeEvent *) {
        qDebug() << "resize event";
        if(m_map) {
            m_map->resize(viewport()->size());
        }
    }
};

MobilityMap::MobilityMap(GMWItemSortFilterProxyModel *model, const QString &cacheDir, QWidget *parent) :
    GMWMap(model, parent),
    m_serviceProvider(0),
    m_map(0),
    m_cacheDir(cacheDir),
    m_positionMarker(0)
{
    QGraphicsScene *scene = new QGraphicsScene;
    m_view = new MapView(scene, this);
    m_view->setVisible(true);
    m_view->setInteractive(true);
    m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_view->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_view->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    layout()->addWidget(m_view);
}

QStringList MobilityMap::supportedMapProviders() const
{
    return QGeoServiceProvider::availableServiceProviders();
}

QString MobilityMap::mapProvider() const
{
    return m_providerName;
}

bool MobilityMap::setMapProvider(const QString &provider)
{
    if(supportedMapProviders().contains(provider)) {
        if(m_serviceProvider != 0) {
            delete m_serviceProvider;
        }
        QMap<QString, QVariant> params;
        // set m_networkManager in ctor and enable this
//        params.insert("mapping.networkaccessmanager", m_networkManager);
        params.insert("mapping.cache.directory", m_cacheDir + "/" + provider);
        params.insert("mapping.cache.size", 1024 * 1024 * 20); // 20MB cache for maps
        m_serviceProvider = new QGeoServiceProvider(provider, params);
        if(m_serviceProvider->routingManager()) {
            m_serviceProvider->routingManager()->setLocale(QLocale("en"));
            connect(m_serviceProvider->routingManager(), SIGNAL(finished(QGeoRouteReply*)), SLOT(routingFinished(QGeoRouteReply*)));
            connect(m_serviceProvider->routingManager(), SIGNAL(error(QGeoRouteReply*,QGeoRouteReply::Error,QString)), SLOT(routingError(QGeoRouteReply*,QGeoRouteReply::Error,QString)));
        }
        m_providerName = provider;
        initialize();
        return true;
    } else {
        qCritical("GeoServiceProvider not found! Falling back to first available one...");
        if(!QGeoServiceProvider::availableServiceProviders().isEmpty()) {
            m_serviceProvider = new QGeoServiceProvider(QGeoServiceProvider::availableServiceProviders().first());
            m_providerName = provider;
            initialize();
        }
        return false;
    }
}

void MobilityMap::initialize()
{
    // Create Map
    m_mappingManager = m_serviceProvider->mappingManager();
    qDebug() << "Mapping Manager supported types and connectivity modes:" << m_mappingManager->supportedMapTypes() << m_mappingManager->supportedConnectivityModes();

    GeoMap *newMap = new GeoMap(m_mappingManager);
    m_view->setMap(newMap);
    if(m_map) {
        m_map->removeMapObject(m_positionMarker);
        m_map->removeMapObject(&m_routeObject);
        m_map->removeMapObject(&m_businessAreaGroup);
        m_map->removeMapObject(&m_itemGroup);
        m_map->deleteLater();
    }
    m_map = newMap;

    connect(m_map, SIGNAL(clicked(QGeoMapObject*)), SLOT(geoObjectClicked(QGeoMapObject*)));
    connect(m_map, SIGNAL(panned()), SLOT(disableTracking()));
    connect(m_map, SIGNAL(geometryChanged()), SIGNAL(resized()));
    connect(m_map, SIGNAL(panned()), SIGNAL(panned()));

    // Current Position
    QPixmap positionMarker(":/currentLoc.png");
    m_positionMarker = new QGeoMapPixmapObject(QGeoCoordinate(0,0), QPoint(-positionMarker.width()/2,-positionMarker.height()/2), positionMarker);
    m_positionMarker->setVisible(false);
    m_positionMarker->setZValue(3);
    m_map->addMapObject(m_positionMarker);

    QPen pen = m_routeObject.pen();
    pen.setColor(Qt::red);

    // See http://bugreports.qt.nokia.com/browse/QTMOBILITY-1655
    pen.setWidth(3);

    m_routeObject.setPen(pen);
    m_routeObject.setZValue(2);
    m_map->addMapObject(&m_routeObject);

    qDebug() << "map intitialzed";
}

void MobilityMap::setBusinessArea(const GMWBusinessArea &businessArea)
{
    // clean old ones
    m_map->removeMapObject(&m_businessAreaGroup);

    while(!m_businessAreaList.isEmpty()) {
        QGeoMapObject *obj = m_businessAreaList.takeFirst();
        m_businessAreaGroup.removeChildObject(obj);
        delete obj;
    }

    QPen pen(Qt::blue);
    pen.setWidth(5);
    QColor color(Qt::blue);
    color.setAlpha(30);
    QBrush brush(color);

    foreach(const Area &area, businessArea.areaList()) {
        QGeoMapPolygonObject *region = new QGeoMapPolygonObject();
        region->setBrush(brush);
        region->setPen(pen);

        region->setPath(area.path());
        m_businessAreaGroup.addChildObject(region);
        m_businessAreaList.append(region);
    }

    color = QColor(Qt::red);
    color.setAlpha(30);
    brush.setColor(color);

    foreach(const Area &area, businessArea.excludes()) {
        QGeoMapPolygonObject *region = new QGeoMapPolygonObject();
        region->setBrush(brush);
        region->setPen(pen);

        region->setPath(area.path());
        region->setZValue(0);
        m_businessAreaGroup.addChildObject(region);
        m_businessAreaList.append(region);
    }

    m_map->addMapObject(&m_businessAreaGroup);
    m_businessAreaGroup.setZValue(0);
}

void MobilityMap::positionUpdated(const QGeoPositionInfo &info)
{
    // It may happens that we get a position updated before we have a positionmarker if the user takes long to set the location initially
    if(!m_positionMarker) {
        return;
    }
    if (!m_positionMarker->isVisible()) {
        m_positionMarker->setVisible(true);
    }
    GMWMap::positionUpdated(info);
    m_positionMarker->setCoordinate(info.coordinate());

    if(!m_routeObject.route().path().isEmpty()) {
        if(m_routeObject.route().path().first().distanceTo(info.coordinate()) > 250) {
            routeTo(m_routeObject.route().path().last());
        }
    }
}

void MobilityMap::moveTo(const QGeoCoordinate &point)
{
    animatedPanTo(point);
}

void MobilityMap::routeTo(const QGeoCoordinate &point)
{
    if(!m_serviceProvider->routingManager()) {
        qDebug() << "Routing not supported by map provider";
        emit mapError(tr("Routing not supported by map provider"));
        return;
    }
    qDebug() << "routing from" << m_positionMarker->coordinate() << "to" << point;

    QGeoRouteRequest request;
    request.setTravelModes(QGeoRouteRequest::PedestrianTravel);
    request.setRouteOptimization(QGeoRouteRequest::ShortestRoute);
    request.setWaypoints(QList<QGeoCoordinate>() << m_positionMarker->coordinate() << point);

    m_serviceProvider->routingManager()->calculateRoute(request);
}

void MobilityMap::routingFinished(QGeoRouteReply *reply)
{
    qDebug() << "Routing finished. Found" << reply->routes().count() << "routes";
    if(reply->routes().count() <= 0) {
        reply->deleteLater();
        return;
    }

    qDebug() << "Nodes:" << reply->routes().first().path().count();
    qDebug() << "Distance:" << reply->routes().first().distance();
    qDebug() << reply->routes().first().routeId();
    qDebug() << reply->routes().first().travelMode();
    qDebug() << reply->routes().first().travelTime();
    qDebug() << reply->routes().first().firstRouteSegment().path();

    m_map->removeMapObject(&m_routeObject);
    m_routeObject.setRoute(reply->routes().first());
    m_map->addMapObject(&m_routeObject);

    reply->deleteLater();
}

void MobilityMap::routingError(QGeoRouteReply *reply, QGeoRouteReply::Error error, const QString &errorString)
{
    Q_UNUSED(error);
    qDebug() << "Routing failed:" << errorString;
    emit mapError(errorString);
    reply->deleteLater();
}

void MobilityMap::zoomIn()
{
    qDebug() << "zooming map";
    m_map->setZoomLevel(m_map->zoomLevel() + 1);
}

void MobilityMap::zoomOut()
{
    qDebug() << "zooming map";
    m_map->setZoomLevel(m_map->zoomLevel() - 1);
}

void MobilityMap::zoomTo(const QGeoBoundingBox &bounds)
{
    m_map->fitInViewport(bounds);

//    QGeoMapCircleObject *obj = new QGeoMapCircleObject(bounds.topLeft(), 20);
//    m_map->addMapObject(obj);
//    obj = new QGeoMapCircleObject(bounds.topRight(), 20);
//    m_map->addMapObject(obj);
//    obj = new QGeoMapCircleObject(bounds.bottomLeft(), 20);
//    m_map->addMapObject(obj);
//    obj = new QGeoMapCircleObject(bounds.bottomRight(), 20);
//    m_map->addMapObject(obj);

    // This works around some strange behavior in QGraphicsGeoMap:
    // fitInViewPort sometimes does not zoom in into the map but only centers
    // to the given bounds. We zoom in to level 12 for now as this
    // seems to be best suited to display an entire city
#ifdef Q_WS_MAEMO_5
    m_map->setZoomLevel(12);
#endif
//    qDebug() << "*** setting center to" << bounds.center();
//    m_map->setCenter(bounds.center());
}

void MobilityMap::refresh()
{

}

void MobilityMap::rowsInserted(const QModelIndex &parent, int start, int end)
{
    Q_UNUSED(parent)

    if(!m_map) {
        return;
    }
    m_map->removeMapObject(&m_itemGroup);

    qDebug() << "Inserting markers into map" << start << "-" << end << "Current markers:" << m_items.count() << "New markers:" << end-start+1;
    QTime timer;
    QTime ges;
    ges.start();
    timer.start();
    for(int i = start; i <= end; ++i) {
        GMWItem *item = m_model->data(m_model->index(i, 0), Qt::UserRole).value<GMWItem*>();
        GMWMarker *marker = new GMWMarker(item);
        marker->setCoordinate(item->location());
        //marker->setZValue((int)item->objectType());
        m_itemGroup.addChildObject(marker);
        m_items.insert(item, marker);
        //qDebug() << "inserted item" << i << timer.restart() << "ms";
    }
    qDebug() << "Markers inserted; Markers:" << m_items.count() << "time:" << ges.elapsed();
    m_itemGroup.setZValue(1);
    m_map->addMapObject(&m_itemGroup);
}

void MobilityMap::rowsAboutToBeRemoved(const QModelIndex &parent, int start, int end)
{
    Q_UNUSED(parent)

//    qDebug() << "Removing markers from map" << start << "-" << end << "Current markers:" << m_items.count() << "Markers to remove:" << end-start+1;
    for(int i = start; i <= end; ++i) {
        QModelIndex index = m_model->index(i, 0);
        GMWItem *item = m_model->data(index, Qt::UserRole).value<GMWItem*>();
        GMWMarker *marker = m_items.value(item);
        m_itemGroup.removeChildObject(marker);
        m_items.remove(item);
        delete marker;
    }

//    qDebug() << "items removed; Items:" << m_items.count();
}

void MobilityMap::geoObjectClicked(QGeoMapObject *object)
{
    GMWMarker *marker = dynamic_cast<GMWMarker*>(object);
    if (!marker) return;

    GMWItem *item = m_items.key(marker);
    emit objectClicked(item);
}

void MobilityMap::animatedPanTo(const QGeoCoordinate &center)
{
    qDebug() << "panning to" << center;
    if (!m_map || !center.isValid()) return;

    QPropertyAnimation *latAnim = new QPropertyAnimation(m_map, "centerLatitude");
    latAnim->setEndValue(center.latitude());
    latAnim->setDuration(200);
    QPropertyAnimation *lonAnim = new QPropertyAnimation(m_map, "centerLongitude");
    lonAnim->setEndValue(center.longitude());
    lonAnim->setDuration(200);

    QParallelAnimationGroup *group = new QParallelAnimationGroup;
    group->addAnimation(latAnim);
    group->addAnimation(lonAnim);
    group->start(QAbstractAnimation::DeleteWhenStopped);
}


void MobilityMap::setActiveItem(GMWItem *highlightItem)
{
    foreach (GMWMarker* marker, m_items) {
        marker->setHighlight(false);
    }
    m_items.value(highlightItem)->setHighlight(true);
}
