/*****************************************************************************
 * Copyright: 2010-2011 Christian Fetzer <fetzer.ch@googlemail.com>          *
 * Copyright: 2010-2011 Michael Zanetti <mzanetti@kde.org>                   *
 *                                                                           *
 * This program is free software: you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation, either version 3 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.     *
 *                                                                           *
 ****************************************************************************/

#include "core.h"

#include "engines/gmwengine.h"
#include "engines/car2go/car2goengine.h"

#include "data/gmwitemmodel.h"
#include "data/gmwitemsortfilterproxymodel.h"

#include "maps/gmwmap.h"
#include "maps/qtm-location/mobilitymap.h"

#include "mainwindow.h"

#include <QtCore/QDebug>
#include <QtCore/QDir>

class CorePrivate
{
public:
    void init();
    void deinit();

    GMWEngine *engine() const { return m_engine; }
    GMWItemModel *model() const { return m_model; }
    GMWItemSortFilterProxyModel *proxyModel() const { return m_proxyModel; }
    GMWMap *map() const { return m_map; }
    MainWindow  *mainWindow() const { return m_mainWindow; }

private:
    GMWEngine* m_engine;
    GMWItemModel* m_model;
    GMWItemSortFilterProxyModel* m_proxyModel;
    GMWMap *m_map;
    MainWindow *m_mainWindow;
};

Q_GLOBAL_STATIC(CorePrivate, instance);

namespace Core
{
    void init() { return instance()->init(); }
    void deinit() { return instance()->deinit(); }

    GMWEngine *engine() { return instance()->engine(); }
    GMWItemModel *model() { return instance()->model(); }
    GMWItemSortFilterProxyModel *proxyModel() { return instance()->proxyModel(); }
    GMWMap *map() { return instance()->map(); }
    MainWindow *mainWindow() { return instance()->mainWindow(); }
};

void CorePrivate::init()
{
    // Create Model
    qDebug() << "Creating model";
    m_model = new GMWItemModel();
    m_proxyModel = new GMWItemSortFilterProxyModel();
    m_proxyModel->setSourceModel(m_model);
    m_proxyModel->setDynamicSortFilter(true);

    // Create Engine
    qDebug() << "Creating engine";
    m_engine = new Car2goEngine(QDir::homePath() + "/.getmewheels/");
    QObject::connect(m_engine, SIGNAL(objectsReceived(QList<GMWItem*>)), m_model, SLOT(addObjects(QList<GMWItem*>)));

    // Create Map
    qDebug() << "Creating map";
    m_map = new MobilityMap(m_proxyModel, QDir::homePath() + "/.getmewheels/map/");


    // Show MainWindow
    m_mainWindow = new MainWindow();
    m_mainWindow->show();
    m_mainWindow->initialize();
}

void CorePrivate::deinit()
{
    delete m_engine;
    delete m_map;
    delete m_model;
    delete m_proxyModel;
    delete m_mainWindow;
}
