/*****************************************************************************
 * Copyright: 2010-2011 Christian Fetzer <fetzer.ch@googlemail.com>          *
 * Copyright: 2010-2011 Michael Zanetti <mzanetti@kde.org>                   *
 *                                                                           *
 * This program is free software: you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation, either version 3 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.     *
 *                                                                           *
 ****************************************************************************/

#include <QtGui>
#include <QDebug>

#include <QtNetwork/QNetworkAccessManager>
#include <QtNetwork/QNetworkRequest>
#include <QNetworkReply>

#include "mainwindow.h"
#include "core.h"

#include "maps/gmwmap.h"
#include "engines/gmwengine.h"
#include "data/gmwitemmodel.h"
#include "data/gmwitemsortfilterproxymodel.h"
#include "data/gmwvehicle.h"
#include "data/gmwgasstation.h"
#include "ui/menubar.h"
#include "ui/settingsdialog.h"
#include "ui/itemlist.h"
#include "ui/itemdialog.h"

#include <QGeoPositionInfoSource>
#include <QMenuBar>
#include <QTimer>

#ifdef Q_WS_MAEMO_5
    #include <QtGui/QX11Info>
    #include <X11/Xlib.h>
    #include <X11/Xatom.h>
#endif

MainWindow::MainWindow() :
    m_itemDialog(0),
    m_direction(0)
{
#ifdef Q_WS_MAEMO_5
    m_orientation = Qt::Horizontal;
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

    setWindowTitle("Get Me Wheels");
}

void MainWindow::initialize()
{
    connect(qApp, SIGNAL(onlineStateChanged(bool)), SLOT(onlineStateChanged(bool)));

    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));

    // select the map displaying engine. Hardcoded for now
    setCentralWidget(Core::map());
    centralWidget()->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding);
    connect(Core::map(), SIGNAL(objectClicked(GMWItem*)), this, SLOT(objectClicked(GMWItem*)));
    connect(Core::map(), SIGNAL(showList()), this, SLOT(showList()));

    // create Engine
    connect(Core::engine(), SIGNAL(downloadStarted()), SLOT(downloadStarted()));
    connect(Core::engine(), SIGNAL(downloadFinished(bool , const QString &)), SLOT(downloadFinished(bool, const QString &)));
    connect(Core::engine(), SIGNAL(loadedFromCache(GMWItem::Type,QDateTime)), SLOT(loadedFromCache(GMWItem::Type,QDateTime)));

    // Menubar
    setMenuBar(new MenuBar(this));

    // Status Bar
    createStatusBar();

    // create Object List
    m_objectList = new GMWObjectList(Core::proxyModel(), this);
    m_objectList->layout()->setMenuBar(new MenuBar(m_objectList));
    connect(m_objectList, SIGNAL(zoomTo(QGeoCoordinate)), Core::map(), SLOT(disableTracking()));
    connect(m_objectList, SIGNAL(zoomTo(QGeoCoordinate)), Core::map(), SLOT(moveTo(QGeoCoordinate)));

#ifdef Q_WS_MAEMO_5
    m_objectList->setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

    // load Settings
    loadSettings();
    qDebug() << "settings loaded";
    refreshAll();

    // GPS
    m_gps = QGeoPositionInfoSource::createDefaultSource(this);
    if (m_gps) {
        m_gps->setUpdateInterval(5000);
        connect(m_gps, SIGNAL(positionUpdated(const QGeoPositionInfo&)), this, SLOT(positionUpdated(const QGeoPositionInfo&)));
        m_gps->startUpdates();
    } else {
        qDebug() << "No GPS available";
        //m_map->positionUpdated(QGeoPositionInfo(QGeoCoordinate(-27.5796, 153.1), QDateTime::currentDateTime()));
    }
    connect(this, SIGNAL(positionUpdated(const QGeoCoordinate&, qreal)), Core::model(), SLOT(currentPositionChanged(const QGeoCoordinate&, qreal)));
    Core::map()->setTracking(true);

    grabZoomKeys(true);

//    staticMetaObject.invokeMethod(this, "centerOnStartCoordinates", Qt::QueuedConnection);

    //startTimer(5000);
}

MainWindow::~MainWindow(){
    grabZoomKeys(false);
    delete m_objectList;
}

void MainWindow::closeEvent(QCloseEvent *event) {
    if (true) {
        event->accept();
    } else {
        event->ignore();
    }
}

void MainWindow::positionUpdated(const QGeoPositionInfo &info) {
    //qDebug() << "GPS Position updated:" << QDateTime::currentDateTime().toString();
    Core::map()->positionUpdated(info);
    statusBarGPS->setText("");

    // Cache the direction (not included in every gps update)
    if (info.hasAttribute(QGeoPositionInfo::Direction)) {
        m_direction = info.attribute(QGeoPositionInfo::Direction);
        //qDebug() << "GPS direection received:" << m_direction;
    }
    emit positionUpdated(info.coordinate(), m_direction);
}

void MainWindow::timerEvent(QTimerEvent *event)
{
    qDebug() << "Timer ID:" << event->timerId();
    //qDebug() << qrand() % 180 - 90;
    QGeoPositionInfo position(QGeoCoordinate(qrand() % 180 - 90, qrand() % 360 - 180), QDateTime::currentDateTime());

    if (!position.isValid()) {
        qDebug() << position;
    }
    position.setAttribute(QGeoPositionInfo::Direction, qrand() % 360);
    positionUpdated(position);
}

void MainWindow::about() {
   QMessageBox::about(this, tr("Get Me Wheels - Car finder"),
                      tr("Get Me Wheels - Car finder <br> Authors: <br>Christian Fetzer (fetzer.ch@googlemail.com) <br> Michael Zanetti (mzanetti@kde.org)"));
}

void MainWindow::createStatusBar() {
    statusBarGPS = new QLabel(tr("Waiting for GPS..."));
    statusBarDownload = new QLabel(tr("No data available"));
    statusBarDownload->setAlignment(Qt::AlignRight);

    statusBar()->addWidget(statusBarGPS, 1);
    statusBar()->addWidget(statusBarDownload, 1);
}

void MainWindow::refreshAll(bool useCache) {
    Core::model()->clearAll();
    Core::engine()->refreshStationary(useCache);
    Core::engine()->refreshVehicles(useCache);
}

void MainWindow::refreshVehicles() {
    Core::model()->clearVehicles();
    Core::engine()->refreshVehicles(false);
}

void MainWindow::objectClicked(GMWItem *object) {
    if(!m_itemDialog) {
        m_itemDialog = new ItemDialog(this);
        m_itemDialog->addObject(object);
        connect(m_itemDialog, SIGNAL(finished(int)), this, SLOT(itemDialogClosed()));
        connect(m_itemDialog, SIGNAL(zoomTo(QGeoCoordinate)), Core::map(), SLOT(disableTracking()));
        connect(m_itemDialog, SIGNAL(zoomTo(QGeoCoordinate)), Core::map(), SLOT(moveTo(QGeoCoordinate)));
        connect(m_itemDialog, SIGNAL(zoomTo(QGeoCoordinate)), Core::map(), SLOT(routeTo(QGeoCoordinate)));
        connect(Core::model(), SIGNAL(layoutChanged()), m_itemDialog, SLOT(currentPositionChanged()));

#ifdef Q_WS_MAEMO_5
        QSettings settings("Get Me Wheels", "getmewheels");
        m_itemDialog->setAttribute(Qt::WA_Maemo5AutoOrientation, settings.value("Autorotation", true).toBool());
#endif
        m_itemDialog->show();
    } else {
        m_itemDialog->addObject(object);
    }
}

void MainWindow::openSettings() {
    SettingsDialog *dialog = new SettingsDialog(this);
    QSettings settings("Get Me Wheels", "getmewheels");

    dialog->setLocations(Core::engine()->supportedLocations(), settings.value("Location").toString());
    dialog->setMapProviders(Core::map()->supportedMapProviders(), settings.value("MapProvider").toString());
    dialog->setRotationEnabled(settings.value("Autorotation", true).toBool());

    if(dialog->exec() == QDialog::Accepted) {
        settings.setValue("Location", dialog->location());
        settings.setValue("MapProvider", dialog->mapProvider());
        settings.setValue("Autorotation", dialog->rotationEnabled());
        // If a user sets the location explicitly he most likely wants to see the location instead his current position => disable tracking
        Core::map()->setTracking(false);
        loadSettings();
        refreshAll(false);
    }
    delete dialog;
}

void MainWindow::loadSettings() {
    QSettings settings("Get Me Wheels", "getmewheels");

    // Check if all settings are here
    if(!settings.contains("MapProvider") || !settings.contains("Location")) {
        openSettings();
    }

    bool changed = false;

    QString mapProvider = settings.value("MapProvider").toString();
    if(mapProvider != Core::map()->mapProvider()) {
        if(!Core::map()->setMapProvider(mapProvider)) {
            if(!Core::map()->supportedMapProviders().isEmpty()) {
                settings.setValue("MapProvider", Core::map()->supportedMapProviders().first());
            }
        }
        changed = true;
    }

    QString location = settings.value("Location").toString();
    qDebug() << "loading settings location:" << location << ". Old location is:" << Core::engine()->location();
    if(Core::engine()->supportedLocations().contains(location) && (Core::engine()->location() != location)) {
        changed = true;
        Core::model()->clearAll();
        Core::engine()->setLocation(location);
    }

    if(changed) {
        qDebug() << "centering!";
        staticMetaObject.invokeMethod(this, "centerOnStartCoordinates", Qt::QueuedConnection);
    }

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5AutoOrientation, settings.value("Autorotation", true).toBool());
#endif
}

void MainWindow::centerOnStartCoordinates()
{
    // Center Map
    qDebug() << "zooming map to " << Core::engine()->startingBounds().center();
    Core::map()->zoomTo(Core::engine()->startingBounds());
}

void MainWindow::showList() {
    m_objectList->show();
}

void MainWindow::orientationChanged(){
#ifdef Q_WS_MAEMO_5
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    if (screenGeometry.width() > screenGeometry.height()) {
        qDebug() << "In Landscape Mode";
        m_orientation = Qt::Horizontal;
    } else {
        qDebug() << "In Portrait Mode";
        m_orientation = Qt::Vertical;
    }
//    if(m_itemDialog) {
//        m_itemDialog->rotate(m_orientation);
//    }
#endif
}

void MainWindow::onlineStateChanged(bool online) {
    if (online) {
        qDebug() << "Now online, refreshing map";
        Core::map()->refresh();
    }
}

void MainWindow::grabZoomKeys(bool grab) {
#ifndef Q_WS_MAEMO_5
    Q_UNUSED(grab)
#else
    if (!winId()) {
        qWarning("Can't grab keys unless we have a window id");
        return;
    }

    unsigned long val = (grab) ? 1 : 0;
    Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
    if (!atom) {
        qWarning("Unable to obtain _HILDON_ZOOM_KEY_ATOM. This example will only work "
                 "on a Maemo 5 device!");
        return;
    }

    XChangeProperty (QX11Info::display(),
             winId(),
             atom,
             XA_INTEGER,
             32,
             PropModeReplace,
             reinterpret_cast<unsigned char *>(&val),
             1);
#endif
 }

void MainWindow::downloadStarted() {
    statusBarDownload->setText(tr("Downloading data..."));
}

void MainWindow::downloadFinished(bool success, const QString &errorString) {
    if(success){
        statusBarDownload->setText(tr("Data download successful @ %1").arg(QDateTime::currentDateTime().toString("h:mm")));
    } else {
        statusBarDownload->setText(tr("Data download failed: %1").arg(errorString));
    }
}

void MainWindow::loadedFromCache(GMWItem::Type type, const QDateTime &cacheDate) {
    QString typeString;
    switch (type) {
    case GMWItem::TypeVehicle: typeString = tr("Vehicles"); break;
    case GMWItem::TypeGasStation: typeString = tr("Gas Stations"); break;
    case GMWItem::TypeParkingLot: typeString = tr("Parking Lots"); break;
    default: qWarning("Unknown data loaded from cache");
    }
    qint64 age = cacheDate.msecsTo(QDateTime::currentDateTime()) / 1000;
    QString ageString;
    if (age < 60) ageString = QString::number(age) + "s";
    else if (age < 60*60) ageString = QString::number(age / 60) + "min";
    else if (age < 60*60*24) ageString = QString::number(age / (60*60)) + "h";
    else ageString = QString::number(age / (60*60*24)) + "day(s)";

    statusBarDownload->setText(tr("Loaded %1 from cache (%2)").arg(typeString).arg(ageString));
}

void MainWindow::itemDialogClosed() {
    qDebug() << "deleting item dialog";
    m_itemDialog->deleteLater();
    m_itemDialog = 0;
}
