/*
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007-2010 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <math.h>

#include "uiview.h"
#include "ui.h"
#include "uisettings.h"
#include "uifilter.h"
#include "uiicons.h"
#include "uimisc.h"

/*******************************************************************
 *
 * Private definitions
 *
 *******************************************************************/

/*******************************************************************
 *
 * Private variables
 *
 *******************************************************************/

static
gchar *column_headers[] = {gettext_noop("Date"),
			   gettext_noop("Km"),
			   gettext_noop("Trip\n[km]"),
			   gettext_noop("Fill\n[litre]"),
			   gettext_noop("[litre/\n100 km]"),
			   gettext_noop("Price"),
			   gettext_noop("Price/\nkm"),
			   gettext_noop("Price/\nlitre"),
			   gettext_noop("Service"),
			   gettext_noop("Oil"),
			   gettext_noop("Tires"),
			   gettext_noop("CO2 Emissions\n[g/km]"),
			   gettext_noop("Notes"),
			   "icon", // for icon
			   "fancy", // for the fancy combined column
};

static
gchar *column_headers_non_SI[] = {gettext_noop("Date"),
				  gettext_noop("Miles"),
				  gettext_noop("Trip\n[miles]"),
				  gettext_noop("Fill\n[gal.]"),
				  gettext_noop("[Miles/\ngallon]"),
				  gettext_noop("Price"),
				  gettext_noop("Price/\nmile"),
				  gettext_noop("Price/\ngal."),
				  gettext_noop("Service"),
				  gettext_noop("Oil"),
				  gettext_noop("Tires"),
				  gettext_noop("CO2 Emissions\n[lb/100 miles]"),
				  gettext_noop("Notes"),
};

/*******************************************************************
 *
 * Prototypes of private functions
 *
 *******************************************************************/

static gboolean visible_func (GtkTreeModel *model, GtkTreeIter *iter, gpointer data);

/*******************************************************************
 *
 * Private functions
 *
 *******************************************************************/

/**
 * \fn void update_column_headers(GtkWidget *view)
 * \brief Updates the headers to correspond to the current unit
 *
 */
static
void update_column_headers(GtkWidget *view)
{
  GtkTreeViewColumn   *col;
  GtkWidget           *label;
  GString *header;
  gint i;

  for (i=COL_DAY;i<COL_NOTES;i++) {
    col = gtk_tree_view_get_column(GTK_TREE_VIEW(view), i);
    label = gtk_tree_view_column_get_widget(col); 
    header = get_column_header(i);
    gtk_label_set_text(GTK_LABEL(label), header->str);
    g_string_free(header, TRUE);
  }
}

static
void sort_column_changed(GtkTreeSortable *sortable,
			  gpointer         pui)
{
  gint colid;
  GtkSortType sortorder;

  if (gtk_tree_sortable_get_sort_column_id(sortable, &colid, &sortorder)) {
    maintablesorted=TRUE;
    maintablesortorder=sortorder;
    maintablesortcol=colid;
  }
  else {
    maintablesorted=FALSE;
  }
}

static
gint sort_iter_compare_func (GtkTreeModel *model,
                        GtkTreeIter  *a,
                        GtkTreeIter  *b,
                        gpointer      userdata)
{
  gint sortcol = GPOINTER_TO_INT(userdata); /* not used yet */
  gint ret = 0;

  switch (sortcol)
    {
    case COL_KM:
    case COL_TRIP:
    case COL_FILL:
    case COL_CONSUM:
    case COL_PRICE:
    case COL_PRICEPERTRIP:
    case COL_PRICEPERLITRE:
    case COL_SERVICE:
    case COL_OIL:
    case COL_TIRES: {
      
      double result1, result2;

      gtk_tree_model_get(model, a, sortcol, &result1, -1);
      gtk_tree_model_get(model, b, sortcol, &result2, -1);
      
      if (result1 != result2)
	{
	  if (!isnan(result1) && !isnan(result2))
	    ret = (result1 > result2) ? 1 : -1;
	  else {
	    if (isnan(result1) && !isnan(result2)) {
		  ret = -1;
	    }
	    else if (!isnan(result1) && isnan(result2)) {
	      ret = 1;
	    }
	  }
	}
      /* else both equal => ret = 0 */
    }
      break;
    case COL_DAY: {
      char *date1, *date2;
      gchar *userdate1, *userdate2;

      gtk_tree_model_get(model, a, sortcol, &userdate1, -1);
      gtk_tree_model_get(model, b, sortcol, &userdate2, -1);

      date2sqlite(&date1, datefmtstr[curunit.dateformat], userdate1);
      date2sqlite(&date2, datefmtstr[curunit.dateformat], userdate2);

      ret=strcmp(date1,date2);

      free(date1);
      free(date2);
      g_free(userdate1);
      g_free(userdate2);

    }
      break;
    case COL_NOTES:{

        gchar *name1, *name2;
	
        gtk_tree_model_get(model, a, sortcol, &name1, -1);
        gtk_tree_model_get(model, b, sortcol, &name2, -1);

        if (name1 == NULL || name2 == NULL)
	  {
	    if (name1 == NULL && name2 == NULL)
	      break; /* both equal => ret = 0 */

	    ret = (name1 == NULL) ? -1 : 1;
	  }
        else
	  {
	    ret = g_utf8_collate(name1,name2);
	  }

        g_free(name1);
        g_free(name2);
      }
      break;
    }

  return ret;
}

static gboolean
visible_func (GtkTreeModel *model,
              GtkTreeIter  *iter,
              gpointer      data)
{
  gchar *str;
  char *date;
  gdouble colval;
  gboolean visible = FALSE;
  gboolean compres;
  GList *filt;
  GList *lp;
  recordfilter_t *p;
  gboolean first;
  gboolean and;

  filt=*(GList **)data;

  gtk_tree_model_get (model, iter, COL_DAY, &str, -1);

  if (str) {
    visible = TRUE;

    if (filter_filterqueue_length() > 0) {

      lp=g_list_first(filt);
      first=TRUE;
      while (lp != NULL) {
	p=(recordfilter_t *)lp->data;
	and=TRUE;
	compres=TRUE;
	if (p->column==COL_DAY) {
	  date2sqlite(&date, datefmtstr[curunit.dateformat], str);
	  compres = strcmp(p->mindate->str, date) <= 0 && strcmp(p->maxdate->str, date) >= 0;
	  and=p->and;
	  free(date);
	  g_free(str);
	}
	else if (p->column==COL_NOTES) {
	  gtk_tree_model_get (model, iter, COL_NOTES, &str, -1);
	  if (str) {
	    compres = g_pattern_match_string(p->pattern, str);
	    and=p->and;
	    g_free(str);
	  }
	}
	else {
	  gtk_tree_model_get (model, iter, p->column, &colval, -1);
	  compres = colval >= p->minval && colval <= p->maxval;
	  and=p->and;
	}

	if (!first) {
	  if (and)
	    visible &= compres;
	  else
	    visible |= compres;
	}
	else {
	  first=FALSE;
	  visible = compres;
	}

	lp=g_list_next(lp);
      }

    }
  }

  return visible;
}

static
void pixbuf_cell_data_func (GtkTreeViewColumn *col,
			    GtkCellRenderer   *renderer,
			    GtkTreeModel      *model,
			    GtkTreeIter       *iter,
			    gpointer           user_data)
{

  static GdkPixbuf *pixbuf_fill=NULL;
  static GdkPixbuf *pixbuf_service=NULL;
  static GdkPixbuf *pixbuf_oil=NULL;
  static GdkPixbuf *pixbuf_tires=NULL;
  static GdkPixbuf *pixbuf_misc=NULL;
  GdkPixbuf *pixbuf;
  double fill;
  double service;
  double oil;
  double tires;

  if (!(model==NULL)) {
    gtk_tree_model_get(model, iter, COL_FILL, &fill, COL_SERVICE, &service,
		       COL_OIL, &oil, COL_TIRES, &tires, -1);
    if (fill>0.0) {
      //		g_print("This record is a fuel fill record\n");
      if (!pixbuf_fill)
	pixbuf_fill=gdk_pixbuf_new_from_xpm_data((const char **)&gaspump_xpm);
      pixbuf = pixbuf_fill;
    }
    else if (service>0.0) {
      //		g_print("This record is a service record\n");
      if (!pixbuf_service)
	pixbuf_service=gdk_pixbuf_new_from_xpm_data((const char **)&wrench_xpm);
      pixbuf = pixbuf_service;
    }
    else if (oil>0.0) {
      //		g_print("This record is an oil record\n");
      if (!pixbuf_oil)
	pixbuf_oil=gdk_pixbuf_new_from_xpm_data((const char **)&water_drop_xpm);
      pixbuf = pixbuf_oil;
    }
    else if (tires>0.0) {
      //		g_print("This record is a tires record\n");
      if (!pixbuf_tires)
	pixbuf_tires=gdk_pixbuf_new_from_xpm_data((const char **)&tyre_xpm);
      pixbuf = pixbuf_tires;
    }
    else {
      //		g_print("This is an other record\n");
      if (!pixbuf_misc)
	pixbuf_misc=gdk_pixbuf_new_from_xpm_data((const char **)&storage_cylinder_xpm);
      pixbuf = pixbuf_misc;
    }
    g_object_set(renderer, "pixbuf", pixbuf, NULL);
  }
}

static
void new_cell_data_func (GtkTreeViewColumn *col,
		    GtkCellRenderer   *renderer,
		    GtkTreeModel      *model,
		    GtkTreeIter       *iter,
		    gpointer           user_data)
{
  char *date;
  double km;
  double trip;
  double fill;
  double price;
  double pricepertrip;
  double priceperlitre;
  double consum;
  double emission;
  double service;
  double oil;
  double tires;
  gchar *notes;
  gchar *markup;
  gchar format_with_notes[] = "<span foreground=\"%s\">%s: %.0f %s %.2f %s (%s)</span>\n<span foreground=\"%s\" size=\"x-small\">%.2f %s (%.1f %s, %.2f %s/%s, %.2f %s/%s) %.1f %s (%.0f %s) </span> ";
  gchar format_without_notes[] = "<span foreground=\"%s\">%s: %.0f %s %.2f %s</span>\n<span foreground=\"%s\" size=\"x-small\">%.2f %s (%.1f %s, %.2f %s/%s, %.2f %s/%s) %.1f %s (%.0f %s) </span> ";

  /* Note: data in model is already in user units, no need to use conversion functions */
  gtk_tree_model_get(model, iter, COL_DAY, &date, COL_KM, &km, COL_TRIP, &trip, COL_FILL, &fill,
		     COL_PRICE, &price, COL_PRICEPERTRIP, &pricepertrip,
		     COL_PRICEPERLITRE, &priceperlitre, COL_CONSUM, &consum,
		     COL_SERVICE, &service, COL_OIL, &oil, COL_TIRES, &tires,
		     COL_CO2EMISSION, &emission, COL_NOTES, &notes, -1);

  if (strlen(notes)==0) {
    markup=g_markup_printf_escaped(format_without_notes,
				   get_default_text_color(),
				   date, 
				   km, ((curunit.lengthunit==SI)? _("km") : _("mi")),
				   price+service+oil+tires, curunit.currency->str, 
				   get_secondary_text_color(),
				   fill, ((curunit.volumeunit==SI)? _("litres") : _("gals.")),
				   trip, ((curunit.lengthunit==SI)? _("km") : _("mi")),
				   pricepertrip, curunit.currency->str,
				   ((curunit.lengthunit==SI)? _("km") : _("mi")),
				   priceperlitre, curunit.currency->str, 
				   ((curunit.volumeunit==SI)? _("litres") : _("gals.")),
				   consum, 
				   ((curunit.consumeunit==SI)? _("l/100 km") : _("mpg")),
				   emission,
				   ( (curunit.massunit==SI)?
				     (curunit.lengthunit==SI)? _("g/km") : _("g/mi")
				     : (curunit.lengthunit==SI)? _("lb/100 km") : _("lb/100 mi")));
  }
  else {
    markup=g_markup_printf_escaped(format_with_notes,
				   get_default_text_color(),
				   date, 
				   km, ((curunit.lengthunit==SI)? _("km") : _("mi")),
				   price+service+oil+tires, curunit.currency->str, 
				   notes,
				   get_secondary_text_color(),
				   fill, ((curunit.volumeunit==SI)? _("litres") : _("gals.")),
				   trip, ((curunit.lengthunit==SI)? _("km") : _("mi")),
				   pricepertrip, curunit.currency->str,
				   ((curunit.lengthunit==SI)? _("km") : _("mi")),
				   priceperlitre, curunit.currency->str, 
				   ((curunit.volumeunit==SI)? _("litres") : _("gals.")),
				   consum, 
				   ((curunit.consumeunit==SI)? _("l/100 km") : _("mpg")),
				   emission,
				   ( (curunit.massunit==SI)?
				     (curunit.lengthunit==SI)? _("g/km") : _("g/mi")
				     : (curunit.lengthunit==SI)? _("lb/100 km") : _("lb/100 mi")));
  }
  g_object_set(renderer, "markup", markup, NULL);
  g_free(markup);
  g_free(notes);
}

static
void cell_data_func (GtkTreeViewColumn *col,
		      GtkCellRenderer   *renderer,
		      GtkTreeModel      *model,
		      GtkTreeIter       *iter,
		      gpointer           user_data)
{
  double  value;
  gchar   buf[20];

  gtk_tree_model_get(model, iter, ((data_func_t *)user_data)->col, &value, -1);

  switch (((data_func_t *)user_data)->col) {
  case COL_CONSUM:
  case COL_CO2EMISSION:
  case COL_PRICEPERTRIP:
    if (fabs(value)<1.0e-5) 
      g_snprintf(buf, sizeof(buf), "%s", "-");
    else
      g_snprintf(buf, sizeof(buf), ((data_func_t *)user_data)->format, value);
    break;
  default:
    g_snprintf(buf, sizeof(buf), ((data_func_t *)user_data)->format, value);
  }

  g_object_set(renderer, "text", buf, NULL);
}

void callback_recordactivated (GtkTreeView        *treeview,
                               GtkTreePath        *path,
                               GtkTreeViewColumn  *col,
                               AppUIData *pui)
{
  GtkTreeModel *model;
  GtkTreeIter   iter;

  model = gtk_tree_view_get_model(treeview);

  if (gtk_tree_model_get_iter(model, &iter, path)) {
    callback_editrecord(NULL, pui);
  }
}

/*******************************************************************
 *
 * Public functions
 *
 *******************************************************************/

/**
 * \fn GString *get_column_header(gint col)
 * \brief Returns the header of column col
 * \param col the column number
 * \return Pointer to a newly allocated column header string
 *
 * This functions returns the column header taking the used unit into
 * account. The string is allocated from the heap. After usage, the
 * string must be freed with g_string_free.
 *
 */
GString *get_column_header(gint col)
{
  GString *header;

  header = g_string_new(NULL);

  switch (col) {
  case COL_DAY:
  case COL_PRICE:
  case COL_SERVICE:
  case COL_OIL:
  case COL_TIRES:
  case COL_NOTES:
    g_string_assign(header, gettext(column_headers[col]));
    break;
  case COL_KM:
  case COL_TRIP:
  case COL_PRICEPERTRIP:
    if (curunit.lengthunit==SI)
      g_string_assign(header, gettext(column_headers[col]));
    else
      g_string_assign(header, gettext(column_headers_non_SI[col]));
    break;
  case COL_FILL:
  case COL_PRICEPERLITRE:
    if (curunit.volumeunit==SI)
      g_string_assign(header, gettext(column_headers[col]));
    else
      g_string_assign(header, gettext(column_headers_non_SI[col]));
    break;
  case COL_CONSUM:
    if (curunit.consumeunit==SI)
      g_string_assign(header, gettext(column_headers[col]));
    else
      g_string_assign(header, gettext(column_headers_non_SI[col]));
    break;
  case COL_CO2EMISSION:
    if (curunit.massunit==SI) {
      if (curunit.lengthunit==SI)
	g_string_assign(header, _("CO2 Emissions\n[g/km]"));
      else
	g_string_assign(header, _("CO2 Emissions\n[g/mile]"));
    }
    else {
    if (curunit.lengthunit==SI)
	g_string_assign(header, _("CO2 Emissions\n[lb/100 km]"));
      else
	g_string_assign(header, _("CO2 Emissions\n[lb/100 miles]"));
    }
    break;
  }

  return header;
}

/**
 * \fn GString *get_column_header_one_line(gint col)
 * \brief Returns the header of column col on one line
 * \param col the column number
 * \return Pointer to a newly allocated column header string
 *
 * This functions returns the column header taking the used unit into
 * account. The string is allocated from the heap. After usage, the
 * string must be freed with g_string_free.
 *
 * This is a version of get_column_header() which puts the header on one line
 * replacing the new lines.
 *
 */
GString *get_column_header_one_line(gint col)
{
  GString *header;
  GRegex *regex;
  gchar *pattern="\\n";
  gchar *newstr;

  header = get_column_header(col);

#if MAEMO_VERSION_MAJOR == 5
  regex = g_regex_new(pattern, G_REGEX_DOTALL, 0, NULL);
  newstr = g_regex_replace(regex, header->str, -1, 1, " ", 0, NULL);
  g_string_assign(header, newstr);
  g_free(newstr);  
  g_regex_unref(regex);
#else
  // MAEMO_VERSION_MAJOR has too old glib, doing nothing
#endif

  return header;
}

void update_record_model(AppUIData *pui)
{
  GtkTreeModel        *model;

  ui_show_banner(_("Updating, please wait"));
  /* Update the UI */
  while (gtk_events_pending())
    gtk_main_iteration();

  carchanged=FALSE;

  model = create_and_fill_model();
  gtk_tree_view_set_model(GTK_TREE_VIEW(pui->view),model);

  update_column_headers(pui->view);
  update_totalkm(pui);
}

void sort_order_ascending_callback( GtkAction * action, AppUIData *pui )
{
  GtkTreeModel *sortable;

  /* Can be called before view ready, so test it */
  if (pui->view) {
    sortable=gtk_tree_view_get_model(GTK_TREE_VIEW(pui->view));
    maintablesortorder = GTK_SORT_ASCENDING;
    maintablesorted=TRUE;
    gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(sortable), maintablesortcol, maintablesortorder);
  }
}

void sort_order_descending_callback( GtkAction * action, AppUIData *pui )
{
  GtkTreeModel *sortable;

  /* Can be called before view ready, so test it */
  if (pui->view) {
    sortable=gtk_tree_view_get_model(GTK_TREE_VIEW(pui->view));
    maintablesortorder = GTK_SORT_DESCENDING;
    maintablesorted=TRUE;
    gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(sortable), maintablesortcol, maintablesortorder);
  }
}

void callback_select_sortcolumn(GtkWidget *widget, AppUIData *pui )
{
  GtkTreeModel *sortable;
  gint col;

  // Might be called before view is ready
  if (pui->view) {
#if MAEMO_VERSION_MAJOR == 5
    HildonTouchSelector *selector;
    selector = hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(widget));
    col = hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(selector), 0);
#else
    col = gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
#endif
    maintablesorted=TRUE;
    maintablesortcol = col;
    sortable=gtk_tree_view_get_model(GTK_TREE_VIEW(pui->view));
    gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(sortable), maintablesortcol, maintablesortorder);
  }
}


GtkTreeModel *create_and_fill_model (void)
{
  GtkListStore  *store;
  GtkTreeIter    iter;
  GtkTreeModel        *filter;
  GtkTreeSortable     *sortable;
  char *date;
  const unsigned char* dbdate;
  double km;
  char notes[ENTRYNOTESMAX];
  gchar *convnotes;
  double fill;
  double trip;
  double price;
  double ppl;
  double ppt;
  double service;
  double oil;
  double tires;
  double consum;
  const unsigned char* dbnotes;
  sqlite_int64 id;

  if (db_open()) {
    store = gtk_list_store_new (NUM_COLS, G_TYPE_STRING, 
				G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE,
				G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE,
				G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE,
				G_TYPE_STRING, 
				GDK_TYPE_PIXBUF, G_TYPE_STRING, 
				G_TYPE_INT64, G_TYPE_BOOLEAN);

    if (DB_OK == db_init_record_query(currentcar)) {
      while(DB_ROW == db_step_record_query_data(&dbdate, &km, &trip, &fill, &consum, &price,
						&ppl, &ppt, &service, &oil, &tires, &dbnotes, &id)) {
	convdate(&date, datefmtstr[curunit.dateformat], "%Y-%m-%d", dbdate);

	/* Convert notes to utf8 (earlier versions might have imported wrongly encoded data */
	g_snprintf(notes, ENTRYNOTESMAX, "%s", dbnotes);
	if (!g_utf8_validate(notes, -1, NULL))
	  convnotes=g_convert(notes, -1, "UTF-8", "ISO-8859-15", NULL, NULL, NULL);
	else
	  convnotes=g_strndup(notes, strlen(notes));

	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    COL_DAY, date,
			    COL_KM, SIlength2user(km),
			    COL_TRIP, SIlength2user(trip),
			    COL_FILL, SIvolume2user(fill),
			    COL_CONSUM, doubleornothing(SIconsumption2user(consum)),
			    COL_PRICE, price,
			    COL_PRICEPERTRIP, doubleornothing(SIpricepertrip2user(ppt)),
			    COL_PRICEPERLITRE, doubleornothing(SIppl2user(ppl)),
			    COL_SERVICE, service,
			    COL_OIL, oil,
			    COL_TIRES, tires,
			    /* 			COL_INSURANCE, sqlite3_column_double(ppStmtRecords,5), */
			    /* 			COL_OTHER, sqlite3_column_double(ppStmtRecords,5), */
			    COL_CO2EMISSION, SIemission2user(calc_co2_emission(consum,currentcar)),
			    COL_NOTES, convnotes,
			    COL_ID, id,
			    COL_VISIBLE, TRUE,
			    -1);
	g_free(convnotes);
	
	free(date);

      }
    }

    db_reset_record_query();

    filter = gtk_tree_model_filter_new(GTK_TREE_MODEL(store), NULL);
/*     gtk_tree_model_filter_set_visible_column(GTK_TREE_MODEL_FILTER(filter), */
/* 					     COL_VISIBLE); */

    gtk_tree_model_filter_set_visible_func(GTK_TREE_MODEL_FILTER(filter),
					   visible_func,
                                           (gpointer)&filterqueue,
					   NULL);
    sortable = GTK_TREE_SORTABLE(gtk_tree_model_sort_new_with_model(filter));

    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_DAY, 
				    sort_iter_compare_func,
				    GINT_TO_POINTER(COL_DAY), NULL);

    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_TRIP,
				    sort_iter_compare_func,
				    GINT_TO_POINTER(COL_TRIP), NULL);

    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_CONSUM,
				    sort_iter_compare_func,
				    GINT_TO_POINTER(COL_CONSUM), NULL);

    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_PRICEPERTRIP,
				    sort_iter_compare_func,
				    GINT_TO_POINTER(COL_PRICEPERTRIP), NULL);

    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_PRICEPERLITRE,
				    sort_iter_compare_func,
				    GINT_TO_POINTER(COL_PRICEPERLITRE), NULL);

    /* set initial sort order */
    gtk_tree_sortable_set_sort_column_id(sortable, maintablesortcol, maintablesortorder);

    g_signal_connect(sortable, "sort-column-changed",
		     G_CALLBACK(sort_column_changed), &ui);

    return GTK_TREE_MODEL (sortable);
  }
  else
    return NULL;
}

GtkWidget *create_view_and_model (void)
{
  GtkWidget *view;
/*   static data_func_t dfdata1; */
  static data_func_t dfdata2;
  static data_func_t dfdata3;
  static data_func_t dfdata4;
  static data_func_t dfdata5;
  static data_func_t dfdata6;
  static data_func_t dfdata7;
  static data_func_t dfdata8;
  static data_func_t dfdata9;
  static data_func_t dfdata10;
  static data_func_t dfdata11;
  static data_func_t dfdata12;

  GtkTreeViewColumn   *col;
  GtkCellRenderer     *renderer;
  GtkTreeModel        *model;
  GtkTreeSelection    *select;

  GtkWidget           *label;
  PangoAttrList       *attrs;
  PangoAttribute      *attr;

  /* Scale texts smaller: create pango attributes */
  attrs = pango_attr_list_new();
  attr = pango_attr_scale_new(fontscalefactors[fontsize]);
  attr->start_index = 0;
  attr->end_index = -1;
  pango_attr_list_insert(attrs, attr);

  model = create_and_fill_model ();
#if MAEMO_VERSION_MAJOR == 5
  view = hildon_gtk_tree_view_new_with_model (HILDON_UI_MODE_EDIT, 
					      model);
#else
  view = gtk_tree_view_new_with_model (model);
#endif


  /* --- Column #1 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_DAY]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_DAY);
/*   gtk_tree_view_column_add_attribute(col, renderer, "markup", COL_DAY); */
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

/*   dfdata1.col = COL_DAY; */
/*   strncpy(dfdata1.format,"%.0f",DFFORMATLEN); */
/*   gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata1, NULL); */
/* /\*   gtk_tree_view_column_set_cell_data_func(col, renderer, new_cell_data_func, NULL, NULL); *\/ */

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_DAY);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_DAY), dispcol & (1<<COL_DAY) && (uistyle==UICLASSIC));

  /* --- Column #2 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_KM]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_KM);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  dfdata2.col = COL_KM;
  strncpy(dfdata2.format,"%.0f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata2, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_KM);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_KM), dispcol & (1<<COL_KM) && (uistyle==UICLASSIC));

  /* --- Column #3 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_TRIP]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_TRIP);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  dfdata3.col = COL_TRIP;
  strncpy(dfdata3.format,"%.1f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata3, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_TRIP);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_TRIP), dispcol & (1<<COL_TRIP) && (uistyle==UICLASSIC));

  /* --- Column #4 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_FILL]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_FILL);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_FILL);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_FILL), dispcol & (1<<COL_FILL) && (uistyle==UICLASSIC));

  dfdata4.col = COL_FILL;
  strncpy(dfdata4.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata4, NULL);

  /* --- Column #5 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_CONSUM]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_CONSUM);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_CONSUM);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_CONSUM), dispcol & (1<<COL_CONSUM) && (uistyle==UICLASSIC));

  dfdata5.col = COL_CONSUM;
  strncpy(dfdata5.format,"%.1f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata5, NULL);


  /* --- Column #6 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_PRICE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_PRICE);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_PRICE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_PRICE), dispcol & (1<<COL_PRICE) && (uistyle==UICLASSIC));

  dfdata6.col = COL_PRICE;
  strncpy(dfdata6.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata6, NULL);

  /* --- Column #7 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_PRICEPERTRIP]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_PRICEPERTRIP);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_PRICEPERTRIP);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_PRICEPERTRIP), dispcol & (1<<COL_PRICEPERTRIP) && (uistyle==UICLASSIC));

  dfdata7.col = COL_PRICEPERTRIP;
  strncpy(dfdata7.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata7, NULL);

  /* --- Column #8 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_PRICEPERLITRE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_PRICEPERLITRE);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_PRICEPERLITRE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_PRICEPERLITRE), dispcol & (1<<COL_PRICEPERLITRE) && (uistyle==UICLASSIC));

  dfdata8.col = COL_PRICEPERLITRE;
  strncpy(dfdata8.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata8, NULL);

  /* --- Column #9 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_SERVICE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_SERVICE);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_SERVICE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_SERVICE), dispcol & (1<<COL_SERVICE) && (uistyle==UICLASSIC));

  dfdata9.col = COL_SERVICE;
  strncpy(dfdata9.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata9, NULL);

  /* --- Column #10 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_OIL]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_OIL);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_OIL);

  dfdata10.col = COL_OIL;
  strncpy(dfdata10.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata10, NULL);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_OIL), dispcol & (1<<COL_OIL) && (uistyle==UICLASSIC));

  /* --- Column #11 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_TIRES]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_TIRES);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_TIRES);

  dfdata11.col = COL_TIRES;
  strncpy(dfdata11.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata11, NULL);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_TIRES), dispcol & (1<<COL_TIRES) && (uistyle==UICLASSIC));

  /* --- Column #12 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_CO2EMISSION]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_CO2EMISSION);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_CO2EMISSION);

  dfdata12.col = COL_CO2EMISSION;
  strncpy(dfdata12.format,"%.0f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata12, NULL);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_CO2EMISSION), dispcol & (1<<COL_CO2EMISSION) && (uistyle==UICLASSIC));

  /* --- Column #13 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_NOTES]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_NOTES);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_NOTES);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_NOTES), dispcol & (1<<COL_NOTES) && (uistyle==UICLASSIC));

  /* Icon column */
  /* --- Column #0 --- */

  col = gtk_tree_view_column_new();

//  gtk_tree_view_column_set_title(col, "Icon");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_pixbuf_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "pixbuf", COL_ICON);
  gtk_tree_view_column_set_cell_data_func(col, renderer, pixbuf_cell_data_func, view, NULL);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_ICON), uistyle==UIFANCY);

  /* --- Column #0.5 --- */

  col = gtk_tree_view_column_new();

/*   label=gtk_label_new(gettext(column_headers[COL_COMBINED])); */
/*   gtk_tree_view_column_set_widget(col, GTK_WIDGET(label)); */
/*   gtk_label_set_attributes(GTK_LABEL(label), attrs); */
/*   gtk_widget_show(label); */

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "markup", COL_COMBINED);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

/*   dfdata1.col = COL_DAY; */
/*   strncpy(dfdata1.format,"%.0f",DFFORMATLEN); */
  gtk_tree_view_column_set_cell_data_func(col, renderer, new_cell_data_func, NULL, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_COMBINED);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_COMBINED), uistyle==UIFANCY);

  /* Invisible column to distinguish the results */
  /* --- Column #last --- */

  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, "Id");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_ID);

/*   renderer = gtk_cell_renderer_text_new (); */
/*   gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view), */
/*                                                -1,       */
/*                                                "Id",   */
/*                                                renderer, */
/*                                                "text", COL_ID, */
/*                                                NULL); */
  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_ID), FALSE);

  /* Invisible visibility column */
  /* --- Column #last+1 --- */

  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, "Visible");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_VISIBLE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_VISIBLE), FALSE);

/*   model = create_and_fill_model (); */

  /* Change the unit dependent headers */
  update_column_headers(view);

/*   gtk_tree_view_set_model (GTK_TREE_VIEW (view), model); */
  if (uistyle==UIFANCY) {
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), FALSE);
  }
  else {
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), TRUE);
  }

  /* The tree view has acquired its own reference to the
   *  model, so we can drop ours. That way the model will
   *  be freed automatically when the tree view is destroyed */

  g_object_unref (model);

  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
  gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
  g_signal_connect(view, "row-activated",
		   G_CALLBACK (callback_recordactivated),
		   &ui);

  return view;
}
