/*
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007-2011 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "uilocations.h"
#include "uilocationview.h"

/*******************************************************************
 *
 * Private definitions
 *
 *******************************************************************/

/*******************************************************************
 *
 * Private variables
 *
 *******************************************************************/

/*******************************************************************
 *
 * Prototypes of private functions
 *
 *******************************************************************/

static
void callback_showonmap( GtkAction * action, AppUIData *pui );
static
void edit_location_response (GtkDialog *dialog, gint arg1, AppUIData *pui);
static
void create_location_editwin(AppUIData *pui, GtkWidget *dialog, gint add);
static
void callback_editlocation( GtkAction * action, AppUIData *pui );
#if PLAINGTK == 1
static void create_locationswin_menu(GtkWidget *window, GtkWidget *root, AppUIData *pui);
#else
static void create_locationswin_menu(HildonWindow *window, AppUIData *pui);
#endif

/*******************************************************************
 *
 * Private functions
 *
 *******************************************************************/

static
void callback_showonmap( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  double lat;
  double lon;

  if (db_open()) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->logview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

      gtk_tree_model_get(model, &iter, 
			 LOCATION_COL_LATITUDE, &lat, LOCATION_COL_LONGITUDE, &lon,
			 -1);

      map_show_location(lat, lon);
    }
  }
}

/**
 * \fn void edit_location_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
 * \brief Response callback for edit location dialog
 *
 * This callback alters the data edited in the edit location dialog to
 * the database.
 *
 */
static
void edit_location_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkListStore  *store;
  GtkListStore  *fullstore;
  GtkTreeIter    iter;
  GtkTreeIter    storeiter;
  GString *alias;

  gint64 id;

  if (!db_open()) {
       gtk_widget_destroy(GTK_WIDGET(dialog));
  }
  else {
    switch (arg1) {
    case GTK_RESPONSE_ACCEPT:
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->locationsview));
      if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
	gtk_tree_model_get (model, &iter, LOCATION_COL_ID, &id, -1);
      

	alias = get_entry(pui->locationsaliasentry);
	if (db_open()) {
	  if (db_update_location(id, alias->str) != id) {
	    PDEBUG("Problem updating location");
	  }
	  else {
	    get_store_and_iter(model, pui->locationsview, &iter, &store, &storeiter);
	    update_locationsview_row_data(store, &storeiter,
					  NULL, alias->str,
					  -1, -1,
					  id);
	    g_print("Location updated\n");
	  }
	}
	g_string_free(alias, TRUE);
      }
      else {
	PDEBUG("Something wrong with finding the selection\n");
      }
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    case GTK_RESPONSE_REJECT:
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    }
  }
}

/**
 * \fn void create_location_editwin(AppUIData *pui, GtkWidget *dialog, gint add)
 * \brief Creates the contents of a location adding or editing dialog
 * \param *pui Pointer to the UI data structure
 * \param *dialog Pointer to the already created dialog
 * \param add Set this to 1 if an add dialog is needed
 *
 * This function is used for creating an edit window both for adding a
 * new location and editing an existing one.
 *
 */
static
void create_location_editwin(AppUIData *pui, GtkWidget *dialog, gint add)
{
  enum
  {
    LOCATION_EDIT_ALIAS=0
  };

  GtkWidget *scrollwin;
#if MAEMO_VERSION_MAJOR == 5
  gint       row;
#else
  GtkWidget *notebook;
#endif
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *button;
  GtkWidget *entry;

  GtkListStore *store;

  gchar *labels[] = {gettext_noop("Alias")};

#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHT1);

  table = gtk_table_new(3, 2, FALSE);

  row=0;

  /* Place (uneditable) */
  row++;
  pui->locationsplacelabel=gtk_label_new(NULL);
  gtk_label_set_ellipsize(GTK_LABEL(pui->locationsplacelabel), PANGO_ELLIPSIZE_END);

  gtk_table_attach(GTK_TABLE(table), pui->locationsplacelabel, 0, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->locationsplacelabel);

  /* Latitude & longitude (uneditable) */
  row++;
  pui->locationslatlonlabel=gtk_label_new(NULL);
  gtk_label_set_ellipsize(GTK_LABEL(pui->locationslatlonlabel), PANGO_ELLIPSIZE_END);

  gtk_table_attach(GTK_TABLE(table), pui->locationslatlonlabel, 0, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->locationslatlonlabel);

  /* Alias entry */
  row++;
  label=gtk_label_new(gettext(labels[LOCATION_EDIT_ALIAS]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);
  pui->locationsaliasentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_table_attach(GTK_TABLE(table), pui->locationsaliasentry, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->locationsaliasentry);

  gtk_widget_show(table);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG (dialog)->vbox), scrollwin, TRUE, TRUE, 0);
  gtk_widget_show(scrollwin);
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(scrollwin),table);

#else /* MAEMO_VERSION_MAJOR != 5 */

/** \todo: Maemo 4 version of location editing window */

#endif

}

/**
 * \fn void callback_editlocation( GtkAction * action, AppUIData *pui )
 * \brief Callback for editing a record
 *
 * This callback creates the location editing dialog
 *
 * \todo This should be changed so that fuelpad.c has a function for
 * obtaining the record data (i.e. sqlite would not be used here)
 *
 */
static
void callback_editlocation( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  GtkWidget *label;
  gchar *header[] = {gettext_noop("Edit a location")};

  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;

  gchar *labeltext;
  const gchar *markupplace="<span size=\"medium\">%s</span>";
  const gchar *markuplatlon="<span size=\"small\">lat %f, lon %f</span>";

  char* place;
  char* alias;
  double latitude;
  double longitude;
  GString *buff;

  if (db_open()) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->locationsview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
      dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					   GTK_WINDOW(pui->app->locationswindow),
					   GTK_DIALOG_MODAL,
					   GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					   GTK_STOCK_CANCEL,
					   GTK_RESPONSE_REJECT,
					   NULL);
      create_location_editwin(pui, dialog, 0);

      gtk_tree_model_get(model, &iter, 
			 LOCATION_COL_LATITUDE, &latitude, LOCATION_COL_LONGITUDE, &longitude,
			 LOCATION_COL_PLACE, &place, LOCATION_COL_ALIAS, &alias,
		     -1);

      gtk_label_set_markup(pui->locationsplacelabel,
			   g_markup_printf_escaped(markupplace, place));
      gtk_label_set_markup(pui->locationslatlonlabel,
			   g_markup_printf_escaped(markuplatlon, latitude, longitude));

      gtk_entry_set_text(GTK_ENTRY(pui->locationsaliasentry), alias);

      g_free(place);
      g_free(alias);

#if LIBHELP == 1
	  help_dialog_help_enable(GTK_DIALOG(dialog),
					 HELP_ID_ADDRECORD,
					 pui->app->osso);
#endif

	  g_signal_connect (dialog, "response",
			    G_CALLBACK (edit_location_response),
			    &ui);
    }
    else {
      dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					   GTK_WINDOW(pui->app->drivinglogwindow),
					   GTK_DIALOG_MODAL,
					   GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					   NULL);

      label=gtk_label_new(_("Select a location first"));
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 5);
      gtk_widget_show(label);

      g_signal_connect_swapped (dialog, "response",
				G_CALLBACK (gtk_widget_destroy),
				dialog);
    }

  }
  else {
    dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					 GTK_WINDOW(pui->app->drivinglogwindow),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 NULL);

    label=gtk_label_new(_("Can't access database - editing records not possible"));
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 5);
    gtk_widget_show(label);

    /* Destroy the dialog when the user responds to it (e.g. clicks a button) */
    g_signal_connect_swapped (dialog, "response",
			      G_CALLBACK (gtk_widget_destroy),
			      dialog);
  }

  gtk_widget_show(dialog);
}

/**
 * \fn void callback_deletelocation( GtkAction * action, AppUIData *pui )
 * \brief Callback for deleting a location
 *
 * This callback deletes the currently selected location. The user is
 * asked for a confirmation.
 *
 */
static
void callback_deletelocation( GtkAction * action, AppUIData *pui )
{
  GtkWidget *confirmdlg;
  GtkWidget *separator;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeIter storeiter;
  GtkListStore *store;
  GtkTreeModel *model;
  gint64 id;
  gint confirm;

  if (db_open()) { /* delete */

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->locationsview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
      gtk_tree_model_get (model, &iter, LOCATION_COL_ID, &id, -1);

#if PLAINGTK == 0
      confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->locationswindow), 
						_("Are you sure you want to delete the selected location"));
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->locationswindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_OK_CANCEL,
					  _("Are you sure you want to delete the selected location"));
      gtk_widget_show(confirmdlg);
#endif

      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));

      if(confirm == GTK_RESPONSE_OK) {

	if (DB_DONE != db_delete_location(id)) {
	  PDEBUG("Problem deleting the location");
	}
	else { /* remove the record if succesfully removed from db */

	  get_store_and_iter(model, pui->locationsview, &iter, &store, &storeiter);
	  gtk_list_store_remove(store, &storeiter);

	}
      } /* if (confirm == GTK_RESPONSE_OK) */
    }
    else
      PDEBUG("Something wrong with finding the selection\n");
  }
  else { /* nothing to delete */
    PDEBUG("Database not open, can't delete anything\n");
  }
}

#if PLAINGTK == 1
static void create_locationswin_menu(GtkWidget *window, GtkWidget *root, AppUIData *pui)
#else
static void create_locationswin_menu(HildonWindow *window, AppUIData *pui)
#endif
{
#if MAEMO_VERSION_MAJOR == 5
  HildonAppMenu *main_menu;
#else
  GtkWidget *main_menu;
#endif
  GtkWidget *item_editlocation;
  GtkWidget *item_deletelocation;
  GtkWidget *item_showonmap;
  GtkWidget *item_startgps;
  GtkWidget *item_stopgps;
#if PLAINGTK == 1
  /** \todo Locations menus for PLAINGTK */
#endif

#if MAEMO_VERSION_MAJOR == 5
  GtkWidget * button;

  main_menu = HILDON_APP_MENU (hildon_app_menu_new ());

  item_editlocation = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					      _("Edit location"),
					      NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_editlocation));

  item_deletelocation = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					      _("Delete location"),
					      NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_deletelocation));

  item_showonmap = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					      _("Show on map"),
					      NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_showonmap));

  g_signal_connect(G_OBJECT(item_editlocation), "clicked",
		   G_CALLBACK(callback_editlocation), pui);
  g_signal_connect(G_OBJECT(item_deletelocation), "clicked",
		   G_CALLBACK(callback_deletelocation), pui);
  g_signal_connect(G_OBJECT(item_showonmap), "clicked",
		   G_CALLBACK(callback_showonmap), pui);

  hildon_window_set_app_menu(HILDON_WINDOW(window), main_menu);

#else

  /** \todo: Driving main menus for Maemo 4 */

#endif

  /* Make all menu widgets visible */
  gtk_widget_show_all(GTK_WIDGET(main_menu));
}

/*******************************************************************
 *
 * Public functions
 *
 *******************************************************************/

void callback_editlocations( GtkAction * action, AppUIData *pui )
{
  GtkWidget *vbox;
  GtkWidget *table;
  GtkWidget *scrollwin;

#if PLAINGTK == 1
  pui->app->locationswindow=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(pui->app->locationswindow), "Locations");
  gtk_widget_set_usize(pui->app->locationswindow,800,400);
  gtk_container_border_width(GTK_CONTAINER(pui->app->locationswindow),0);
#else
#if MAEMO_VERSION_MAJOR == 5
  pui->app->locationswindow = hildon_stackable_window_new();
#else
  pui->app->locationswindow = HILDON_WINDOW(hildon_window_new());
#endif
  hildon_program_add_window(pui->app->program, pui->app->locationswindow);
#endif

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (pui->app->locationswindow), vbox);

#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  g_object_set(G_OBJECT(scrollwin), "mov-mode", HILDON_MOVEMENT_MODE_BOTH, NULL);
#else
  scrollwin = gtk_scrolled_window_new(NULL, NULL);
#endif
  gtk_box_pack_start (GTK_BOX(vbox), scrollwin, TRUE, TRUE, 0);

  pui->locationsview = create_locationview_and_model ();
  gtk_container_add (GTK_CONTAINER (scrollwin), pui->locationsview);
  gtk_widget_show(pui->locationsview);
  gtk_widget_show(scrollwin);

  gtk_widget_show_all(GTK_WIDGET(vbox));

#if PLAINGTK == 1
  create_locationswin_menu(pui->app->locationswindow, vbox, pui);
#else
  create_locationswin_menu(pui->app->locationswindow, pui);
#endif

  gtk_widget_show_all(GTK_WIDGET(pui->app->locationswindow));

}
