/****************************************************************************
**
**    Friendly FM - The Friendly FM Widget
**    Copyright (C) 2011 Percy Leonhardt
**
**    This program is free software: you can redistribute it and/or modify
**    it under the terms of the GNU General Public License as published by
**    the Free Software Foundation, either version 3 of the License, or
**    (at your option) any later version.
**
**    This program is distributed in the hope that it will be useful,
**    but WITHOUT ANY WARRANTY; without even the implied warranty of
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**    GNU General Public License for more details.
**
**    You should have received a copy of the GNU General Public License
**    along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
****************************************************************************/

#include "ffmwidget.h"

#include <QtDBus/QDBusReply>

#define BG_PATH           "/etc/hildon/theme/images/"
#define TOP_BG            BG_PATH "MediaPlayerAppletHead.png"
#define LEFT_BG           BG_PATH "MediaPlayerAppletPrevious.png"
#define LEFT_PRESSED_BG   BG_PATH "MediaPlayerAppletPreviousPressed.png"
#define MIDDLE_BG         BG_PATH "MediaPlayerAppletStopPlay.png"
#define MIDDLE_PRESSED_BG BG_PATH "MediaPlayerAppletStopPlayPressed.png"
#define RIGHT_BG          BG_PATH "MediaPlayerAppletNext.png"
#define RIGHT_PRESSED_BG  BG_PATH "MediaPlayerAppletNextPressed.png"

#define ICON32_PATH    "/usr/share/icons/hicolor/32x32/apps/"
#define ICON40_PATH    "/usr/share/icons/hicolor/40x40/apps/"
#define ICON48_PATH    "/usr/share/icons/hicolor/48x48/apps/"
#define LEFT_ICON      ICON32_PATH "ffm_dec.png"
#define RIGHT_ICON     ICON32_PATH "ffm_inc.png"
#define MIDDLE_ICON    ICON40_PATH "ffm_off.png"
#define MIDDLE_ON_ICON ICON40_PATH "ffm_on.png"
#define BT_ICON        ICON48_PATH "ffm_bt.png"
#define BT_ON_ICON     ICON48_PATH "ffm_bt_on.png"
#define TM_ICON        ICON48_PATH "ffm_tm.png"
#define TM_ON_ICON     ICON48_PATH "ffm_tm_on.png"


FFMWidget::FFMWidget()
    : QWidget()
    , mBtOn(false)
    , mTmOn(false)
    , mLeftButton(LEFT_BG, LEFT_PRESSED_BG, LEFT_ICON, this)
    , mRightButton(RIGHT_BG, RIGHT_PRESSED_BG, RIGHT_ICON, this)
    , mMiddleButton(MIDDLE_BG, MIDDLE_PRESSED_BG, MIDDLE_ICON, MIDDLE_ON_ICON, this)
    , mTopButton(TOP_BG, "", BT_ICON, BT_ON_ICON, TM_ICON, TM_ON_ICON, this)
    , mSettingsDlg(0)
    , mSettings("friendlyfm", "friendlyfm")
    , mSystemBus(QDBusConnection::systemBus())
{
    mFMIface = new QDBusInterface("com.nokia.FMTx", "/com/nokia/fmtx/default", "org.freedesktop.DBus.Properties", mSystemBus, this);
    QDBusInterface btManager("org.bluez", "/", "org.bluez.Manager", mSystemBus);
    QDBusObjectPath path = qvariant_cast<QDBusObjectPath>(btManager.call("DefaultAdapter").arguments().first());
    mBTIface = new QDBusInterface("org.bluez", path.path(), "org.bluez.Adapter", mSystemBus, this);

    QGridLayout *grid = new QGridLayout();
    grid->addWidget(&mTopButton, 0, 0, 1, 3);
    grid->addWidget(&mLeftButton, 1, 0);
    grid->addWidget(&mMiddleButton, 1, 1);
    grid->addWidget(&mRightButton, 1, 2);
    grid->setSpacing(0);
    grid->setContentsMargins(0, 0, 0, 0);
    setLayout(grid);

    setAttribute(Qt::WA_TranslucentBackground);

    connect(&mLeftButton, SIGNAL(clicked()), this, SLOT(decrementFrequency()));
    connect(&mMiddleButton, SIGNAL(clicked()), this, SLOT(switchFMTransmitter()));
    connect(&mRightButton, SIGNAL(clicked()), this, SLOT(incrementFrequency()));
    connect(&mTopButton, SIGNAL(btClicked()), this, SLOT(switchBTRadio()));
    connect(&mTopButton, SIGNAL(tmClicked()), this, SLOT(switchPreset()));

    mSystemBus.connect("com.nokia.FMTx", "/com/nokia/fmtx/default", "com.nokia.FMTx.Device", "Changed", this, SLOT(fmPropertyChanged()));
    mSystemBus.connect("org.bluez", path.path(), "org.bluez.Adapter", "PropertyChanged", this, SLOT(btPropertyChanged(QString, QDBusVariant)));

    QDBusVariant dbv = qvariant_cast<QDBusVariant>(mFMIface->call("Get", "com.nokia.FMTx.Device", "freq_step").arguments().first());
    mFreqStep = dbv.variant().toInt();
    dbv = qvariant_cast<QDBusVariant>(mFMIface->call("Get", "com.nokia.FMTx.Device", "freq_min").arguments().first());
    mFreqMin = dbv.variant().toInt();
    dbv = qvariant_cast<QDBusVariant>(mFMIface->call("Get", "com.nokia.FMTx.Device", "freq_max").arguments().first());
    mFreqMax = dbv.variant().toInt();
    dbv = qvariant_cast<QDBusVariant>(mFMIface->call("Get", "com.nokia.FMTx.Device", "frequency").arguments().first());
    mFreqCurrent = dbv.variant().toInt();
    mTopButton.setFrequency(mFreqCurrent);

    mFMBoostInstalled = QFile::exists("/sbin/fm-boost");
    mFMBoostActivated = mSettings.value("FMBoostActivated", false).toBool();
    mPreset1 = mSettings.value("Preset1", mFreqMin).toUInt();
    mPreset2 = mSettings.value("Preset2", mFreqMin).toUInt();
}

FFMWidget::~FFMWidget()
{
    if (mSettingsDlg)
    {
        delete mSettingsDlg;
    }
}

QSize FFMWidget::sizeHint() const
{
    return QSize(338, 148);
}

void FFMWidget::showSettingsDialog()
{
    if (!mSettingsDlg)
    {
        mSettingsDlg = new FFMSettingsDialog(this);
    }

    // could have changed in the meantime
    mFMBoostInstalled = QFile::exists("/sbin/fm-boost");

    mSettingsDlg->updateUi();
    mSettingsDlg->exec();

    mFMBoostActivated = mSettingsDlg->getBoostCheckBoxState();
    mSettings.setValue("FMBoostActivated", mFMBoostActivated);
    mPreset1 = mSettingsDlg->getPreset1SpinBoxValue();
    mSettings.setValue("Preset1", mPreset1);
    mPreset2 = mSettingsDlg->getPreset2SpinBoxValue();
    mSettings.setValue("Preset2", mPreset2);
}

void FFMWidget::incrementFrequency()
{
    uint newFreq = mFreqCurrent + mFreqStep;
    if (newFreq > mFreqMax)
    {
        newFreq = mFreqMin;
    }
    mFMIface->call("Set", "com.nokia.FMTx.Device", "frequency", QVariant::fromValue(QDBusVariant(newFreq)));
}

void FFMWidget::decrementFrequency()
{
    uint newFreq = mFreqCurrent - mFreqStep;
    if (newFreq < mFreqMin)
    {
        newFreq = mFreqMax;
    }
    mFMIface->call("Set", "com.nokia.FMTx.Device", "frequency", QVariant::fromValue(QDBusVariant(newFreq)));
}

void FFMWidget::switchFMTransmitter()
{
    if (mTmOn)
    {
        mFMIface->call("Set", "com.nokia.FMTx.Device", "state", QVariant::fromValue(QDBusVariant("disabled")));
    }
    else
    {
        mFMIface->call("Set", "com.nokia.FMTx.Device", "state", QVariant::fromValue(QDBusVariant("enabled")));
    }
}

void FFMWidget::switchBTRadio()
{
    if (mBtOn)
    {
        mBTIface->call("SetProperty", "Powered", QVariant::fromValue(QDBusVariant(false)));
    }
    else
    {
        mBTIface->call("SetProperty", "Powered", QVariant::fromValue(QDBusVariant(true)));
    }
}

void FFMWidget::switchPreset()
{
    if (mFreqCurrent == mPreset1)
    {
        mFMIface->call("Set", "com.nokia.FMTx.Device", "frequency", QVariant::fromValue(QDBusVariant(mPreset2)));
    }
    else
    {
        mFMIface->call("Set", "com.nokia.FMTx.Device", "frequency", QVariant::fromValue(QDBusVariant(mPreset1)));
    }
}

void FFMWidget::fmPropertyChanged()
{
    QDBusVariant dbv = qvariant_cast<QDBusVariant>(mFMIface->call("Get", "com.nokia.FMTx.Device", "state").arguments().first());
    QString state = dbv.variant().toString();

    if (state == "enabled" && !mTmOn)
    {
        mTmOn = true;
        if (mFMBoostInstalled && mFMBoostActivated)
        {
            QString program("/usr/bin/sudo");
            QStringList args("/sbin/fm-boost");
            QProcess::execute(program, args);
        }
    }
    else if (state == "disabled" && mTmOn)
    {
        mTmOn = false;
    }
    mTopButton.setTmOn(mTmOn);
    mMiddleButton.setOn(mTmOn);

    dbv = qvariant_cast<QDBusVariant>(mFMIface->call("Get", "com.nokia.FMTx.Device", "frequency").arguments().first());
    uint freq = dbv.variant().toInt();

    if (freq != mFreqCurrent)
    {
        mFreqCurrent = freq;
        mTopButton.setFrequency(mFreqCurrent);
    }
}

void FFMWidget::btPropertyChanged(QString property, QDBusVariant value)
{
    if (property == "Powered")
    {
        mBtOn = value.variant().toBool();
        mTopButton.setBtOn(mBtOn);
    }
}

