#include "mainwindow.h"
#include "ui_mainwindow.h"

#include "about.h"
#include "help.h"
#include "instructions.h"
#include "settings.h"

#include <QProcess>
#include <QtCore/QCoreApplication>
#include <QDesktopServices>
#include <QShortcut>
#include <QUrl>
#include <QFile>
#include <QFileInfo>
#include <QSettings>
#include <QString>
#include <QTextStream>
#include <QMaemo5InformationBox>
#include <QtMaemo5/QtMaemo5>

MainWindow::MainWindow(QWidget *parent):
    QMainWindow(parent), ui(new Ui::MainWindow),
      about(0),
      help(0),
      instructions(0),
      settingspage(0),
      m_process(new QProcess(this))

{
    ui->setupUi(this);

    //Set Window
    this->setWindowTitle("FlopSwap");
    this->setAttribute(Qt::WA_Maemo5StackedWindow);

    //Set Button Text
    ui->pushButtonSwap->setText(tr("Fresh Swap"));
    ui->pushButtonEject->setText(tr("Swap to Device"));
    ui->pushButtonGet->setText(tr("Show Current"));
    ui->pushButtonSwapUsage->setText(tr("Check Status"));

    // Set Button Icons
    ui->pushButtonSwap->setIcon(QPixmap("/opt/flopswap/icons/refresh_icon&32.png"));
    ui->pushButtonEject->setIcon(QPixmap("/opt/flopswap/icons/eject_icon&32.png"));
    ui->pushButtonGet->setIcon(QPixmap("/opt/flopswap/icons/info_icon&32.png"));
    ui->pushButtonSwapUsage->setIcon(QPixmap("/opt/flopswap/icons/checkbox_checked_icon&32.png"));

    //Set Action Text
    ui->actionSettings->setText(tr("Settings"));
    ui->actionReadme->setText(tr("Quick Guide"));
    ui->actionHelp->setText(tr("Help"));
    ui->actionAbout->setText(tr("About"));

    //Set Actions
    about = new About(this);
    help = new Help(this);
    instructions = new Instructions(this);
    settingspage = new Settings(this);

    //For Connecting Keyboard Buttons
    connectSignals();

    //Initial Text - Show Once
    initialshow();

    //Initial QSettings for Spin Box
    initialpsb();

    //Check for existance of /tmp/flopswapint for button disable
    disableStatus();

    //Check for existance of /tmp/flopswapref for progress indication
    swappingActive();

    //Check QSettings for Startup Check option
    if (settingspage->startupCheck())
    {
        //If true click immediately after initialization
        ui->pushButtonSwapUsage->click();
    }

    //GUI Blocking on Swap
    connect(m_process, SIGNAL(started()), this, SLOT(disableButtons()));
    connect(m_process, SIGNAL(finished(int,QProcess::ExitStatus)), this, SLOT(enableButtons()));

}

MainWindow::~MainWindow()
{
    initialseen();

    if (about) delete (about);
    if (help) delete (help);
    if (instructions) delete (instructions);
    if (settingspage) delete (settingspage);

    delete ui;
}

void MainWindow::setOrientation(ScreenOrientation orientation)
{
#if defined(Q_OS_SYMBIAN)
    // If the version of Qt on the device is < 4.7.2, that attribute won't work
    if (orientation != ScreenOrientationAuto) {
        const QStringList v = QString::fromAscii(qVersion()).split(QLatin1Char('.'));
        if (v.count() == 3 && (v.at(0).toInt() << 16 | v.at(1).toInt() << 8 | v.at(2).toInt()) < 0x040702) {
            qWarning("Screen orientation locking only supported with Qt 4.7.2 and above");
            return;
        }
    }
#endif // Q_OS_SYMBIAN

    Qt::WidgetAttribute attribute;
    switch (orientation) {
#if QT_VERSION < 0x040702
    // Qt < 4.7.2 does not yet have the Qt::WA_*Orientation attributes
    case ScreenOrientationLockPortrait:
        attribute = static_cast<Qt::WidgetAttribute>(128);
        break;
    case ScreenOrientationLockLandscape:
        attribute = static_cast<Qt::WidgetAttribute>(129);
        break;
    default:
    case ScreenOrientationAuto:
        attribute = static_cast<Qt::WidgetAttribute>(130);
        break;
#else // QT_VERSION < 0x040702
    case ScreenOrientationLockPortrait:
        attribute = Qt::WA_LockPortraitOrientation;
        break;
    case ScreenOrientationLockLandscape:
        attribute = Qt::WA_LockLandscapeOrientation;
        break;
    default:
    case ScreenOrientationAuto:
        attribute = Qt::WA_AutoOrientation;
        break;
#endif // QT_VERSION < 0x040702
    };
    setAttribute(attribute, true);
}

void MainWindow::showExpanded()
{
#if defined(Q_OS_SYMBIAN) || defined(Q_WS_SIMULATOR)
    showFullScreen();
#elif defined(Q_WS_MAEMO_5)
    showMaximized();
#else
    show();
#endif
}

   //Keyboard Button Signals - remember to include QShortcut
void MainWindow::connectSignals()
{

    //Keyboard Key Signals
    connect(new QShortcut(QKeySequence(Qt::Key_Q), this), SIGNAL(activated()), this, SLOT(on_pushButtonSwap_clicked()));
    connect(new QShortcut(QKeySequence(Qt::Key_W), this), SIGNAL(activated()), this, SLOT(on_pushButtonEject_clicked()));
    connect(new QShortcut(QKeySequence(Qt::Key_A), this), SIGNAL(activated()), this, SLOT(on_pushButtonGet_clicked()));
    connect(new QShortcut(QKeySequence(Qt::Key_S), this), SIGNAL(activated()), this, SLOT(on_pushButtonSwapUsage_clicked()));
}

//Setup QSettings for Spin Box on First Run
void MainWindow::initialpsb()
{
     QSettings settings("flopswap", "flopswap");
     if (!settings.contains("PercentForDialog"))
     {
       settings.setValue("PercentForDialog", 94);
       settings.value("PercentForDialog").toInt();
     }
}

//Initial First Run Dialog Text
void MainWindow::initialshow()
{
   QSettings settings("flopswap", "flopswap");

   if (!settings.contains("InitialWarningShown"))
       {
       //Run swapswitch QProcess to draw pushbutton status correctly on first installation
       on_pushButtonGet_clicked();
       //Dialog
        QMaemo5InformationBox::information(
          0,
          " <br> \
            <br> \
          Thank you for installing FlopSwap. <br> \
          For this application to function it requires two linux-swap spaces creating on your uSD. <br> \
          If you have not already done this, please see the Wiki shortcut located in the Help Menu for instructions. <br> \
          <br> ",
          0);
       }
}

//Initial First Run Dialog Text Seen
void MainWindow::initialseen()
{
    //Set Dialog as seen
    QSettings settings("flopswap", "flopswap");

    if (!settings.contains("InitialWarningShown"))
        {
         settings.setValue("InitialWarningShown", true);
        }
}


//Eject First Run Dialog Text
void MainWindow::ejectshow()
{
   //Dialog
   QSettings settings("flopswap", "flopswap");

   if (!settings.contains("EjectWarningShown"))
       {
        QMaemo5InformationBox::information(
          0,
          " <br> \
            <br> \
          This is the Swap to Device Button<br> \
          This button will move your swap to the device to allow you to remove the back cover and/or uSD card. <br> \
          CAUTION:- While swap is on the uSD DO NOT remove the back cover. <br> \
          This will cause the device to reboot and may cause data loss or damage. <br> \
          Please wait for the Safe to Remove Back Cover message. <br> \
            <br> ",
          0);
       }
}

//Eject First Run Dialog Seen
void MainWindow::ejectseen()
{
    QSettings settings("flopswap", "flopswap");

    if (!settings.contains("EjectWarningShown"))
        {
         settings.setValue("EjectWarningShown", true);
        }
}

//Check for existance of /tmp/flopswapint to disable Buttons
void MainWindow::disableStatus()
{
 if ( QFileInfo("/tmp/flopswapint").exists() )
   {
        ui->pushButtonEject->setDisabled(true);
        ui->pushButtonSwapUsage->setDisabled(true);
        ui->labelStat->setText("Safe to Remove Back Cover");
   }
 else
   {
        ui->pushButtonEject->setDisabled(false);
        ui->pushButtonSwapUsage->setDisabled(false);
          if ( QFileInfo("/tmp/flopswapcur").exists() )
          {
              QFile file("/tmp/flopswapcur");
              file.open(QIODevice::ReadOnly | QIODevice::Text);
              QTextStream in(&file);
              QString line = in.readLine();
              ui->labelStat->setText(line);
          }
          else
          {
              ui->labelStat->setText("Initialized...");
          }
   }
}

//Check for existance of /tmp/flopswapref on Initial to disable Buttons
void MainWindow::swappingActive()
{
    if ( QFileInfo("/tmp/flopswapref").exists() )
    {
        disableButtons();
        //Loop this again in 5 seconds
        QTimer::singleShot(5000, this, SLOT(swappingActive()));
    }
    else
    {
        enableButtons();
    }
}

//Use to disable Buttons on button press
void MainWindow::disableButtons()
{
    this->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    ui->pushButtonSwap->setDisabled(true);
    ui->pushButtonEject->setDisabled(true);
    ui->pushButtonGet->setDisabled(true);
    ui->pushButtonSwapUsage->setDisabled(true);
}

//Use to enable Buttons on QProcess complete
void MainWindow::enableButtons()
{
    this->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
    ui->pushButtonSwap->setDisabled(false);
    ui->pushButtonEject->setDisabled(false);
    ui->pushButtonGet->setDisabled(false);
    ui->pushButtonSwapUsage->setDisabled(false);
    disableStatus();
}

//Buttons
void MainWindow::on_pushButtonSwap_clicked()
{ 
    m_process->start("sudo /opt/flopswap/script/swapswitch.sh");
}

void MainWindow::on_pushButtonEject_clicked()
{
    ejectshow();
    m_process->start("sudo /opt/flopswap/script/swapswitchejectsd.sh");
    ejectseen();
}

void MainWindow::on_pushButtonGet_clicked()
{
    m_process->start("sudo /opt/flopswap/script/catprocswap.sh");
}

void MainWindow::on_pushButtonSwapUsage_clicked()
{
    m_process->start("sudo /opt/flopswap/script/swapused.sh");
}

//Actions
void MainWindow::on_actionAbout_triggered()
{
    about->show();
}

void MainWindow::on_actionHelp_triggered()
{
    help->show();
}

void MainWindow::on_actionSettings_triggered()
{
    settingspage->show();
}

void MainWindow::on_actionReadme_triggered()
{
    instructions->show();
}
