#include <QDir>
#include <QFile>
#include <QLocale>
#include <QXmlQuery>
#include <QDebug>

#include "fbconfig.h"

const QString fbConfig::SYSTEM_SHARED_DIR = QString("/usr/share/FBReader");
const QString fbConfig::SYSTEM_CONFIG_DIR = QString("default");

fbConfig::fbConfig(QObject *parent) : QObject(parent), loadedData(false)
{
    QString country_code = QLocale::system().name();

    const int dotIndex = country_code.indexOf(".");
    if (dotIndex != -1) {
        country_code.truncate(dotIndex);
    }

    const int dashIndex = std::min(country_code.indexOf("_"), country_code.indexOf("-"));
    if (dashIndex == -1) {
        myLanguage = country_code;
    } else {
        myLanguage = country_code.left(dashIndex);
        myCountry = country_code.mid(dashIndex);

        if ((myLanguage == "es") && (myCountry != "ES")) {
            myCountry = "LA";
        }
    }
}

const QStringList *fbConfig::getCurrentEntries() {
    if (!loadedData) {
        loadData();
    }

    return &myCurrentEntries;
}

const QStringList fbConfig::getAvailableItems() {
    if (!loadedData) {
        loadData();
    }

    return myItemDescriptions.keys();
}

const QString fbConfig::getItemDescription(const QString item) {
    if (!loadedData) {
        loadData();
    }

    return myItemDescriptions.value(item, QString());
}

const QString fbConfig::getItemName(const QString description) {
    if (!loadedData) {
        loadData();
    }

    return myItemDescriptions.key(description, QString());
}

void fbConfig::loadData() {
    if (myResourceQuery.isEmpty() || myConfigQuery.isEmpty() || myConfigFilePath.isEmpty()) {
        return;
    }

    readItemDescriptions("en");

    if (myLanguage != "en") {
        readItemDescriptions(myLanguage);
    }
    if (!myCountry.isEmpty()) {
        readItemDescriptions(myLanguage + "_" + myCountry);
    }

    myCurrentEntries = runQuery(myConfigFilePath, myConfigQuery);

    loadedData = true;
}

void fbConfig::readItemDescriptions(const QString language) {
    QStringList res = runQuery(SYSTEM_SHARED_DIR + "/resources/" + language + ".xml", myResourceQuery);

    for (int i = 0; i < res.size(); i += 2) {
        myItemDescriptions.insert(res.at(i), res.at(i+1));
    }
}

QStringList fbConfig::runQuery(const QString file_name, const QString query_string) {
    QStringList res;

    if ((!query_string.isEmpty()) && (QFile::exists(file_name))) {
        QXmlQuery query;

        if (query.setFocus(QUrl(file_name))) {
            query.setQuery(query_string);
            if (query.isValid()) {
                query.evaluateTo(&res);
            }
        }
    }

    return res;
}

const QString fbConfig::getSystemConfigDir() {
    return SYSTEM_SHARED_DIR + "/" + SYSTEM_CONFIG_DIR;
}

const QString fbConfig::getUserConfigDir() {
    return QDir::homePath() + "/.FBReader";
}

int fbConfig::getMaxEntries() {
   return myMaxEntries;
}
