/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include <QDBusConnection>
#include <QDBusInterface>
#include "packageselector.h"
#include "ui_packageselector.h"
#include "package.h"
#include "packageview.h"
#include "repository.h"
#include "blacklistselect.h"
#include "aaptinterface.h"

PackageSelector::PackageSelector(Package* pkg, AAptInterface* apt, QWidget *parent) :
    QDialog(parent),
    ui(new Ui::PackageSelector)
{
    ui->setupUi(this);	
	iPkg = pkg;
	iAptInterface = apt;

	ui->pushButton_website->setIcon(QPixmap("/usr/share/icons/hicolor/48x48/hildon/general_web.png"));
	if( getMaemoOrgUrl(pkg)=="" && getMaemoOrgUrl(pkg->availablePackage())=="" ) {
		ui->pushButton_website->setEnabled(false);
	}
	if( pkg->isInstalled() || pkg->isMarkedForOperation() )
		ui->pushButton_blacklist->setEnabled(false);

	if( pkg->icon() && !pkg->icon()->isNull() ) {
		ui->label_appicon->setPixmap( *pkg->icon() );
	} else {
		ui->label_appicon->setPixmap( QPixmap(":/icons/icons/appdefault.png") );
	}
	this->setWindowTitle( pkg->name() );

	if( pkg->maemoDisplayName() != "" )
		ui->label_header->setText("<b>" + pkg->maemoDisplayName() + "</b>");
	else
		ui->label_header->setText("<b>" + pkg->name() + "</b>");

	updateHeader2();

	QString longtext;

	longtext += "Category: " + pkg->section() + "<br>";

	if( pkg->isInstalled() ) {
		longtext += "Installation date: ";
		if( pkg->date().isValid() )
			longtext += pkg->date().toString("yyyy-MM-dd hh:mm");
		else
			longtext += "Unknown";
		longtext += "<br>";
	}

	if( pkg->isInstalled() && pkg->installedSize() > 0 )
		longtext += QString("Size: %L1 kB<br>").arg(pkg->installedSize() );


	if( !pkg->isInstalled() ){
		longtext += "Available version <b>" + pkg->version() + "</b>";
		if( pkg->size() > 0 )
			longtext += QString(" (%L1 kB)").arg(pkg->size()/1024 );
		longtext += "<br>";
	}
	else if( pkg->isUpgradeable() ){
		longtext += "Available version <b>" + pkg->upgradeableVersion() + "</b>";
		if( pkg->size() > 0 )
			longtext += QString(" (%L1 kB)").arg(pkg->size()/1024 );
		longtext += "<br>";
	}

	if( !pkg->isInstalled() ) {
		longtext += "Available package date: ";
		if( pkg->date().isValid() )
			longtext += pkg->date().toString("yyyy-MM-dd hh:mm");
		else
			longtext += "Unknown";
		longtext += "<br>";
	} else if( pkg->isUpgradeable() ) {
		longtext += "Available package date: ";
		if( pkg->availablePackage() && pkg->availablePackage()->date().isValid() )
			longtext += pkg->availablePackage()->date().toString("yyyy-MM-dd hh:mm");
		else
			longtext += "Unknown";
		longtext += "<br>";
	}

	if( !pkg->isInstalled() )
	{
		longtext += "Repositories: ";
		if( pkg->repositories().count()>0 ) {
			for(int i=0; i<pkg->repositories().count(); i++ ) {
				if( pkg->repositories().at(i) )
					longtext += pkg->repositories().at(i)->name();
				else
					longtext += "unknown";
				if( i<pkg->repositories().count()-1 )
					longtext += ", ";
			}
			longtext += "<br>";
		} else {
			longtext += "unknown<br>";
		}
	}

	if( pkg->isUpgradeable() )
	{
		longtext += "Repositories: ";
		Package* upg_pkg = pkg->availablePackage();
		if( upg_pkg && upg_pkg->repositories().count()>0 ) {
			for(int i=0; i<upg_pkg->repositories().count(); i++ ) {
				if( upg_pkg->repositories().at(i) )
					longtext += upg_pkg->repositories().at(i)->name();
				else
					longtext += "unknown";
				if( i<upg_pkg->repositories().count()-1 )
					longtext += ", ";
			}
			longtext += "<br>";
		} else {
			longtext += "unknown<br>";
		}
	}

	longtext += "<br>";
	longtext += pkg->descShort();
	QString descLong = pkg->descLong();
	if( descLong.length()>0 ) {
		descLong.replace('\n',"<br>");
		longtext += "<br><font size=\"-1\"><br>" + descLong + "</font>";
	}

	ui->label_text->setText( longtext );

	if( pkg->isUpgradeable() )
		ui->radioInstall->setText("Upgrade");
	else
		ui->radioInstall->setText("Install");

	if( pkg->markedOperation() == Package::PkgOpNone )
	{
		if( pkg->isInstalled() && pkg->isUpgradeable() )
			ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_nop_instupgr.png"));
		else if( pkg->isInstalled() )
			ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_nop_installed.png"));
		else if( !pkg->isInstalled() )
			ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_nop_notinstalled.png"));

		ui->radioNothing->setChecked(true);
	} else if( pkg->markedOperation() == Package::PkgOpInstallUpgrade ) {
		if( pkg->isUpgradeable() )
			ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_upgrade.png"));
		else
			ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_install.png"));
		ui->radioInstall->setChecked(true);
	} else if( pkg->markedOperation() == Package::PkgOpRemove ) {
		ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_remove.png"));
		ui->radioRemove->setChecked(true);
	}

#ifndef Q_WS_MAEMO_5	// for simulator debug
	ui->radioInstall->setChecked(true);
#endif
}

PackageSelector::~PackageSelector()
{
    delete ui;
}

void PackageSelector::changeEvent(QEvent *e)
{
    QDialog::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

Package::operation PackageSelector::selectedOperation()
{
	if( ui->radioInstall->isChecked() )
		return Package::PkgOpInstallUpgrade;
	if( ui->radioRemove->isChecked() )
		return Package::PkgOpRemove;

	return Package::PkgOpNone;
}

void PackageSelector::on_pushButton_blacklist_clicked()
{
	BlacklistSelect s(iPkg, this);
	BlacklistSelect::blackList old = iPkg->blacklisted();
	if( s.exec() ) {
		updateHeader2();
		if( old != iPkg->blacklisted() ) {
			iAptInterface->removeFromBlacklist(iPkg, old);
			iAptInterface->writeBlacklist();
		}
	}
}

QString PackageSelector::getMaemoOrgUrl(Package* pkg)
{
	QString url = "";

	if( !pkg )
		return url;

	if( pkg->repositories().count() == 0 )
		return url;

	for( int i=0; i<pkg->repositories().count(); i++ )
	{
		if( pkg->repositories().at(i) && pkg->repositories().at(i)->url().startsWith("http://repository.maemo.org") )
			url = "http://maemo.org/packages/view/" + pkg->name() + "/";
	}
	return url;
}

void PackageSelector::updateHeader2()
{
	QString header2;

	if( iPkg->isInstalled() ) {
		header2 += "Installed, version <b>" + iPkg->version() + "</b>";
	} else {
		header2 += "Not installed";
	}
	header2 += "<br>";

	if( iPkg->markedOperation() == Package::PkgOpInstallUpgrade )
	{
		if( iPkg->isUpgradeable() )
			header2 += "Marked for <b>upgrade</b>";
		else
			header2 += "Marked for <b>installation</b>";
	} else if(iPkg->markedOperation() == Package::PkgOpRemove)
	{
		header2 += "Marked for <b>removal</b>";
	}

	if(iPkg->isBlacklisted()) {
		header2 = "BLACKLISTED";
		if( iPkg->blacklisted() == BlacklistSelect::BlacklistAll )
			header2 += " (all)";
		else if( iPkg->blacklisted() == BlacklistSelect::BlacklistThis )
			header2 += " (this)";
		iPkg->setMarkedForOperation(Package::PkgOpNone);
	}

	ui->radioInstall->setEnabled(true);
	ui->radioRemove->setEnabled(true);
	ui->radioInstall->show();
	ui->radioRemove->show();

	if( iPkg->isInstalled() && !iPkg->isUpgradeable() )
	{
		ui->radioInstall->setEnabled(false);
		ui->radioInstall->hide();
	} else if( !iPkg->isInstalled() ) {
		ui->radioRemove->setEnabled(false);
		ui->radioRemove->hide();
	}

	if( iPkg->isBlacklisted() ) {
		ui->radioInstall->setEnabled(false);
		ui->radioRemove->setEnabled(false);
	}

	ui->label_header2->setText(header2);
}

void PackageSelector::on_pushButton_website_clicked()
{
	QString url = getMaemoOrgUrl(iPkg);
	if( url == "" ) {
		url = getMaemoOrgUrl(iPkg->availablePackage());
		if( url == "")
			return;
	}

	QDBusConnection conn = QDBusConnection::connectToBus(QDBusConnection::SessionBus, "faster_application_manager");

	QString service = "com.nokia.osso_browser";
	QString path = "/com/nokia/osso_browser/request";
	QString method = "open_new_window";

	QDBusInterface browser(service, path, service, conn, this);
	browser.call(method, url);
}
