/*******************************************************************************

    This file is a part of Fahrplan for maemo 2009-2010

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

*/

#include "mainwindow.h"

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent), resultWindow(0), detailsWindow(0)
{
    QWidget *mainWidget = new QWidget;
    QGtkStyle *gtkStyle = new QGtkStyle();
    setCentralWidget(mainWidget);

    //About Menu Button
    QAction *aboutAct = new QAction(tr("About"), this);
    connect(aboutAct, SIGNAL(triggered()), this, SLOT(aboutButtonClicked()));
    menuBar()->addAction(aboutAct);

    //Settings Menu Button
    QAction *settingsAct = new QAction(tr("Settings"), this);
    connect(settingsAct, SIGNAL(triggered()), this, SLOT(settingsButtonClicked()));
    menuBar()->addAction(settingsAct);

    //Toggle GPS Button
    QAction *gpsAct = new QAction(tr("Toggle GPS"), this);
    connect(gpsAct, SIGNAL(triggered()), this, SLOT(gpsButtonClicked()));
    menuBar()->addAction(gpsAct);

    //Toggle GPS Button
    QAction *donateAct = new QAction(tr("Donate"), this);
    connect(donateAct, SIGNAL(triggered()), this, SLOT(donateButtonClicked()));
    menuBar()->addAction(donateAct);

    setWindowTitle("Fahrplan");

    //Main Window Layout
    QVBoxLayout *layout = new QVBoxLayout;

    //From and To and Swap Buttons
    QHBoxLayout *stationsHLayout = new QHBoxLayout;
    QVBoxLayout *stationsVLayout = new QVBoxLayout;

    QWidget *stationsVContainer  = new QWidget;
    QWidget *stationsHContainer  = new QWidget;

    stationDeparture             = new QMaemo5ValueButton(tr("Station from"));
    stationArrival               = new QMaemo5ValueButton(tr("Station to"));
    QPushButton *stationSwap     = new QPushButton("");

    stationDeparture->setValueText(tr("please select"));
    stationDeparture->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    stationArrival->setValueText(tr("please select"));
    stationArrival->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    stationSwap->setFixedWidth(50);
    stationSwap->setFixedHeight(150);
    stationSwap->setStyle(gtkStyle);

    connect(stationDeparture, SIGNAL(clicked(bool)), this, SLOT(stationDepartureClicked()));
    connect(stationArrival, SIGNAL(clicked(bool)), this, SLOT(stationArrivalClicked()));
    connect(stationSwap, SIGNAL(clicked(bool)), this, SLOT(swapClicked()));

    stationsVLayout->setMargin(1);
    stationsVLayout->addWidget(stationDeparture);
    stationsVLayout->addWidget(stationArrival);
    stationsVContainer->setLayout(stationsVLayout);

    stationsHLayout->setMargin(0);
    stationsHLayout->addWidget(stationsVContainer);
    stationsHLayout->addWidget(stationSwap);
    stationsHContainer->setLayout(stationsHLayout);

    //Date and Time Buttons
    QHBoxLayout *datetimeHLayout = new QHBoxLayout;
    QWidget     *datetimeHContainer = new QWidget;

    dateSelector  = new QMaemo5ValueButton(tr("Date"));
    dateSelector->setPickSelector(new QMaemo5DatePickSelector());
    dateSelector->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);

    timeSelector  = new QMaemo5ValueButton(tr("Time"));
    timeSelector->setPickSelector(new QMaemo5TimePickSelector());
    timeSelector->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);

    datetimeHLayout->setMargin(0);
    datetimeHLayout->setSpacing(0);
    datetimeHLayout->addWidget(dateSelector);
    datetimeHLayout->addWidget(timeSelector);
    datetimeHContainer->setLayout(datetimeHLayout);

    QHBoxLayout *optionsHLayout = new QHBoxLayout;
    QWidget     *optionsHContainer = new QWidget;

    //Departure/Arrival Selector Button
    modeButton = new QMaemo5ValueButton(tr("Use selected date/time as"));
    modeButton->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    modeButton->setValueText(tr("Departure time"));
    ModeButtonUserData *mode = new ModeButtonUserData();
    mode->mode = 1;
    modeButton->setUserData(0, mode);
    connect(modeButton, SIGNAL(clicked(bool)), this, SLOT(modeClicked()));

    //Restrict Train Types
    traintypesButton = new QMaemo5ValueButton(tr("Restrict train types"));
    traintypesButton->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);

    optionsHLayout->setMargin(0);
    optionsHLayout->setSpacing(0);
    optionsHLayout->addWidget(modeButton);
    optionsHLayout->addWidget(traintypesButton);
    optionsHContainer->setLayout(optionsHLayout);

    //Search Button
    searchButton = new QPushButton();
    searchButton->setText(tr("Search"));
    searchButton->setFixedHeight(100);
    searchButton->setStyle(gtkStyle);
    connect(searchButton, SIGNAL(clicked(bool)), this, SLOT(searchButtonClicked(bool)));

    //Add all to Layout
    layout->setMargin(1);
    layout->setStretch(0, 2);
    layout->addWidget(stationsHContainer);
    layout->addWidget(datetimeHContainer);
    layout->addWidget(optionsHContainer);
    layout->addWidget(searchButton);
    mainWidget->setLayout(layout);

    #ifdef Q_WS_MAEMO_5
        setAttribute(Qt::WA_Maemo5StackedWindow);
    #endif

    //Load Settings
    settings = new QSettings("smurfy", "fahrplan");

    //Init GPS if in settings
    locationDataSource = QGeoPositionInfoSource::createDefaultSource(this);
    locationDataSource->setPreferredPositioningMethods(locationDataSource->AllPositioningMethods);

    QObject::connect(locationDataSource, SIGNAL(positionUpdated(QGeoPositionInfo)), this, SLOT(positionUpdated(QGeoPositionInfo)));

    if (settings->value("autostartGPS", true).toBool() ==  true)
    {
        locationDataSource->startUpdates();
        locationDataSourceIsUpdating = true;
    } else
    {
        locationDataSourceIsUpdating = false;
    }

    //Prefill stations
    if ((settings->value("prefillsearchbox", true).toBool() ==  true) &&
        (settings->value("storelastsearch", true).toBool() ==  true)
        )
    {
        QStringList stationsList;
        stationsList = settings->value("lastsearch/stations", stationsList).toStringList();
        //we reverse the list to get the latest stations on the top
        stationsList = fahrplanUtils::reverseQStringList(stationsList);

        if (stationsList.count() > 1)
        {
            stationDeparture->setValueText(stationsList[1]);
            stationArrival->setValueText(stationsList[0]);
        }
    }

    initBackend();

    //Show first launch msg
    if (settings->value("firstLaunch", "") != APP_VER)
    {
        QMessageBox::about(this,
                           tr("Welcome to a new version of fahrplan this has changed"),
                           tr("- sbb.ch: fixed parser")
                           );

        settings->setValue("firstLaunch", APP_VER);
    }
}

void MainWindow::initBackend()
{
    QString backendType = settings->value("backend", "parserMobileBahnDe").toString();

    //Init Parser Backend
    if (backendType == "parserMobileBahnDe")
    {
        backendParser = new parserMobileBahnDe();
        setWindowTitle("Fahrplan (bahn.de)");
    } else if (backendType == "parserMvvDe")
    {
        backendParser = new parserMvvDe();
        setWindowTitle("Fahrplan (mvv-muenchen.de)");
    } else if (backendType == "parserSbbCh")
    {
        backendParser = new parserSbbCh();
        setWindowTitle("Fahrplan (sbb.ch)");
    } else
    {
       //fallback is bahn.de
       backendParser = new parserMobileBahnDe();
       setWindowTitle("Fahrplan (bahn.de)");
    }

    //Update gui based on Backend
    QStringList trainRestrictions = backendParser->getTrainRestrictions();
    traintypesButton->setPickSelector(NULL);
    traintypesButton->setValueText(tr("not supported"));
    traintypesButton->setEnabled(false);
    if (trainRestrictions.count() > 0)
    {
        QStandardItemModel * trainrestrModel = new QStandardItemModel(0, 1);
        for (int i = 0; i < trainRestrictions.count(); i++)
        {
            QStandardItem *item = new QStandardItem(trainRestrictions[i]);
            item->setTextAlignment(Qt::AlignCenter);
            item->setEditable(false);
            trainrestrModel->appendRow(item);
        }
        QMaemo5ListPickSelector *trainrestrSelector = new QMaemo5ListPickSelector;
        trainrestrSelector->setModel(trainrestrModel);
        trainrestrSelector->setCurrentIndex(0);
        traintypesButton->setPickSelector(trainrestrSelector);
        traintypesButton->setEnabled(true);
    }
}

void MainWindow::positionUpdated(QGeoPositionInfo geoPositionInfo)
{
    if (!myPositionInfo.isValid() && geoPositionInfo.isValid())
    {
        QMaemo5InformationBox::information(this, tr("<br>GPS is now available!<br>"), QMaemo5InformationBox::DefaultTimeout);
    }
    myPositionInfo = geoPositionInfo;
}

void MainWindow::gpsButtonClicked()
{
    if (locationDataSourceIsUpdating)
    {
        locationDataSource->stopUpdates();
        locationDataSourceIsUpdating = false;
        QMaemo5InformationBox::information(this, tr("<br>GPS is now <b>disabled</b><br>"), QMaemo5InformationBox::DefaultTimeout);
    } else
    {
        locationDataSource->startUpdates();
        locationDataSourceIsUpdating = true;
        QMaemo5InformationBox::information(this, tr("<br>GPS is now <b>enabled</b><br>"), QMaemo5InformationBox::DefaultTimeout);
    }
}

void MainWindow::swapClicked()
{
    QString tmp = stationDeparture->valueText();
    stationDeparture->setValueText(stationArrival->valueText());
    stationArrival->setValueText(tmp);
}

void MainWindow::modeClicked()
{
    QMaemo5ValueButton *button = (QMaemo5ValueButton *)sender();
    ModeButtonUserData *mode = (ModeButtonUserData*)button->userData(0);
    if (mode->mode == 0)
    {
        mode->mode = 1;
        modeButton->setValueText(tr("Departure time"));

    } else {
        mode->mode = 0;
        modeButton->setValueText(tr("Arrival time"));
    }
    button->setUserData(0,mode);
}

void MainWindow::stationDepartureClicked()
{
    QPointer<selectStationDialog> selectDialog = new selectStationDialog(stationDeparture, &myPositionInfo, backendParser, settings);
    selectDialog->show();
}

void MainWindow::stationArrivalClicked()
{
    QPointer<selectStationDialog> selectDialog = new selectStationDialog(stationArrival, &myPositionInfo, backendParser, settings);
    selectDialog->show();
}

void MainWindow::alarmButtonClicked()
{
    calendarDialog *calDlg = new calendarDialog(lastSearchDetails, this);
    calDlg->open();
}

void MainWindow::donateButtonClicked()
{
    QDesktopServices::openUrl(QUrl("https://www.paypal.com/cgi-bin/webscr?cmd=_s-xclick&hosted_button_id=DW3LGJ6VEKFVS"));
    QMessageBox::about(this,
                       tr("Donate"),
                       tr("Thank you for Donating!"));
}

void MainWindow::aboutButtonClicked()
{
    QString version = APP_VER;
    QString aboutMsg = tr("by smurfy (maemo@smurfy.de)<br>Version ") + version + tr("<br><br>Current supported databackends:<br><b>bahn.de</b>, <b>mvv-muenchen.de</b>, <b>sbb.ch</b><br><br>This or prior versions uses code contribution by:<br><i>hcm</i>, <i>thp</i>, <i>qwerty12</i>, <i>qbast</i>");
    QMessageBox::about(this,
                       tr("About Fahrplan"),
                       tr(aboutMsg.toAscii()));

}

void MainWindow::settingsButtonClicked()
{
    configDialog *cfgDlg = new configDialog(settings, this);
    cfgDlg->show();
}

void MainWindow::resultItemClicked()
{
    QPushButton *button = (QPushButton *)sender();
    ResultItemUserData *data = (ResultItemUserData*)button->userData(0);

    if (detailsLoading != true)
    {
        detailsLoading = true;

        lastSearchDetails = backendParser->getJourneyDetailsData(data->item.detailsUrl);

        detailsLoading = false;

        showDetailsWindow();
    } else
    {
        QMaemo5InformationBox::information(this, tr("<br>Details already loading, please wait.<br>"), QMaemo5InformationBox::DefaultTimeout);
        return;
    }
}

void MainWindow::showDetailsWindow()
{
    if (detailsWindow)
    {
        detailsWindow->hide();
    }

    if (lastSearchDetails.items.count() == 0)
    {
        QMaemo5InformationBox::information(this, tr("<br>Error opening the details.<br>") + lastSearchResult.errorMsg, QMaemo5InformationBox::DefaultTimeout);
        return;
    }

    //Details Window
    detailsWindow = new QMainWindow(resultWindow);
    detailsWindow->setWindowTitle(tr("Details"));
    #ifdef Q_WS_MAEMO_5
        detailsWindow->setAttribute(Qt::WA_Maemo5StackedWindow);
    #endif
    detailsWindow->setAttribute(Qt::WA_DeleteOnClose);
    connect(detailsWindow, SIGNAL(destroyed(QObject*)),
            this, SLOT(windowDestroyed(QObject*)));

    //Alarm Menu Button
    QAction *alarmAct = new QAction(tr("Add to calendar"), this);
    connect(alarmAct, SIGNAL(triggered()), this, SLOT(alarmButtonClicked()));
    detailsWindow->menuBar()->addAction(alarmAct);

    QWidget *itemsWidget = new QWidget;
    QVBoxLayout *itemLayout = new QVBoxLayout;
    itemLayout->setMargin(10);
    QScrollArea *scrollArea = new QScrollArea;

/*
    qDebug() <<"Result:";
    qDebug() <<lastSearchDetails.duration;
    qDebug() <<lastSearchDetails.info;
*/

    foreach(DetailResultItem item, lastSearchDetails.items)
    {
        QString buttonText = "";

        if (item.train != "")
        {
            buttonText.append("<b>" + item.train + "</b><br>");
        }

        buttonText.append( item.fromStation );

        if (item.fromInfo != "")
        {
            buttonText.append(" - " + item.fromInfo + "<br>");
        } else
        {
            buttonText.append("<br>");
        }

        buttonText.append( item.toStation );

        if (item.toInfo != "")
        {
            buttonText.append(" - " + item.toInfo);
        }

        if (item.info != "")
        {
            buttonText.append("<br><i>" + item.info + "</i>");
        }

        QLabel *itemButtonLabel = new QLabel(buttonText);
        itemButtonLabel->setFixedWidth(770);
        itemLayout->addWidget(itemButtonLabel);

        QFrame *line = new QFrame();
        line->setGeometry(QRect(20, 150, 381, 16));
        line->setFrameShape(QFrame::HLine);
        line->setFrameShadow(QFrame::Sunken);
        line->setLineWidth(1);
        QPalette palette = line->palette();
        palette.setColor(QPalette::Dark, palette.color(QPalette::Mid));
        line->setPalette(palette);

        itemLayout->addWidget(line);
/*
        qDebug() <<"Item:";
        qDebug() <<item.train;
        qDebug() <<item.fromStation;
        qDebug() <<item.toStation;
        qDebug() <<item.toInfo;
        qDebug() <<item.fromInfo;
        qDebug() <<item.info;
*/
    }

    //Creating result data
    //Main Widget
    QWidget *mainWidget = new QWidget;
    detailsWindow->setCentralWidget(mainWidget);

    //Header
    QString headerText = "<center><b>" + lastSearchDetails.items[0].fromStation + "</b>" + tr(" to ") +
                         "<b>" + lastSearchDetails.items[lastSearchDetails.items.count() - 1].toStation + "</b>";

    if (lastSearchDetails.duration != "")
    {
        headerText.append(tr("<br><b>Duration: </b>") + lastSearchDetails.duration);
    }

    if (lastSearchDetails.info != "")
    {
        headerText.append("<br><i>" + lastSearchDetails.info + "</i>");
    }

    headerText.append("</center>");

    QLabel *headerLabel = new QLabel(headerText);
    itemsWidget->setLayout(itemLayout);

    scrollArea->setWidget(itemsWidget);

    //Main Layout
    QVBoxLayout *layout = new QVBoxLayout;
    layout->setMargin(1);
    layout->addWidget(headerLabel);
    layout->addWidget(scrollArea);

    mainWidget->setLayout(layout);

    detailsWindow->show();
}

void MainWindow::showResultWindow()
{
    if (resultWindow)
    {
        resultWindow->hide();
    }

    if (lastSearchResult.items.count() == 0)
    {
        QMaemo5InformationBox::information(this, tr("<br>No results found.<br>") + lastSearchResult.errorMsg + "<br>", QMaemo5InformationBox::DefaultTimeout);
        searchButton->setEnabled(true);
        return;
    }

    detailsLoading = false;

    /*
    qDebug() <<"Results:";
    qDebug() <<lastSearchResult.fromStation;
    qDebug() <<lastSearchResult.toStation;
    qDebug() <<lastSearchResult.timeInfo;
    qDebug() <<lastSearchResult.laterUrl;
    qDebug() <<lastSearchResult.earlierUrl;
    qDebug() <<lastSearchResult.items.count();
    */

    //Result Window
    resultWindow = new QMainWindow(this);
    resultWindow->setWindowTitle(tr("Results"));
    #ifdef Q_WS_MAEMO_5
        resultWindow->setAttribute(Qt::WA_Maemo5StackedWindow);
    #endif
    resultWindow->setAttribute(Qt::WA_DeleteOnClose);
    connect(resultWindow, SIGNAL(destroyed(QObject*)),
            this, SLOT(windowDestroyed(QObject*)));


    //Creating result data
    //Main Widget
    QWidget *mainWidget = new QWidget;
    resultWindow->setCentralWidget(mainWidget);

    //Header
    QLabel *headerLabel = new QLabel("<center><b>" + lastSearchResult.fromStation +
                                     "</b> " +
                                     tr("to") + " <b>" +
                                     lastSearchResult.toStation +
                                     "</b><br>" +
                                     "<b>" +
                                     lastSearchResult.timeInfo +
                                     "</center>"
                                     );

    //Later and earlier Menu Button
    if (lastSearchResult.earlierUrl != "")
    {
        QAction *earlierAct = new QAction(tr("Earlier"), this);
        connect(earlierAct, SIGNAL(triggered()), this, SLOT(earlierButtonClicked()));
        resultWindow->menuBar()->addAction(earlierAct);
    }
    if (lastSearchResult.laterUrl != "")
    {
        QAction *laterAct = new QAction(tr("Later"), this);
        connect(laterAct, SIGNAL(triggered()), this, SLOT(laterButtonClicked()));
        resultWindow->menuBar()->addAction(laterAct);
    }

    QWidget *itemsWidget = new QWidget;
    QVBoxLayout *itemLayout = new QVBoxLayout;
    itemLayout->setMargin(10);
    QScrollArea *scrollArea = new QScrollArea;

    ResultItem item;
    foreach(item, lastSearchResult.items)
    {
        QString changes = item.changes + tr(" transfer");
        if (item.changes != "1")
        {
            changes = item.changes + tr(" transfers");
        }
        QPushButton *itemButton = new QPushButton();
        itemButton->setFlat(true);
        QString buttonText = "";

        if (item.fromTime != "")
        {
            buttonText.append("<b>" + item.fromTime + " - " +
                              item.toTime + "</b> ");
        }

        buttonText.append(item.trainType + " (" +
                          changes + ") - " +
                          item.duration);

        itemButton->setFixedHeight(72);
        if (item.state == "+0")
        {
            buttonText.append(tr("<br>OnTime"));
            itemButton->setFixedHeight(100);
        } else if (item.state != "")
        {
            buttonText.append("<br>" + item.state);
            itemButton->setFixedHeight(100);
        }

        QVBoxLayout *itemButtonLayout = new QVBoxLayout;
        itemButton->setFixedWidth(770);
        QGtkStyle *gtkStyle = new QGtkStyle();
        itemButton->setStyle(gtkStyle);

        ResultItemUserData *data = new ResultItemUserData();
        data->item = item;
        itemButton->setUserData(0, data);

        QLabel *itemButtonLabel = new QLabel(buttonText, itemButton);
        itemButtonLabel->installEventFilter(this);
        itemButtonLayout->addWidget(itemButtonLabel);

        itemButtonLayout->setMargin(10);
        itemButtonLayout->setSpacing(5);
        itemButton->setLayout(itemButtonLayout);
        itemLayout->addWidget(itemButton);

        connect(itemButton, SIGNAL(clicked()), this, SLOT(resultItemClicked()));

        QFrame *line = new QFrame();
        line->setGeometry(QRect(20, 150, 381, 16));
        line->setFrameShape(QFrame::HLine);
        line->setFrameShadow(QFrame::Sunken);
        line->setLineWidth(1);
        QPalette palette = line->palette();
        palette.setColor(QPalette::Dark, palette.color(QPalette::Mid));
        line->setPalette(palette);

        itemLayout->addWidget(line);
        /*
        qDebug() <<"Item:";
        qDebug() <<item.fromTime;
        qDebug() <<item.toTime;
        qDebug() <<item.trainType;
        qDebug() <<item.duration;
        qDebug() <<item.state;
        qDebug() <<item.changes;
        */

    }

    itemsWidget->setLayout(itemLayout);

    scrollArea->setWidget(itemsWidget);

    //Main Layout
    QVBoxLayout *layout = new QVBoxLayout;
    layout->setMargin(1);
    layout->addWidget(headerLabel);
    layout->addWidget(scrollArea);

    mainWidget->setLayout(layout);

    resultWindow->show();
    //qDebug() << "Window" << resultWindow;
}

bool MainWindow::eventFilter(QObject* pObject, QEvent* pEvent)
{
    if ((pEvent->type() == QEvent::MouseButtonPress) || (pEvent->type() == QEvent::MouseButtonRelease))
    {
        pEvent->ignore();
        return false;
    } else
    {
        return QObject::eventFilter(pObject, pEvent);
    }
}

void MainWindow::earlierButtonClicked()
{
    lastSearchResult = backendParser->getJourneyData(lastSearchResult.earlierUrl);
    showResultWindow();
}

void MainWindow::laterButtonClicked()
{
    lastSearchResult = backendParser->getJourneyData(lastSearchResult.laterUrl);
    showResultWindow();
}

void MainWindow::searchButtonClicked(bool checked)
{
    Q_UNUSED(checked);

    if (!resultWindow) {

        QMaemo5TimePickSelector *timeSel = (QMaemo5TimePickSelector *)timeSelector->pickSelector();
        QMaemo5DatePickSelector *dateSel = (QMaemo5DatePickSelector *)dateSelector->pickSelector();
        ModeButtonUserData *mode = (ModeButtonUserData*)modeButton->userData(0);

        int trainrestrictions = 0;
        if (backendParser->getTrainRestrictions().count() > 0)
        {
            QMaemo5ListPickSelector *trainSel = (QMaemo5ListPickSelector *)traintypesButton->pickSelector();
            trainrestrictions = trainSel->currentIndex();
        }

        if (stationDeparture->valueText() != stationArrival->valueText() && stationDeparture->valueText() != tr("please select") && stationArrival->valueText() != tr("please select"))
        {
            //Store last search data
            if (settings->value("storelastsearch", true).toBool() ==  true)
            {
                QStringList oldList;
                oldList = settings->value("lastsearch/stations", oldList).toStringList();
                oldList.append(stationDeparture->valueText().trimmed());
                oldList.append(stationArrival->valueText().trimmed());

                //We need to reverse the order before removing duplicates
                //to ensure the latest station is later on the bottom
                oldList = fahrplanUtils::reverseQStringList(oldList);
                oldList.removeDuplicates();
                oldList = fahrplanUtils::reverseQStringList(oldList);

                //We will store only the last active stations used
                while (oldList.count() > 20)
                {
                    oldList.removeFirst();
                }

                settings->setValue("lastsearch/stations", oldList);
            }

            searchButton->setEnabled(false);
            lastSearchResult = backendParser->getJourneyData(stationDeparture->valueText(), stationArrival->valueText(), "", dateSel->currentDate(), timeSel->currentTime(), mode->mode, trainrestrictions);
            showResultWindow();
        } else
        {
            QMaemo5InformationBox::information(this, tr("<br>Please select a from and to station and make sure they are not the same.<br>") + lastSearchResult.errorMsg, QMaemo5InformationBox::DefaultTimeout);
        }
    }
}

void MainWindow::closeEvent(QCloseEvent *event)
{
    //qDebug() <<"close event";
    hide();
    event->accept();
}

int MainWindow::top_application()
{
    //qDebug() <<"top event";
    show();
    return 0;
}

void MainWindow::windowDestroyed(QObject* obj)
{
    disconnect(obj, SIGNAL(destroyed(QObject*)),
               this, SLOT(windowDestroyed(QObject*)));
    //qDebug() << "No window" << resultWindow;

    searchButton->setEnabled(true);
}
