/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Xavier Claessens <xavier.claessens@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include <string.h>

#include <libebook/e-book-util.h>

#include "ecs-match.h"


/* NOTICE: Most of this code is copy/pasted from EmpathyLiveSearch */

#define STR_EMPTY(x) ((x) == NULL || (x)[0] == '\0')

/**
 * stripped_char:
 *
 * Returns a stripped version of @ch, removing any case, accentuation
 * mark, or any special mark on it.
 **/
static gunichar
stripped_char (gunichar ch)
{
  gunichar retval = 0;
  GUnicodeType utype;
  gunichar *decomp;
  gsize dlen;

  utype = g_unichar_type (ch);

  switch (utype)
    {
    case G_UNICODE_CONTROL:
    case G_UNICODE_FORMAT:
    case G_UNICODE_UNASSIGNED:
    case G_UNICODE_NON_SPACING_MARK:
    case G_UNICODE_COMBINING_MARK:
    case G_UNICODE_ENCLOSING_MARK:
      /* Ignore those */
      break;
    default:
      ch = g_unichar_tolower (ch);
      decomp = g_unicode_canonical_decomposition (ch, &dlen);
      if (decomp != NULL)
        {
          retval = decomp[0];
          g_free (decomp);
        }
    }

  return retval;
}

static GPtrArray *
strip_utf8_string (const gchar *string)
{
  GPtrArray *words = NULL;
  const gchar *p;

  if (STR_EMPTY (string))
    return NULL;

  for (p = string; *p != '\0'; p = g_utf8_next_char (p))
    {
      GString *str = NULL;

      /* Search the start of the word (skip non alpha-num chars) */
      while (*p != '\0' && !g_unichar_isalnum (g_utf8_get_char (p)))
        p = g_utf8_next_char (p);

      /* Strip this word */
      while (*p != '\0')
        {
          gunichar sc;

          sc = stripped_char (g_utf8_get_char (p));
          if (sc != 0)
            {
              if (!g_unichar_isalnum (sc))
                break;

              if (str == NULL)
                str = g_string_new (NULL);
              g_string_append_unichar (str, sc);
            }

          p = g_utf8_next_char (p);
        }

      if (str != NULL)
        {
          if (words == NULL)
            words = g_ptr_array_new ();
          g_ptr_array_add (words, g_string_free (str, FALSE));
        }

      if (*p == '\0')
        break;
    }

  return words;
}

static gboolean
ecs_match_prefix (const gchar *string,
    const gchar *prefix)
{
  const gchar *p;

  if (prefix == NULL || prefix[0] == 0)
    return TRUE;

  if (STR_EMPTY (string))
    return FALSE;

  for (p = string; *p != '\0'; p = g_utf8_next_char (p))
    {
      const gchar *prefix_p = prefix;

      /* Search the start of the word (skip non alpha-num chars) */
      while (*p != '\0' && !g_unichar_isalnum (g_utf8_get_char (p)))
        p = g_utf8_next_char (p);

      /* Check if this word match prefix */
      while (*p != '\0')
        {
          gunichar sc;

          sc = stripped_char (g_utf8_get_char (p));
          if (sc != 0)
            {
              /* If the char does not match, stop */
              if (sc != g_utf8_get_char (prefix_p))
                break;

              /* The char matched. If it was the last of prefix, stop */
              prefix_p = g_utf8_next_char (prefix_p);
              if (*prefix_p == '\0')
                return TRUE;
            }

          p = g_utf8_next_char (p);
        }

      /* This word didn't match, go to next one (skip alpha-num chars) */
      while (*p != '\0' && g_unichar_isalnum (g_utf8_get_char (p)))
        p = g_utf8_next_char (p);

      if (*p == '\0')
        break;
    }

  return FALSE;
}

static gboolean
ecs_match_words (const gchar *string,
    GPtrArray *words)
{
  guint i;

  if (words == NULL)
    return TRUE;

  for (i = 0; i < words->len; i++)
    if (!ecs_match_prefix (string, g_ptr_array_index (words, i)))
      return FALSE;

  return TRUE;
}

typedef struct
{
  gchar *text;
  gchar *tel;
  GPtrArray *words;
} MatchData;

static void
match_data_reset (MatchData *data)
{
  g_free (data->text);
  data->text = NULL;

  if (data->words != NULL)
    {
      g_ptr_array_foreach (data->words, (GFunc) g_free, NULL);
      g_ptr_array_free (data->words, TRUE);
      data->words = NULL;
    }

  g_free (data->tel);
  data->tel = NULL;
}

static void
match_data_free (MatchData *data)
{
  if (data == NULL)
    return;

  match_data_reset (data);
  g_slice_free (MatchData, data);
}

static gboolean
is_profile_field (const gchar *attr_name)
{
  static GList *vcard_fields = NULL;

  if (G_UNLIKELY (vcard_fields == NULL))
    {
      GList *profiles;

      profiles = mc_profiles_list ();
      while (profiles)
        {
          McProfile *profile = profiles->data;
          const gchar *vfield;

          vfield = mc_profile_get_vcard_field (profile);
          if (vfield != NULL &&
              g_list_find_custom (vcard_fields, vfield,
                  (GCompareFunc) g_strcmp0) == NULL)
            vcard_fields = g_list_prepend (vcard_fields, g_strdup (vfield));

          g_object_unref (profile);
          profiles = g_list_delete_link (profiles, profiles);
        }
    }

  return (g_list_find_custom (vcard_fields, attr_name,
      (GCompareFunc) g_strcmp0) != NULL);
}

static gchar *
fixup_phone_number (const gchar *text)
{
  gchar *tel = NULL;

  if (text != NULL)
    tel = e_normalize_phone_number (text);

  if (tel != NULL)
    {
      gsize len;

      len = (tel[0] == '+') ? 1 : 0;
      len += strspn (tel + len, "0123456789");
      tel[len] = '\0';

      if (len == 0)
        {
          g_free (tel);
          tel = NULL;
        }
    }

  return tel;
}

static gboolean
match_contact (OssoABookContact *contact,
    MatchData *data)
{
  const gchar *str;
  GList *list;

  /* Check various unique fields */
  str = e_contact_get_const (E_CONTACT (contact), E_CONTACT_GIVEN_NAME);
  if (ecs_match_words (str, data->words))
    return TRUE;

  str = e_contact_get_const (E_CONTACT (contact), E_CONTACT_FAMILY_NAME);
  if (ecs_match_words (str, data->words))
    return TRUE;

  str = e_contact_get_const (E_CONTACT (contact), E_CONTACT_FULL_NAME);
  if (ecs_match_words (str, data->words))
    return TRUE;

  str = e_contact_get_const (E_CONTACT (contact), E_CONTACT_NICKNAME);
  if (ecs_match_words (str, data->words))
    return TRUE;

  str = e_contact_get_const (E_CONTACT (contact), E_CONTACT_ORG);
  if (ecs_match_words (str, data->words))
    return TRUE;

  /* Check vairous multiple fields */
  for (list = e_vcard_get_attributes (E_VCARD (contact));
       list != NULL; list = list->next)
    {
      EVCardAttribute *attr = list->data;
      const gchar *attr_name;
      const gchar *attr_value;
      GList *l;
      gboolean match = FALSE;

      attr_name = e_vcard_attribute_get_name (attr);
      l = e_vcard_attribute_get_values (attr);
      attr_value = l ? l->data : NULL;

      if (g_strcmp0 (attr_name, EVC_TEL) == 0)
        {
          if (data->tel != NULL)
            {
              gchar *tel;

              tel = fixup_phone_number (attr_value);
              if (tel != NULL)
                match = (strstr (tel, data->tel) != NULL);
              g_free (tel);
            }
        }
      else if (g_strcmp0 (attr_name, EVC_EMAIL) == 0 ||
               is_profile_field (attr_name))
        {
          const gchar *p = NULL;
          gchar *stripped = NULL;

          /* Strip the @server.com part */
          if (attr_value != NULL)
            p = strstr (attr_value, "@");
          if (p != NULL)
            stripped = g_strndup (attr_value, p - attr_value);

          match = ecs_match_words (stripped ? stripped : attr_value,
              data->words);

          g_free (stripped);
        }

      if (match)
        return TRUE;
    }

  /* Recurse on roster contacts, if any */
  list = osso_abook_contact_get_roster_contacts (contact);
  while (list)
    {
      OssoABookContact *rc = list->data;

      if (match_contact (rc, data))
        {
          g_list_free (list);
          return TRUE;
        }

      list = g_list_delete_link (list, list);
    }

  return FALSE;
}

gboolean
ecs_match_visible_func (OssoABookListStore *store,
    GtkTreeIter *iter,
    const gchar *text)
{
  OssoABookListStoreRow *row;
  MatchData *data;

  g_return_val_if_fail (OSSO_ABOOK_IS_LIST_STORE (store), FALSE);
  g_return_val_if_fail (iter != NULL, FALSE);

  row = osso_abook_list_store_iter_get_row (store, iter);
  g_return_val_if_fail (NULL != row, FALSE);

  /* The visible func is called for each row with the same text pointer,
   * to avoid stripping it each time, we keep a little data struct on the
   * store */
  data = g_object_get_data (G_OBJECT (store), "match-data");
  if (G_UNLIKELY (data == NULL))
    {
      data = g_slice_new0 (MatchData);
      g_object_set_data_full (G_OBJECT (store), "match-data", data,
          (GDestroyNotify) match_data_free);
    }
  if (g_strcmp0 (text, data->text) != 0)
    {
      match_data_reset (data);
      data->text = g_strdup (text);
      data->words = strip_utf8_string (text);
      data->tel = fixup_phone_number (text);
    }

  return match_contact (row->contact, data);
}
