/*
  DiskUsage - shows the disk usage under a number of operating systems.
  Copyright (C) 2004  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <Map.h>

#include <cmath>

#include <Lum/Base/String.h>

#include <Lum/OS/Bitmap.h>
#include <Lum/OS/Driver.h>

#include <Lum/Panel.h>
#include <Lum/TextValue.h>

class Map : public Lum::Object
{
private:
  class Rect
  {
  public:
    int          x;
    int          y;
    size_t       width;
    size_t       height;
    std::wstring label;
    double       size;
  };

private:
  DirEntry              *top;
  double                size;

  std::list<Rect>       labels;

  Lum::Model::StringRef label;
  Lum::OS::Bitmap       *bitmap;

public:
  Map(DirEntry *top, double size);
  ~Map();

  Lum::Model::String* GetLabel() const;

  void CalcSize();

  void DrawEntry(Lum::OS::DrawInfo *draw,
                 DirEntry* entry,
                 int x, int y, size_t width, size_t height,
                 int depth);

  void Draw(Lum::OS::DrawInfo* draw,
            int x, int y, size_t w, size_t h);

  void ShowLabel(int x, int y);

  bool HandleMouseEvent(const Lum::OS::MouseEvent& event);
  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg);
};

DirEntry::DirEntry(const Lum::Base::Path& path)
 : path(path),
   size(0)
{
  // no code
}

DirEntry::~DirEntry()
{
  for (std::map<std::wstring, DirEntry*>::const_iterator iter=subDirs.begin();
       iter!=subDirs.end();
       ++iter) {
    delete iter->second;
  }
}

void DirEntry::AccumulateSize()
{
  for (std::map<std::wstring, DirEntry*>::const_iterator iter=subDirs.begin();
       iter!=subDirs.end();
       ++iter) {
    iter->second->AccumulateSize();
    size+=iter->second->size;
  }
}

Map::Map(DirEntry *top, double size)
 : top(top),
   size(size),
   label(new Lum::Model::String(L"")),
   bitmap(NULL)
{
  Observe(label);
}

Map::~Map()
{
  delete bitmap;
  delete top;
}

Lum::Model::String* Map::GetLabel() const
{
  return label.Get();
}

void Map::CalcSize()
{
  minWidth=100;
  minHeight=100;

  width=minWidth;
  height=minHeight;

  Object::CalcSize();
}

void Map::DrawEntry(Lum::OS::DrawInfo *draw,
                    DirEntry* entry,
                    int x, int y, size_t width, size_t height,
                    int depth)
{
  size_t full=lround(entry->size);
  size_t area=width*height;
  int    offset;

  // TODO: Instead dividing hardcode by 1024 to avoid overlow,
  // Calculate factor based on the full value!

  if (area==0 || full==0) {
    return;
  }

  if (width>=height) {
    offset=x-1;
  }
  else {
    offset=y-1;
  }

  for (std::map<std::wstring, DirEntry*>::const_iterator iter=entry->subDirs.begin();
       iter!=entry->subDirs.end();
       ++iter) {

    if (iter->second->size==0) {
      continue;
    }

    Rect rect;

    rect.label=iter->second->path.GetPath();
    rect.size=iter->second->size;

    size_t subArea=lround((area*iter->second->size)/full);

    if (subArea==0) {
      continue;
    }

    if (width>=height) {
      size_t w;

      w=subArea/height;

      if (w<=1) {
        continue;
      }

      rect.x=offset+1;
      rect.y=y;
      rect.width=w-1;
      rect.height=height;

      labels.push_front(rect);

      draw->PushForeground(Lum::OS::Display::fillColor);
      draw->FillRectangle(offset+1,y,w-1,height);
      draw->PopForeground();

      draw->PushForeground(Lum::OS::Display::blackColor);
      draw->DrawLine(offset+w,y,offset+w,y+height-1);
      draw->PopForeground();

      DrawEntry(draw,iter->second,offset+1,y,w-1,height,depth+1);

      offset+=w;
    }
    else {
      size_t h;

      h=subArea/width;

      if (h<=1) {
        continue;
      }

      rect.x=x;
      rect.y=offset+1;
      rect.width=width;
      rect.height=h-1;

      labels.push_front(rect);

      draw->PushForeground(Lum::OS::Display::fillColor);
      draw->FillRectangle(x,offset+1,width,h-1);
      draw->PopForeground();

      draw->PushForeground(Lum::OS::Display::blackColor);
      draw->DrawLine(x,offset+h,x+width-1,offset+h);
      draw->PopForeground();

      DrawEntry(draw,iter->second,x,offset+1,width,h-1,depth+1);

      offset+=h;
    }
  }
}

void Map::Draw(Lum::OS::DrawInfo* draw,
               int x, int y, size_t w, size_t h)
{
  Object::Draw(draw,x,y,w,h); /* We must call Draw of our superclass */

  if (!OIntersect(x,y,w,h)) {
    return;
  }

  size_t fillFactor;

  draw->PushForeground(Lum::OS::Display::whiteColor);
  draw->FillRectangle(this->x,this->y,width,height);
  draw->PopForeground();

  draw->PushForeground(Lum::OS::Display::blackColor);
  draw->DrawRectangle(this->x,this->y,width,height);
  draw->PopForeground();

  fillFactor=lround((100*top->size)/size);

  if (bitmap==NULL ||
      bitmap->GetWidth()!=width-2 ||
      bitmap->GetHeight()!=height-2) {
    delete bitmap;
    bitmap=Lum::OS::driver->CreateBitmap(width-2,height-2);

    if (bitmap!=NULL) {
      bitmap->GetDrawInfo()->PushForeground(Lum::OS::Display::whiteColor);
      bitmap->GetDrawInfo()->FillRectangle(0,0,bitmap->GetWidth(),bitmap->GetHeight());
      bitmap->GetDrawInfo()->PopForeground();

      labels.clear();
      DrawEntry(bitmap->GetDrawInfo(),top,0,0,((width-2)*fillFactor)/100,height-2,1);
    }
  }

  if (bitmap!=NULL) {
    draw->CopyFromBitmap(bitmap,
                         0,0,((width-2)*fillFactor)/100,height-2,
                         this->x+1,this->y+1);

    if (label.Valid() &&
        !label->IsNull() &&
        label->Get().length()>0) {
      Lum::OS::FontRef    font;
      std::wstring        text=label->Get();
      Lum::OS::FontExtent extent;
      int                 xOffset;
      int                 yOffset;

      if (Lum::OS::display->GetSize()>=Lum::OS::Display::sizeNormal) {
        font=Lum::OS::display->GetFont(Lum::OS::Display::fontScaleCaption1);
      }
      else {
        font=Lum::OS::display->GetFont();
      }
      font->StringExtent(text,extent,Lum::OS::Font::bold);

      xOffset=this->x+(width-extent.width)/2;
      yOffset=this->y+(height-font->height)/2+font->ascent;

      draw->PushFont(font,Lum::OS::Font::bold);

      draw->PushForeground(Lum::OS::Display::blackColor);
      draw->DrawString(xOffset+1,yOffset+1,text);
      draw->PopForeground();

      draw->PushForeground(Lum::OS::Display::whiteColor);
      draw->DrawString(xOffset,yOffset,text);
      draw->PopForeground();

      draw->PopFont();
    }
  }
}

void Map::ShowLabel(int x, int y)
{
  for (std::list<Rect>::const_iterator iter=labels.begin();
       iter!=labels.end();
       ++iter) {
    if (x>=iter->x && y>=iter->y &&
        x<iter->x+(int)iter->width && y<iter->y+(int)iter->height) {
      label->Set(iter->label+L" "+Lum::Base::ByteSizeToWString(iter->size*1024));
      return;
    }
  }

  label->SetNull();
}

bool Map::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (!visible) {
    return false;
  }

  if (event.type==Lum::OS::MouseEvent::down && PointIsIn(event) &&
      event.button==Lum::OS::MouseEvent::button1) {
    ShowLabel(event.x-this->x,event.y-this->y);
    return true;
  }
  else if (event.type==Lum::OS::MouseEvent::move && event.IsGrabed()) {
    ShowLabel(event.x-this->x,event.y-this->y);
  }
  else if (event.IsGrabEnd()) {
    label->SetNull();
  }

  return false;
}

void Map::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==label) {
    Redraw();
  }

  Lum::Object::Resync(model,msg);
}


MapWindow::MapWindow(DirEntry *top, double size)
 : top(top),
   size(size)
{
  GetWindow()->SetScreenOrientationHint(Lum::OS::Window::screenOrientationBothSupported);

  Observe(GetClosedAction());
}

Lum::Object* MapWindow::GetContent()
{
  Lum::Panel *vert;
  Map        *map;

  vert=Lum::VPanel::Create(true,true);
  vert->SetWidth(Lum::Base::Size::workHRel,80);
  vert->SetHeight(Lum::Base::Size::workVRel,60);

  map=new Map(top,size);
  map->SetFlex(true,true);

  vert->Add(map);

  return vert;
}

void MapWindow::GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
{
  Lum::Dlg::ActionDialog::CreateActionInfosClose(actions,GetClosedAction());
}

void MapWindow::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==GetClosedAction() &&
      GetClosedAction()->IsFinished()) {
    Exit();
  }

  Lum::Dlg::ActionDialog::Resync(model,msg);
}

ScanTask::ScanTask()
  : top(NULL),
  size(0)
{
  // no code
}

void ScanTask::SetSize(double size)
{
  this->size=size;
}

void ScanTask::SetDir(const Lum::Base::Path& dir)
{
  this->dir=dir;
}

void ScanTask::Run()
{
  unsigned long              scannedDirs=0;
  std::list<Lum::Base::Path> todos;
  std::set<unsigned long>    hardLinks;

  delete top;

  todos.push_back(dir);

  while (todos.size()>0 && !IsAborted()) {
    Lum::Base::Path              todo;
    Lum::Base::DirScanner::Entry entry;
    Lum::Base::Status            status;
    Lum::Base::DirScannerRef     scanner;
    DirEntry                     *e;

    todo=todos.front();
    todos.pop_front();
    e=new DirEntry(todo);

    scannedDirs++;
    SetAction(L"Scanned "+
              Lum::Base::NumberToWString(scannedDirs)+
              L" from "+
              Lum::Base::NumberToWString(scannedDirs+todos.size())+
              L" directories...");
    SetProgress((100*scannedDirs)/(scannedDirs+todos.size()));

    //
    // Scan the directory
    //

    scanner=Lum::Base::DirScanner::Create(todo);

    while (scanner->GetNext(entry,status)) {
      if (entry.name!=L"." && entry.name!=L"..") {
        if (entry.isDir && !entry.isSoftLink &&
            !(entry.isHardLink && hardLinks.find(entry.linkId)!=hardLinks.end())) {
          Lum::Base::Path subDir(todo);

          subDir.AppendDir(entry.name);

          todos.push_back(subDir);
        }
        else if (!(entry.isHardLink && hardLinks.find(entry.linkId)!=hardLinks.end())) {
          e->size+=entry.size;
          //e->size+=entry.blockSize;
        }
      }

      if (entry.isHardLink) {
        hardLinks.insert(entry.linkId);
      }
    }

    e->size/=1024;

    //
    // Link into the directory tree
    //

    if (top==NULL) {
      top=e;
    }
    else {
      DirEntry *current=top;

      for (size_t i=dir.GetPartCount(); i<todo.GetPartCount()-1; i++) {
        assert(current->subDirs.find(todo.GetPart(i))!=current->subDirs.end());

        current=current->subDirs[todo.GetPart(i)];
      }

      assert(current!=NULL);

      current->subDirs[todo.GetPart(todo.GetPartCount()-1)]=e;
    }
  }

  if (IsAborted()) {
    delete top;
    top=NULL;
    todos.clear();

    return;
  }

  top->AccumulateSize();
}


