/*
  DiskUsage - shows the disk usage under a number of operating systems.
  Copyright (C) 2004  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cmath>
#include <iomanip>
#include <iostream>
#include <sstream>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Object.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Def/Menu.h>

#include <Lum/Dlg/About.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/DataStream.h>
#include <Lum/Model/Number.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/OS/Color.h>

#include <Lum/Manager/Display.h>
#include <Lum/Manager/FileSystem.h>

#include <Lum/Application.h>
#include <Lum/Label.h>
#include <Lum/PercentPie.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>

#include "config.h"

#include "DiskInfo.h"
#include "Map.h"

static Lum::Def::AppInfo info;

static Lum::TextValue* GetTextValueLeft(const Lum::Model::StringRef& value)
{
  Lum::TextValue   *text;
  Lum::OS::FontRef font(Lum::OS::display->GetFont(Lum::OS::Display::fontScaleCaption1));

  text=Lum::TextValue::Create(value,Lum::TextValue::left,true,false);
  text->SetMinWidth(Lum::Base::Size::stdCharWidth,25);
  // TODO: font

  return text;
}

static Lum::TextValue* GetTextValueRight(const Lum::Model::StringRef& value)
{
  Lum::TextValue   *text;
  Lum::OS::FontRef font(Lum::OS::display->GetFont(Lum::OS::Display::fontScaleCaption1));

  text=Lum::TextValue::Create(value,Lum::TextValue::right,true,false);
  text->SetMinWidth(Lum::Base::Size::stdCharWidth,25);
  // TODO: font

  return text;
}

class DiskInfoThread : public Lum::OS::Thread
{
public:
  Lum::Model::ActionRef dataAvailable;
  std::list<DiskInfo>   diskInfos;

public:
  DiskInfoThread()
   : dataAvailable(new Lum::Model::Action())
  {
    // no code
  }

  ~DiskInfoThread()
  {
    Join();
  }

  void Run()
  {
    diskInfos.clear();
    GetDisks(diskInfos);

    Lum::OS::display->QueueActionForAsyncNotification(dataAvailable);
  };
};

typedef Lum::Model::StdTable<DiskInfo>      DiskInfoModel;
typedef Lum::Base::Reference<DiskInfoModel> DiskInfoModelRef;

class DiskInfoModelPainter : public Lum::TableCellPainter
{
private:
  Lum::OS::ColorRef red;
  Lum::OS::ColorRef green;

public:
  DiskInfoModelPainter()
  : red(1,0,0,Lum::OS::display->GetColor(Lum::OS::Display::tableTextColor)),
    green(0,1,0,Lum::OS::display->GetColor(Lum::OS::Display::tableTextColor))
  {
    // no code
  }

  size_t GetProposedHeight() const
  {
    return Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
           GetFont()->height+
           GetFont()->height+
           Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder);
  }

  void Draw(Lum::OS::DrawInfo* draw,
            int x, int y, size_t width, size_t height) const
  {
    DiskInfo     diskInfo=dynamic_cast<const DiskInfoModel*>(GetModel())->GetEntry(GetRow());
    std::wstring sizeText=Lum::Base::ByteSizeToWString(diskInfo.avail*diskInfo.unit*1024);
    size_t       sizeTextWidth=GetFont()->StringWidth(sizeText);

    draw->PushFont(GetFont());
    draw->PushForeground(GetTextColor(draw));
    draw->DrawString(x,
                     y+Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+GetFont()->ascent,
                     diskInfo.dir);

    draw->DrawString(x+width-sizeTextWidth+1,
                     y+Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+GetFont()->ascent,
                     sizeText);

    draw->PopForeground();
    draw->PopFont();


    double percent=(diskInfo.blocks-diskInfo.avail)/diskInfo.blocks;

    if (percent>=0.9) {
      draw->PushForeground(red);
    }
    else {
      draw->PushForeground(green);
    }

    draw->FillRectangle(x,
                        y+
                        Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
                        GetFont()->height,
                        (size_t)(width*percent),
                        GetFont()->height);

    draw->PopForeground();
  }
};

class MainWindow : public Lum::Dialog
{
private:
  Lum::Model::ActionRef              about;

  Lum::Model::ActionRef              showMap;

  Lum::Model::ULongRef               refreshTime;
  Lum::Model::ActionRef              timer;

  Lum::Table                         *table;
  Lum::Model::SingleLineSelectionRef selection;
  DiskInfoModelRef                   tableModel;
  Lum::Model::StringRef              device;
  Lum::Model::StringRef              dir;
  Lum::Model::StringRef              used;
  Lum::Model::StringRef              free;
  Lum::Model::StringRef              total;
  Lum::Model::DoubleDataStreamRef    data;

  std::list<DiskInfo>                list;

  DiskInfoThread                     diskInfoThread;
  ScanTask                           scanTask;

public:
  MainWindow()
  : about(new Lum::Model::Action()),
    showMap(new Lum::Model::Action()),
    refreshTime(new Lum::Model::ULong(15)),
    timer(new Lum::Model::Action()),
    selection(new Lum::Model::SingleLineSelection),
    tableModel(new DiskInfoModel()),
    device(new Lum::Model::String()),
    dir(new Lum::Model::String()),
    used(new Lum::Model::String()),
    free(new Lum::Model::String()),
    total(new Lum::Model::String()),
    data(new Lum::Model::DoubleDataStream())
  {
    GetWindow()->SetScreenOrientationHint(Lum::OS::Window::screenOrientationBothSupported);

    tableModel->SetEmptyText(_(L"NO_PARTITIONS",L"(No partitions found)"));

    data->SetNotificationMode(Lum::Model::DoubleDataStream::notifyExplicit);
    data->SetChannels(3);

    scanTask.SetParent(this);
    scanTask.SetCaption(_(L"SCANNING",L"Scanning directory..."));

    Observe(about);
    Observe(showMap);
    Observe(refreshTime);
    Observe(timer);
    Observe(selection);
    Observe(GetOpenedAction());

    Observe(diskInfoThread.dataAvailable);

    Observe(scanTask.GetAbortedAction());
    Observe(scanTask.GetFinishedAction());
  }

  void PreInit()
  {
    Lum::Panel            *panel,*panel2;
    Lum::PercentPie       *percent;
    Lum::Model::HeaderRef headerModel;

    if (Lum::OS::display->IsLandscape()) {
      panel=Lum::VPanel::Create(true,true);

      panel2=Lum::HPanel::Create(true,false);
      panel2->Add(Lum::Label::Create()
                  ->AddLabel(_(L"LABEL_DEVICE",L"Device:"),
                             GetTextValueLeft(device))
                  ->AddLabel(_(L"LABEL_DIRECTORY",L"Directory:"),
                             GetTextValueLeft(dir))
                  ->AddLabel(_(L"LABEL_SIZE",L"Size:"),
                             GetTextValueRight(total))
                  ->AddLabel(_(L"LABEL_USED",L"Used:"),
                             GetTextValueRight(used))
                  ->AddLabel(_(L"LABEL_FREE",L"Free:"),
                             GetTextValueRight(free)));
      panel2->AddSpace(true);

      percent=new Lum::PercentPie();
      percent->SetModel(data);
      panel2->Add(percent);

      panel->Add(panel2);
      panel->AddSpace();

      headerModel=new Lum::Model::HeaderImpl();
      headerModel->AddColumn(L"Dir",Lum::Base::Size::stdCharWidth,25,true);

      table=new Lum::Table();
      table->SetHeight(Lum::Base::Size::workVRel,20);
      table->SetFlex(true,true);
      table->SetModel(tableModel);
      table->SetHeaderModel(headerModel);
      table->SetSelection(selection);
      table->SetDoubleClickAction(showMap);
      table->SetShowHeader(false);
      table->SetPainter(new DiskInfoModelPainter());
      table->GetTableView()->SetAutoFitColumns(true);
      panel->Add(table);

      SetMain(panel);
    }
    else {
      panel=Lum::VPanel::Create(true,true);

      percent=new Lum::PercentPie();
      percent->SetModel(data);
      panel->Add(percent);

      panel->AddSpace();

      panel2=Lum::HPanel::Create(true,false);
      panel2->Add(Lum::Label::Create()
                  ->AddLabel(L"Device:",GetTextValueLeft(device))
                  ->AddLabel(L"Directory:",GetTextValueLeft(dir))
                  ->AddLabel(L"Size:",GetTextValueRight(total))
                  ->AddLabel(L"Used:",GetTextValueRight(used))
                  ->AddLabel(L"Free:",GetTextValueRight(free)));
      panel2->AddSpace(true);
      panel->Add(panel2);

      panel->AddSpace();

      headerModel=new Lum::Model::HeaderImpl();
      headerModel->AddColumn(L"Dir",Lum::Base::Size::stdCharWidth,25,true);

      table=new Lum::Table();
      table->SetMinHeight(Lum::Base::Size::workVRel,20);
      table->SetFlex(true,true);
      table->SetModel(tableModel);
      table->SetHeaderModel(headerModel);
      table->SetSelection(selection);
      table->SetDoubleClickAction(showMap);
      table->SetShowHeader(false);
      table->SetPainter(new DiskInfoModelPainter());
      table->GetTableView()->SetAutoFitColumns(true);
      panel->Add(table);

      SetMain(panel);
    }

    Lum::Def::Menu *menu=Lum::Def::Menu::Create();

    menu
      ->GroupProject()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->GroupEdit()
        ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_OPEN_MAP",L"_Treemap"))
                                  .SetShortcut(Lum::OS::qualifierControl,L"m"),
                                  showMap)
                 .SetOpensDialog())
      ->End()
      ->GroupHelp()
        //->AddMenuItemAction(_ld(menuHelpHelp),NULL)
        ->ActionAbout(about)
      ->End();

    SetMenu(menu);

    Dialog::PreInit();
  }

  void UpdateDisplay()
  {
    std::list<DiskInfo>::const_iterator iter;

    // First try to update existing entries in the current table
    iter=list.begin();
    while (iter!=list.end()) {
      size_t i;

      for (i=1; i<=tableModel->GetRows(); i++) {
        if (iter->dir==tableModel->GetEntry(i).dir) {
          break;
        }
      }

      if (i<=tableModel->GetRows()) {
        tableModel->GetEntry(i)=*iter;
        tableModel->RedrawRow(i);
      }
      else {
        tableModel->Append(*iter);
      }

      ++iter;
    }

    // Now see, if entries in the table have been removed and must
    // be deleted
    size_t i=1;
    while(i<=tableModel->GetRows()) {
      std::wstring                        directory=tableModel->GetEntry(i).dir;
      std::list<DiskInfo>::const_iterator iter;

      iter=list.begin();
      while (iter!=list.end()) {
        if (iter->dir==directory) {
          break;
        }

        ++iter;
      }

      if (iter==list.end()) {
        tableModel->Delete(i);
      }
      else {
        i++;
      }
    }
  }

  void ShowDetail()
  {
    std::list<DiskInfo>::const_iterator iter;

    if (selection->HasSelection()) {
      iter=list.begin();
      while (iter!=list.end() && iter->dir!=tableModel->GetEntry(selection->GetLine()).dir) {
        ++iter;
      }
    }
    else {
      iter=list.end();
    }

    if (iter!=list.end()) {
      device->Set(iter->name);
      dir->Set(iter->dir);

      total->Set(Lum::Base::ByteSizeToWString(iter->blocks*iter->unit*1024));
      used->Set(Lum::Base::ByteSizeToWString((iter->blocks-iter->free)*iter->unit*1024));
      free->Set(Lum::Base::ByteSizeToWString(iter->avail*iter->unit*1024)+
                L" (root: "+Lum::Base::ByteSizeToWString(iter->free*iter->unit*1024)+L")");

      data->Enable();
      data->Set(0,(iter->blocks-iter->free)/iter->blocks); // used
      data->Set(1,iter->avail/iter->blocks); // free without reserved
      data->Set(2,(iter->free-iter->avail)/iter->blocks); // reserved
      data->Notify();
    }
    else {
      device->Set(L"");

      used->Set(L"");
      free->Set(L"");
      total->Set(L"");

      data->Disable();
    }
  }

  void InitiateScan()
  {
    if (!selection->HasSelection()) {
      return;
    }

    std::list<DiskInfo>::const_iterator iter=list.begin();

    while (iter!=list.end() && iter->dir!=tableModel->GetEntry(selection->GetLine()).dir) {
      ++iter;
    }

    if (iter==list.end()) {
      return;
    }

    Lum::Base::Path path;

    path.SetNativeDir(iter->dir);

    scanTask.SetSize(iter->blocks*iter->unit);
    scanTask.SetDir(path);
    scanTask.Start();
  }

  void ShowMap()
  {
    DirEntry  *top;
    MapWindow *window;

    top=scanTask.top;
    scanTask.top=NULL;

    window=new MapWindow(top,top->size/*scanTask.size*/);
    window->SetParent(this);

    if (window->Open()) {
      window->EventLoop();
      window->Close();
    }

    delete window;
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      diskInfoThread.Start();

      if (Lum::Manager::Display::Instance()!=NULL) {
        Observe(Lum::Manager::Display::Instance()->GetDisplayStateChangeAction());
      }
    }
    else if (model==diskInfoThread.dataAvailable &&
             diskInfoThread.dataAvailable->IsFinished()) {
      list=diskInfoThread.diskInfos;
      UpdateDisplay();

      if (tableModel->GetRows()>0 &&
          !selection->HasSelection()) {
        selection->SelectLine(1);
      }

      if (Lum::Manager::Display::Instance()==NULL ||
          (Lum::Manager::Display::Instance()->GetDisplayState()==Lum::Manager::Display::DisplayStateOn ||
           Lum::Manager::Display::Instance()->GetDisplayState()==Lum::Manager::Display::DisplayStateUnknown)) {
        Lum::OS::display->AddTimer(refreshTime->Get(),0,timer);
      }
    }
    else if (model==timer && timer->IsFinished()) {
      diskInfoThread.Start();
    }
    else if (model==refreshTime && IsOpen()) {
      /*
      Lum::OS::display->RemoveTimer(timer);
      if (refreshTime->Get()>0) {
        Lum::OS::display->AddTimer(refreshTime->Get(),0,timer);
      }*/
    }
    else if (model==selection) {
      ShowDetail();
    }
    else if (model==showMap && showMap->IsFinished()) {
      InitiateScan();
    }
    else if (model==scanTask.GetFinishedAction() &&
             scanTask.GetFinishedAction()->IsFinished()) {
      ShowMap();
    }
    else if (model==about && about->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (Lum::Manager::Display::Instance()!=NULL &&
             Lum::Manager::Display::Instance()->GetDisplayStateChangeAction()==model &&
             Lum::Manager::Display::Instance()->GetDisplayStateChangeAction()->IsFinished()) {
      if (Lum::Manager::Display::Instance()->GetDisplayState()==Lum::Manager::Display::DisplayStateOn ||
        Lum::Manager::Display::Instance()->GetDisplayState()==Lum::Manager::Display::DisplayStateUnknown) {
        Lum::OS::display->RemoveTimer(timer);
        Lum::OS::display->AddTimer(refreshTime->Get(),0,timer);
      }
    }

    Dialog::Resync(model,msg);
  }

};

class Main : public Lum::GUIApplication<MainWindow>
{
public:
  bool Initialize()
  {
#if defined(APP_DATADIR)
    Lum::Manager::FileSystem::Instance()->SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"Shows usage of all available disks"));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2004, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    return Lum::GUIApplication<MainWindow>::Initialize();
  }
};

LUM_MAIN(Main,L"DiskUsage")
