/*
  DiceJinni - A dice simulator
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "GameArea.h"

#include <Lum/Base/String.h>

#include <Lum/OS/Driver.h>

#include "Configuration.h"

static size_t diceFrame = 3;

GameArea::GameArea(bool horizontalFlex, bool verticalFlex)
 : area(NULL),
   backgroundColor(0.0,0.5,0.0,Lum::OS::display->GetColor(Lum::OS::Display::backgroundColor)),
   cellBackgroundColor(1.0,1.0,1.0,Lum::OS::display->GetColor(Lum::OS::Display::whiteColor)),
   normalFrameColor(0.0,0.0,0.0,Lum::OS::display->GetColor(Lum::OS::Display::blackColor)),
   staysFrameColor(1.0,0.0,0.0,Lum::OS::display->GetColor(Lum::OS::Display::backgroundColor)),
   textColor(0.0,0.0,0.0,Lum::OS::display->GetColor(Lum::OS::Display::textColor)),
   font(Lum::OS::display->GetFont(200)),
   cellSize(0)
{
  SetFlex(horizontalFlex,verticalFlex);
  SetBackground(new Lum::OS::PlainFill(backgroundColor));

  SetCanFocus(true);
  RequestFocus();

  Observe(currentSet);
  Observe(setsModel);
}

GameArea::~GameArea()
{
  delete area;
}

bool GameArea::HandlesClickFocus() const
{
  return CanFocus() && IsVisible();
}

bool GameArea::HandlesKeyFocus() const
{
  return CanFocus() && IsVisible();
}

void GameArea::UpdateDimensions()
{
  if (area!=NULL) {
    hAdjustment->SetDimension(width,area->GetWidth());
    vAdjustment->SetDimension(height,area->GetHeight());
  }
  else {
    hAdjustment->SetInvalid();
    vAdjustment->SetInvalid();
  }
}

void GameArea::CalcAreaSize(size_t width, size_t height)
{
  if (!currentSet.Valid() ||
      currentSet->IsNull() ||
      currentSet->Get()>setsModel->GetRows()) {
    cellSize=0;
    areaWidth=0;
    areaHeight=0;
    return;
  }

  //
  // Calculate minimum cell size so that all texts can be displayed in full.
  //

  std::list<DiceSet>::const_iterator iter=sets.begin();

  if (currentSet->Get()>0) {
    std::advance(iter,currentSet->Get()-1);
  }

  cellSize=font->height;
  for (std::list<Dice>::const_iterator dice=iter->dices.begin();
       dice!=iter->dices.end();
       dice++) {
    for (std::vector<Value>::const_iterator value=dice->values.begin();
         value!=dice->values.end();
         value++) {
      Lum::OS::FontExtent extent;

      font->StringExtent(value->value,extent,Lum::OS::Font::normal);

      cellSize=std::max(cellSize,extent.width);
    }
  }

  cellSize+=2*std::max(Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceObjectBorder),
                       Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder));
  cellSize+=2*diceFrame; // Frame around the dice

  cellSize=std::max(cellSize,(size_t)Lum::Base::GetSize(Lum::Base::Size::mm,9));

  columns=0;

  //
  // Now place as many dices in one row as can be shown without horizontal scroll bar
  // First make sure that we make the cells as big as possible without adding a vertical scrollbar.
  //

  size_t cellArea=0;

  for (size_t i=1; i<=iter->dices.size(); i++) {
    size_t w=(i+1)*distance+i*cellSize;
    size_t r;
    size_t c;
    size_t aw;
    size_t ah;

    if (w>width && i>1) {
      break;
    }

    c=i;
    aw=w;

    r=iter->dices.size()/i;
    if (iter->dices.size()%i>0) {
      r++;
    }

    ah=(r+1)*distance+r*cellSize;

    if (ah<=height) {
      size_t cs;
      size_t ca;

      cs=std::min((width-(i+1)*distance)/i,(height-(r+1)*distance)/r);

      ca=cs*cs*iter->dices.size();

      if (ca>cellArea) {
        columns=c;
        cellSize=cs;
        areaWidth=(i+1)*distance+i*cellSize;
        areaHeight=(r+1)*distance+r*cellSize;
      }
    }
  }

  //
  // If this does not work, place as many cells in a row as possible without showing
  // a vertical scrollbar.
  //

  if (columns==0) {
    for (size_t i=1; i<=iter->dices.size(); i++) {
      size_t w=(i+1)*distance+i*cellSize;
      size_t j;

      if (w>width && i>1) {
        break;
      }

      columns=i;
      areaWidth=w;

      j=iter->dices.size()/i;
      if (iter->dices.size()%i>0) {
        j++;
      }

      areaHeight=(j+1)*distance+j*cellSize;
    }
  }
}

/**
  Create bitmap containing the game area background build up from walls, floors,
  goals and backgrounds.
 */
void GameArea::RebuildArea()
{
  delete area;

  area=NULL;

  if (areaWidth>0 && areaHeight>0) {
    area=Lum::OS::driver->CreateBitmap(areaWidth,areaHeight);
  }

  UpdateDimensions();
}

void GameArea::RedrawArea()
{
  if (!currentSet.Valid() ||
      currentSet->IsNull() ||
      area==NULL) {
    return;
  }

  size_t                             x,y,i;
  std::list<DiceSet>::const_iterator iter=sets.begin();
  Lum::OS::DrawInfo                  *draw=area->GetDrawInfo();

  i=1;
  x=distance;
  y=distance;

  GetBackground()->Draw(draw,
                        0,0,areaWidth,areaHeight,
                        0,0,areaWidth,areaHeight);

  if (currentSet->Get()>0) {
    std::advance(iter,currentSet->Get()-1);
  }

  for (std::list<Dice>::const_iterator dice=iter->dices.begin();
       dice!=iter->dices.end();
       dice++) {

    if (dice->stays) {
      draw->PushForeground(staysFrameColor);
    }
    else {
      draw->PushForeground(normalFrameColor);
    }

    draw->PushPen(diceFrame,Lum::OS::DrawInfo::penNormal);
    draw->DrawRectangle(x,y,cellSize,cellSize);
    draw->PopForeground();
    draw->PopPen();

    draw->PushForeground(cellBackgroundColor);
    draw->FillRectangle(x+diceFrame,y+diceFrame,cellSize-2*diceFrame,cellSize-2*diceFrame);
    draw->PopForeground();

    if (!dice->value.empty()) {
      Lum::OS::FontExtent extent;

      font->StringExtent(dice->value,extent,Lum::OS::Font::normal);

      draw->PushFont(font);
      draw->PushForeground(textColor);
      draw->DrawString(x-extent.left+(cellSize-(extent.width-extent.left-extent.right))/2,
                       y+(cellSize-font->height)/2+font->ascent,
                       dice->value);
      draw->PopForeground();
      draw->PopFont();
    }

    if (i%columns==0) {
      x=distance;
      y+=cellSize+distance;
    }
    else {
      x+=cellSize+distance;
    }

    i++;
  }
}

void GameArea::Rebuild(size_t width, size_t height)
{
  CalcAreaSize(width,height);
  RebuildArea();
  if (area!=NULL) {
    RedrawArea();
  }
}

void GameArea::CalcSize()
{
  size_t cellSize=font->height;

  distance=std::max(Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceInterObject),
                    Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceInterObject));

  for (size_t d=1; d<=6; d++) {
    Lum::OS::FontExtent extent;

    font->StringExtent(Lum::Base::NumberToWString(d),extent,Lum::OS::Font::normal);

    cellSize=std::max(cellSize,extent.width);
  }

  cellSize+=2*std::max(Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceObjectBorder),
                       Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder));
  cellSize+=2*diceFrame; // Frame around the dice

  // For finger usablility dices should hvae at least 9 mm size
  cellSize=std::max(cellSize,(size_t)Lum::Base::GetSize(Lum::Base::Size::mm,9));

  // 10 nummerical dices (1-6) should be enough for all of them ;-)
  minWidth=6*distance+5*cellSize;
  minHeight=3*distance+2*cellSize;

  width=minWidth;
  height=minHeight;

  Scrollable::CalcSize();
}

void GameArea::GetDimension(size_t& width, size_t& height)
{
  Rebuild(width,height);

  width=areaWidth;
  height=areaHeight;
}

void GameArea::Layout()
{
  UpdateDimensions();

  Scrollable::Layout();
}

void GameArea::Draw(Lum::OS::DrawInfo* draw,
                    int x, int y, size_t w, size_t h)
{
  Scrollable::Draw(draw,x,y,w,h);

  if (!OIntersect(x,y,w,h)) {
    return;
  }

  /* --- */

  if (area==NULL) {
    return;
  }

  draw->CopyFromBitmap(area,
                       hAdjustment->GetTop()-1,vAdjustment->GetTop()-1,
                       width,height,
                       this->x,this->y);
}

bool GameArea::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (!currentSet.Valid() || currentSet->IsNull() || area==NULL) {
    return false;
  }

  if (event.type==Lum::OS::MouseEvent::down && PointIsIn(event) && event.button==Lum::OS::MouseEvent::button1) {
    size_t                       x,y,i,xi,yi;
    std::list<DiceSet>::iterator iter=sets.begin();

    x=(event.x-this->x+hAdjustment->GetTop()-1);
    y=(event.y-this->y+vAdjustment->GetTop()-1);

    i=1;
    xi=distance;
    yi=distance;

    if (currentSet->Get()>0) {
      std::advance(iter,currentSet->Get()-1);
    }

    for (std::list<Dice>::iterator dice=iter->dices.begin();
         dice!=iter->dices.end();
         dice++) {

      if (x>=xi && x<xi+cellSize && y>=yi && y<yi+cellSize) {
        if (!dice->value.empty()) {
          dice->stays=!dice->stays;
          RedrawArea();
          Redraw();
          currentSet->Notify();
        }
        return true;
      }

      if (i%columns==0) {
        xi=distance;
        yi+=cellSize+distance;
      }
      else {
        xi+=cellSize+distance;
      }

      i++;
    }

    return true;
  }

  return false;
}

void GameArea::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==hAdjustment->GetTopModel() || model==vAdjustment->GetTopModel()) {
    Redraw();
  }
  else if (model==currentSet || model==setsModel) {
    Rebuild(width,height);
    Redraw();
  }

  Lum::Scrollable::Resync(model,msg);
}


