/*
  DiceJinni - A dice simulator
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "DiceEdit.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/Value.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/String.h>
#include <Lum/Table.h>
#include <Lum/Text.h>

class ValuesModelPainter : public Lum::StringCellPainter
{
public:
  std::wstring GetCellData() const
  {
    Value value=dynamic_cast<const ValuesModel*>(GetModel())->GetEntry(GetRow());

    return value.value;
  }
};

class RangeDialog : public Lum::Dlg::ActionDialog
{
private:
  Lum::Model::ULongRef  startModel;
  Lum::Model::ULongRef  endModel;

  Lum::Model::ActionRef okAction;

  bool                  hasResult;

public:
  RangeDialog()
   : startModel(new Lum::Model::ULong(1)),
     endModel(new Lum::Model::ULong(6)),
     okAction(new Lum::Model::Action()),
     hasResult(false)
  {
    Observe(startModel);
    Observe(endModel);
    Observe(okAction);
    Observe(GetClosedAction());
  }

  Lum::Object* GetContent()
  {
    Lum::String *s,*e;

    s=Lum::String::Create(startModel,5,true,false);
    s->SetAlignment(Lum::String::right);
    s->SetInputTypeHint(Lum::Object::inputTypeUnsignedDecimalNumber);
    e=Lum::String::Create(endModel,5,true,false);
    e->SetAlignment(Lum::String::right);
    e->SetInputTypeHint(Lum::Object::inputTypeUnsignedDecimalNumber);

    return Lum::Label::Create(true,false)
                  ->AddLabel(L"Number range:",
                             Lum::HPanel::Create(true,false)
                             ->Add(s)
                             ->AddSpace()
                             ->Add(Lum::Text::Create(L"-"))
                             ->AddSpace()
                             ->Add(e));
  }

  void GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
  {
    Lum::Dlg::ActionDialog::CreateActionInfosOkCancel(actions,okAction,GetClosedAction());
  }

  bool HasResult() {
    return hasResult;
  }

  void GetResult(unsigned long &start, unsigned long& end)
  {
    assert(hasResult);

    start=startModel->Get();
    end=endModel->Get();
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==okAction &&
        okAction->IsFinished()) {
      hasResult=true;
      Exit();
    }
    else if (model==GetClosedAction() &&
             GetClosedAction()->IsFinished()) {
      Exit();
    }
    else if (model==startModel || model==endModel) {
      if (startModel.Valid() &&
          !startModel->IsNull() &&
          endModel.Valid() &&
          !endModel->IsNull() &&
          endModel->Get()>=startModel->Get()) {
        okAction->Enable();
      }
      else {
        okAction->Disable();
      }
    }

    Lum::Dlg::ActionDialog::Resync(model,msg);
  }

  static bool GetRange(Lum::OS::Window* parent, unsigned long& start, unsigned long& end)
  {
    RangeDialog *dlg;
    bool        res;

    dlg=new RangeDialog();
    dlg->SetParent(parent);
    dlg->SetTitle(L""/*caption*/);

    if (dlg->Open()) {
      dlg->EventLoop();
      dlg->Close();
    }

    res=dlg->HasResult();

    if (res) {
      dlg->GetResult(start,end);
    }

    delete dlg;

    return res;
  }
};

DiceEdit::DiceEdit(Dice& dice)
 : dice(dice),
   valuesModel(new ValuesModel(dice.values)),
   selection(new Lum::Model::SingleLineSelection()),
   okAction(new Lum::Model::Action()),
   addAction(new Lum::Model::Action()),
   addNumbersAction(new Lum::Model::Action()),
   editAction(new Lum::Model::Action()),
   removeAction(new Lum::Model::Action()),
   success(false)
{
  EvaluateRights();

  Observe(selection);

  Observe(okAction);
  Observe(addAction);
  Observe(addNumbersAction);
  Observe(editAction);
  Observe(removeAction);

  Observe(GetClosedAction());
}

void DiceEdit::EvaluateRights()
{
  if (valuesModel->GetRows()>0) {
    okAction->Enable();
  }
  else {
    okAction->Disable();
  }

  if (selection.Valid() && selection->HasSelection()) {
    editAction->Enable();
    removeAction->Enable();
  }
  else {
    editAction->Disable();
    removeAction->Disable();
  }
}

Lum::Object* DiceEdit::GetContent()
{
  Lum::Panel            *horiz;
  Lum::Table            *table;
  Lum::Model::HeaderRef headerModel;

  valuesModel->SetEmptyText(L"(No dice values defined)");

  horiz=Lum::HPanel::Create(true,true);

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(L"Value",Lum::Base::Size::stdCharWidth,20,true);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetHeight(Lum::Base::Size::stdCharHeight,8);
  table->SetShowHeader(false);
  table->GetTableView()->SetAutoHSize(true);
  table->GetTableView()->SetAutoFitColumns(true);
  table->SetModel(valuesModel);
  table->SetPainter(new ValuesModelPainter());
  table->SetHeaderModel(headerModel);
  table->SetSelection(selection);
  table->SetDoubleClickAction(editAction);
  horiz->Add(table);

  horiz->AddSpace();

  horiz->Add(Lum::VPanel::Create(false,true)
             ->Add(Lum::Button::Create(_(L"BTN_ADD",L"_Add"),addAction,true,false))
             ->AddSpace()
             ->Add(Lum::Button::Create(_(L"BTN_ADD_NUMBERS",L"Add _numbers"),addNumbersAction,true,false))
             ->AddSpace()
             ->Add(Lum::Button::Create(_(L"BTN_UPDATE",L"_Edit"),editAction,true,false))
             ->AddSpace()
             ->Add(Lum::Button::Create(_(L"BTN_REMOVE",L"_Remove"),removeAction,true,false)));

  return horiz;
}

void DiceEdit::GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
{
  Lum::Dlg::ActionDialog::CreateActionInfosOkCancel(actions,okAction,GetClosedAction());
}

void DiceEdit::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==okAction && okAction->IsFinished()) {
    success=true;
    Exit();
  }
  else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
    Exit();
  }
  else if (model==selection && dynamic_cast<const Lum::Model::Selection::Selected*>(&msg)!=NULL) {
    EvaluateRights();
  }
  else if (model==addAction && addAction->IsFinished()) {
    bool                  result;
    Lum::Model::StringRef value(new Lum::Model::String(L""));

    result=Lum::Dlg::TextInput::GetText(GetWindow(),
                                        L"Enter dice value",
                                        L"Please enter the value for this\n"
                                        L"side of the current dice:",
                                        value);

    if (result && value->Length()>0) {
      Value v;

      v.value=*value;
      valuesModel->Append(v);

      configurationChanged=true;
      EvaluateRights();
    }
  }
  else if (model==addNumbersAction && addNumbersAction->IsFinished()) {
    bool          result=false;
    unsigned long start=0;
    unsigned long end=0;

    result=RangeDialog::GetRange(GetWindow(),start,end);

    if (result) {
      for (size_t i=start; i<=end; i++) {
        Value v;

        v.value=Lum::Base::NumberToWString(i);
        valuesModel->Append(v);
      }
      configurationChanged=true;
      EvaluateRights();
    }
  }
  else if (model==editAction && editAction->IsFinished()  && editAction->IsEnabled()) {
    assert(selection->HasSelection());

    bool                  result;
    Lum::Model::StringRef value(new Lum::Model::String(valuesModel->GetEntry(selection->GetLine()).value));

    result=Lum::Dlg::TextInput::GetText(GetWindow(),
                                        L"Enter dice value",
                                        L"Please enter the value for this\n"
                                        L"side of the current dice:",
                                        value);

    if (result) {
      valuesModel->GetEntry(selection->GetLine()).value=*value;
      valuesModel->RedrawRow(selection->GetLine());

      configurationChanged=true;
      EvaluateRights();
    }
  }
  else if (model==removeAction && removeAction->IsFinished() && removeAction->IsEnabled()) {
    assert(selection->HasSelection());

    valuesModel->Delete(selection->GetLine());
    // Could be the currently displayed one
    setsModel->Notify();

    configurationChanged=true;
    EvaluateRights();
  }

  Lum::Dlg::ActionDialog::Resync(model,msg);
}

bool DiceEdit::GetSuccess() const
{
  return success;
}

