/*
  DiceJinni - A dice simulator
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Configuration.h"

#include <algorithm>
#include <iostream>

#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Config/Config.h>

std::list<DiceSet>   sets;
DiceSetModelRef      setsModel(new DiceSetModel(sets));
Lum::Model::SizeTRef currentSet(new Lum::Model::SizeT());
bool                 configurationChanged=false;

Dice::Dice()
 : canStay(true),
   stays(false)
{
  // no code
}

bool DiceSet::operator<(const DiceSet& set) const
{
  return name<set.name;
}

bool LoadConfig()
{
  Lum::Config::Node      *top;
  Lum::Config::ErrorList errors;
  Lum::Base::Path        path(Lum::Base::Path::GetApplicationConfigPath());

  top=Lum::Config::LoadConfigFromXMLFile(path.GetPath(),errors);

  if (top==NULL) {
    return false;
  }

  if (top->GetName()!=L"DiceJinni") {
    std::cerr << "'" << Lum::Base::WStringToString(path.GetPath()) << "' is a valid config file!" << std::endl;
    delete top;
    return false;
  }

  for (Lum::Config::Node::NodeList::const_iterator setIter=top->GetChildren().begin();
       setIter!=top->GetChildren().end();
       ++setIter) {
    Lum::Config::Node *s=*setIter;

    if (s->GetName()!=L"DiceSet") {
      std::cout << "Unknown tag '" << Lum::Base::WStringToString(s->GetName()) << "', expected 'DiceSet'!" << std::endl;
      continue;
    }

    DiceSet set;

    if (!s->GetAttribute(L"name",set.name)) {
      std::cout << "Cannot read DiceSet.name!" << std::endl;
      continue;
    }

    for (Lum::Config::Node::NodeList::const_iterator diceIter=s->GetChildren().begin();
         diceIter!=s->GetChildren().end();
         ++diceIter) {
      Lum::Config::Node *d=*diceIter;

      if (d->GetName()!=L"Dice") {
        if (d->GetName()!=L"name") {
          std::cout << "Unknown tag '" << Lum::Base::WStringToString(d->GetName()) << "', expected 'Dice'!" << std::endl;
        }
        continue;
      }

      Dice dice;

      if (!d->GetAttribute(L"canStay",dice.canStay)) {
        std::cout << "Cannot read Dice.canStay!" << std::endl;
        continue;
      }

      for (Lum::Config::Node::NodeList::const_iterator valueIter=d->GetChildren().begin();
           valueIter!=d->GetChildren().end();
           ++valueIter) {
        Lum::Config::Node *v=*valueIter;

        if (v->GetName()!=L"Value") {
          if (v->GetName()!=L"canStay") {
            std::cout << "Unknown tag '" << Lum::Base::WStringToString(v->GetName()) << "', expected 'Value'!" << std::endl;
          }
          continue;
        }

        Value value;

        if (!v->GetAttribute(L"value",value.value)) {
          std::cout << "Cannot read Value.value!" << std::endl;
          continue;
        }

        dice.values.push_back(value);
      }

      if (dice.values.size()>0) {
        set.dices.push_back(dice);
      }
    }

    if (set.dices.size()>0) {
      std::cout << "Loaded dice set '" << Lum::Base::WStringToString(set.name) << "'" << std::endl;
      sets.push_back(set);
    }
  }

  delete top;

  // TODO: Sort list by name

 sets.sort();

  return true;
}

bool SaveConfig()
{
  Lum::Config::Node *top;
  Lum::Base::Path   path(Lum::Base::Path::GetApplicationConfigPath());
  std::wstring      config;
  bool              res;

  top=new Lum::Config::Node();
  top->SetName(L"DiceJinni");

  for (std::list<DiceSet>::const_iterator set=sets.begin();
       set!=sets.end();
       set++) {
    Lum::Config::Node *s;

    s=new Lum::Config::Node(L"DiceSet");
    s->SetAttribute(L"name",set->name);

    for (std::list<Dice>::const_iterator dice=set->dices.begin();
         dice!=set->dices.end();
         dice++) {
      Lum::Config::Node *d;

      d=new Lum::Config::Node(L"Dice");
      d->SetAttribute(L"canStay",dice->canStay);


      for (std::vector<Value>::const_iterator value=dice->values.begin();
           value!=dice->values.end();
           value++) {
        Lum::Config::Node *v;

        v=new Lum::Config::Node(L"Value");
        v->SetAttribute(L"value",value->value);

        d->Add(v);
      }

      s->Add(d);
    }

    top->Add(s);
  }

  Lum::Base::Status status;

  status=path.CreateDirRecursive();

  if (!status) {
    std::cerr << "Cannot create config directory '" << Lum::Base::WStringToString(path.GetDir()) << "': " << Lum::Base::WStringToString(status.GetDescription()) << std::endl;
    return false;
  }

  res=Lum::Config::SaveConfigToXMLFile(path.GetPath(),top);

  delete top;

  return res;
}


