/*
  DiceJinni - A dice simulator
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <time.h>

#include <cstdlib>

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Def/Menu.h>

#include <Lum/Button.h>
#include <Lum/Combo.h>
#include <Lum/Dialog.h>
#include <Lum/Panel.h>
#include <Lum/View.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Model/Action.h>

#include <Lum/OS/Main.h>

#include "Configuration.h"
#include "DiceSetList.h"
#include "GameArea.h"

#include "config.h"

#include <iostream>

static Lum::Def::AppInfo info;

class MainDialog : public Lum::Dialog
{
private:
  Lum::Model::ActionRef configureAction;
  Lum::Model::ActionRef throwAction;
  Lum::Model::ActionRef rethrowAction;
  Lum::Model::ActionRef aboutAction;

public:
  MainDialog()
  : configureAction(new Lum::Model::Action),
    throwAction(new Lum::Model::Action),
    rethrowAction(new Lum::Model::Action),
    aboutAction(new Lum::Model::Action)
  {
    currentSet->Disable();

    Observe(currentSet);

    Observe(configureAction);
    Observe(throwAction);
    Observe(rethrowAction);
    Observe(aboutAction);
    Observe(GetOpenedAction());
    Observe(GetClosedAction());
  }

  void PreInit()
  {
    Lum::Combo  *combo;
    Lum::Panel  *vert;

    vert=Lum::VPanel::Create(true,true);

    combo=new Lum::IndexCombo();
    combo->SetMinWidth(Lum::Base::Size::stdCharWidth,20);
    combo->SetFlex(true,false);
    combo->SetTableModel(setsModel);
    combo->SetModel(currentSet);
    vert->Add(combo);

    vert->AddSpace();
    vert->Add(Lum::HPanel::Create(true,true)
              ->Add(Lum::View::Create(new GameArea(true,true), true,true))
              ->AddSpace()
              ->Add(Lum::VPanel::Create(false,true)
                    ->Add(Lum::Button::Create(L"_Throw",throwAction,true,false))
                    ->AddSpace()
                    ->Add(Lum::Button::Create(L"_Rethrow",rethrowAction,true,false))
                    ->AddSpace(true)
                    ->Add(Lum::Button::Create(L"_Configure",configureAction,true,false))));

    SetMain(vert);

    Lum::Def::Menu *menu=Lum::Def::Menu::Create();

    menu
      ->GroupProject()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->GroupEdit()
        ->ActionSettings(configureAction)
      ->End()
      ->GroupHelp()
        //->Action(Lum::Def::Action(Lum::Def::Desc(_ld(menuHelpHelp)),NULL))
        ->ActionAbout(aboutAction)
      ->End();

    SetMenu(menu);

    Dialog::PreInit();
  }

  void Throw(bool all)
  {
    if (!currentSet.Valid() || currentSet->IsNull()) {
      return;
    }

    std::list<DiceSet>::iterator iter=sets.begin();

    if (*currentSet>0) {
      std::advance(iter,*currentSet-1);
    }

    for (std::list<Dice>::iterator dice=iter->dices.begin();
         dice!=iter->dices.end();
         dice++) {
      if (dice->values.size()>0 && (!dice->stays || (dice->stays && all))) {
        size_t pos=rand()%dice->values.size();

        dice->value=dice->values[pos].value;
      }

      if (all) {
        dice->stays=false;
      }
    }

    currentSet->Notify();
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      LoadConfig();

      if (sets.size()>0) {
        currentSet->Set(1);
        currentSet->Enable();
      }
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      if (configurationChanged && !SaveConfig()) {
        Lum::Dlg::Msg::ShowOk(this,
                             L"Error while storing configuration...",
                             L"There was an error while storing the configuration!");
      }
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==throwAction && throwAction->IsFinished()) {
      Throw(true);
    }
    else if (model==rethrowAction && rethrowAction->IsFinished()) {
      Throw(false);
    }
    else if (model==configureAction && configureAction->IsFinished()) {
      DiceSetList *dialog;

      dialog=new DiceSetList();
      dialog->SetParent(this);
      if (dialog->Open()) {
        dialog->EventLoop();
        dialog->Close();
      }

      delete dialog;

      if (sets.size()>0) {
        currentSet->Set(1);
        currentSet->Enable();
      }
      else {
        currentSet->SetNull();
        currentSet->Disable();
      }
    }
    else if (model==currentSet) {
      if (currentSet.Valid() && !currentSet->IsNull()) {
        throwAction->Enable();
        rethrowAction->Enable();
      }
      else {
        throwAction->Disable();
        rethrowAction->Disable();
      }
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MainDialog>
{
public:
  Main()
  {
    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(L"Role the dice...");
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2008, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    srand(time(NULL));
  }

  bool Prepare()
  {
    return Lum::OS::MainDialog<MainDialog>::Prepare();
  }

  void Cleanup()
  {
    Lum::OS::MainDialog<MainDialog>::Cleanup();
  }
};

LUM_MAIN(Main,Lum::Base::StringToWString(PACKAGE_NAME))

