#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "desktop-cmd-exec.h"
#include <hildon/hildon.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <libintl.h>
#include <stdlib.h>

#define HOME_DIR g_get_home_dir()

#define _(String) dgettext("hildon-libs", String)

#define DESKTOP_CMD_EXEC_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE (obj, DESKTOP_CMD_EXEC_TYPE, DesktopCmdExecPrivate))

//main settings window
#define NON_GTK_RESPONSE_ADD_CMD 10
#define NON_GTK_RESPONSE_EDIT_CMD 20
//save is GTK_RESPONSE_OK

//Add/Edit/Delete window
#define NON_GTK_RESPONSE_DELETE_CMD 30
//save is GTK_RESPONSE_OK

void desktop_cmd_exec_write_settings (DesktopCmdExec *self, gboolean reset);
gboolean desktop_cmd_exec_update_content (DesktopCmdExec *self);

//Dialogs prototypes:
void desktop_cmd_exec_edit_add_dialog ( DesktopCmdExec *self, gboolean new);
void desktop_cmd_exec_settings (HDHomePluginItem *hitem, DesktopCmdExec *self);

//Auxiliary functions' prototypes:
void AddCommand(DesktopCmdExec *self, gchar * s_title, gchar * s_command);
void EditCommand(DesktopCmdExec *self, gchar * s_title, gchar * s_command, int index);
void DelCommand(DesktopCmdExec *self, int index);
guint GetSeconds(guint index);

struct _DesktopCmdExecPrivate
{

	GtkWidget *homeWidget;
	GtkWidget *event;
	GtkWidget *contents;
	
	GtkWidget *cmdTitle_lb;
	GtkWidget *cmdResult_lb;
	
	gboolean isPressed;

	/* new */
	gchar ** c_titles;
	gchar ** c_commands;
	guint c_size;
	guint c_current;
	
	gboolean updOnClick;
	gboolean updOnDesktop;

	//Update on timer data
	guint delaySeconds;
	guint delayIndex;//index of delaySeconds in the array defined inside "desktop_cmd_exec_settings"

	//timer process id; used to stop when closing app or changing delay
	guint updateTimerID;
};

HD_DEFINE_PLUGIN_MODULE (DesktopCmdExec, desktop_cmd_exec, HD_TYPE_HOME_PLUGIN_ITEM);


//Settings file functions
void desktop_cmd_exec_read_settings ( DesktopCmdExec *self )
{
// 	g_warning("desktop_cmd_exec_read_settings ");
	gchar *filename;
	gboolean fileExists;
	GKeyFile *keyFile;

	gboolean gotAnErrorFlag;
	gotAnErrorFlag = FALSE;

	keyFile = g_key_file_new();
	filename = g_strconcat (HOME_DIR, "/.desktop_cmd_exec", NULL);
	fileExists = g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);


	if (fileExists) {
		GError *error=NULL;

		//BEGIN get list
		self->priv->c_commands = (gchar **) g_key_file_get_string_list (keyFile, "config", "c_commands", &(self->priv->c_size) ,&error);
		if (error) {
			gotAnErrorFlag = TRUE;
			g_error_free (error);
			error = NULL;
		}

		gint consistencyCheck = -1;
		self->priv->c_titles = g_key_file_get_string_list (keyFile, "config", "c_titles", &consistencyCheck ,&error);
		if (error) {
			gotAnErrorFlag = TRUE;
			g_error_free (error);
			error = NULL;
		}

		self->priv->c_current = g_key_file_get_integer (keyFile, "config", "c_current", &error);
		if (error) {
			gotAnErrorFlag = TRUE;
			g_error_free (error);
			error = NULL;
		}

		if(consistencyCheck != self->priv->c_size)
		{
			g_warning("Settings file corrupted!");
			gotAnErrorFlag = TRUE;
		}
		//END


		//BEGIN update policy
		self->priv->updOnClick = g_key_file_get_boolean (keyFile, "config", "updOnClick", &error);
		if (error) {

			gotAnErrorFlag = TRUE;
			g_error_free (error);
			error = NULL;
		}

		self->priv->updOnDesktop = g_key_file_get_boolean (keyFile, "config", "updOnDesktop", &error);
		if (error) {

			gotAnErrorFlag = TRUE;
			g_error_free (error);
			error = NULL;
		}

		self->priv->delayIndex = (guint) g_key_file_get_integer (keyFile, "config", "delayIndex", &error);
		if (error) {

			gotAnErrorFlag = TRUE;
			g_error_free (error);
			error = NULL;
		}
		else
			self->priv->delaySeconds = GetSeconds(self->priv->delayIndex);
		
		//END
		g_key_file_free (keyFile);
		g_free (filename);
	}
	else
		gotAnErrorFlag = TRUE;

	if(gotAnErrorFlag)
	{
// 		g_warning ("Something happened... resetting all");
		self->priv->updOnClick = TRUE;
		self->priv->updOnDesktop = TRUE;
		self->priv->delayIndex = 0;
		self->priv->delaySeconds = 0;

		self->priv->c_commands = NULL;
		self->priv->c_titles = NULL;
		self->priv->c_size = 0;
		self->priv->c_current = 0;
		desktop_cmd_exec_write_settings(self, TRUE);
	}
}

void desktop_cmd_exec_write_settings (DesktopCmdExec *self, gboolean reset)
{
//  	g_warning ("desktop_cmd_exec_write_settings");
	GKeyFile *keyFile;
	gchar *fileData;
	FILE *iniFile;
	gsize size;
	gchar *filename;
	
	keyFile = g_key_file_new();

	if(reset)
	{
// 		g_warning("records empty, filling with pre-loaded vals");
		//records empty, filling with pre-loaded vals
		self->priv->c_size = 4;
		gchar * p_titles[] = { "Uptime:", "Battery(%):", "Battery(mAh):", "Boot Reason:", NULL};
		gchar * p_commands[] = {
		"uptime | awk 'NF == 7 {print $2 \" \" $3}; NF == 8 {print $2 \" \" $3}; NF == 11 {print $3 \" \" $4 \" \" $5}'",
		"hal-device | grep percen | awk '{print $3 \"%\"}'",
		"hal-device | grep battery.reporting | awk -F. '{print $3}' | awk '$1 == \"current\" { curval = $3}; $1 == \"design\" {print curval \"/\" $3}'",
		"cat /proc/bootreason", NULL};

		//clean possible oldies
		g_strfreev(self->priv->c_titles);
		g_strfreev(self->priv->c_commands);

		//assign new from stack values
		self->priv->c_titles = g_strdupv(p_titles);
		self->priv->c_commands = g_strdupv(p_commands);
		
		self->priv->c_current = 2;

		self->priv->updOnClick = TRUE;
		self->priv->updOnDesktop = TRUE;
		self->priv->delayIndex = 0;
		
// 		g_warning("Diiiiid it");
	}
	
	g_key_file_set_string_list(keyFile, "config", "c_titles", (const gchar **)(self->priv->c_titles),self->priv->c_size);
	g_key_file_set_string_list(keyFile, "config", "c_commands", (const gchar **)(self->priv->c_commands),self->priv->c_size);
	g_key_file_set_integer(keyFile, "config", "c_current", self->priv->c_current);

	g_key_file_set_boolean (keyFile, "config", "updOnClick", self->priv->updOnClick);
	g_key_file_set_boolean (keyFile, "config", "updOnDesktop", self->priv->updOnDesktop);
	g_key_file_set_integer (keyFile, "config", "delayIndex", self->priv->delayIndex);
	
	filename = g_strconcat (HOME_DIR, "/.desktop_cmd_exec", NULL);
	fileData = g_key_file_to_data (keyFile, &size, NULL);
	iniFile = fopen (filename, "w");
	fputs (fileData, iniFile);
	fclose (iniFile);
	g_key_file_free (keyFile);
	g_free (fileData);
	g_free (filename);
	if(reset)
		desktop_cmd_exec_read_settings(self);
}



//widget/touchscreen interaction callbacks
void desktop_cmd_exec_button_press (GtkWidget *widget, GdkEventButton *event, DesktopCmdExec *self)
{
// 	g_warning ("desktop_cmd_exec_button_press");
	if (self->priv->updOnClick) {
		self->priv->isPressed = TRUE;
		gtk_widget_queue_draw (GTK_WIDGET (self));
	}
}

void desktop_cmd_exec_button_release (GtkWidget *widget, GdkEventButton *event, DesktopCmdExec *self)
{
// 	g_warning ("desktop_cmd_exec_button_release");
	if (self->priv->updOnClick) {
		self->priv->isPressed = FALSE;
		gtk_widget_queue_draw (GTK_WIDGET (self));
		desktop_cmd_exec_update_content(self);
	}
}

void desktop_cmd_exec_leave_event (GtkWidget *widget, GdkEventCrossing *event, DesktopCmdExec *self)
{
// 	g_warning ("desktop_cmd_exec_leave_event");
	if (self->priv->updOnClick) {
		self->priv->isPressed = FALSE;
		gtk_widget_queue_draw (GTK_WIDGET (self));
	}
}

static void desktop_cmd_exec_check_desktop (GObject *gobject, GParamSpec *pspec, DesktopCmdExec *self)
{
	if (self->priv->updOnDesktop) {
		desktop_cmd_exec_update_content(self);
	}
	// 	g_warning ("desktop_cmd_exec_check_desktop");
}



//Content/widget creation/update
void desktop_cmd_exec_content_create (DesktopCmdExec *self)
{
// 	g_warning ("desktop_cmd_exec_content_create");
	self->priv->contents = gtk_event_box_new ();
	gtk_event_box_set_visible_window (GTK_EVENT_BOX (self->priv->contents), FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (self->priv->contents), 0);
// 	GtkSizeGroup *group = GTK_SIZE_GROUP (gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL));

	self->priv->cmdTitle_lb = gtk_label_new ("title:");
	self->priv->cmdResult_lb = gtk_label_new ("result");
	
	GtkWidget *box = gtk_hbox_new (FALSE, 0);
	
	gtk_box_pack_start (GTK_BOX (box), self->priv->cmdTitle_lb, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (box), self->priv->cmdResult_lb, TRUE, TRUE, 0);
	
	hildon_helper_set_logical_color (self->priv->cmdResult_lb, GTK_RC_FG, GTK_STATE_NORMAL, "ActiveTextColor");

	gtk_misc_set_alignment (GTK_MISC (self->priv->cmdTitle_lb), 0, 1);
	gtk_misc_set_alignment (GTK_MISC (self->priv->cmdResult_lb), 1, 1);

	gtk_misc_set_padding (GTK_MISC (self->priv->cmdTitle_lb), HILDON_MARGIN_DEFAULT, HILDON_MARGIN_HALF);
	gtk_misc_set_padding (GTK_MISC (self->priv->cmdResult_lb), HILDON_MARGIN_DEFAULT, HILDON_MARGIN_HALF);

	gtk_container_add (GTK_CONTAINER (self->priv->contents), box);
	gtk_box_pack_start (GTK_BOX (self->priv->homeWidget), self->priv->contents, FALSE, FALSE, 0);

	//widget drawing signal connections
	g_signal_connect (self->priv->contents, "button-release-event", G_CALLBACK (desktop_cmd_exec_button_release), self);
	g_signal_connect (self->priv->contents, "button-press-event", G_CALLBACK (desktop_cmd_exec_button_press), self);
	g_signal_connect (self->priv->contents, "leave-notify-event", G_CALLBACK (desktop_cmd_exec_leave_event), self);
	
	gtk_widget_show_all (self->priv->homeWidget);
	gtk_widget_show (self->priv->cmdTitle_lb);
}



gboolean desktop_cmd_exec_update_content (DesktopCmdExec *self)
{
// 	g_warning ("desktop_cmd_exec_update_content");
	FILE *fp;
	gchar line[256];
	gchar *result;
	
	gboolean found = FALSE;
	if(self->priv->c_current >= self->priv->c_size)
	{
		gtk_label_set_text (GTK_LABEL (self->priv->cmdTitle_lb), "Error:");
		gtk_label_set_text (GTK_LABEL (self->priv->cmdResult_lb), "No commands");
		return 0;
	}
// 	g_warning("To exec %s", self->priv->c_titles[self->priv->c_current]);

	gtk_label_set_text (GTK_LABEL (self->priv->cmdTitle_lb), self->priv->c_titles[self->priv->c_current]);

	fp = popen (self->priv->c_commands[self->priv->c_current], "r");
	
	while (fgets (line, sizeof line, fp)) {
		result = g_strstrip (line);
		gtk_label_set_text (GTK_LABEL (self->priv->cmdResult_lb), result);
		found = TRUE;
	}
	pclose (fp);
	
	if (!found) {
		gtk_label_set_text (GTK_LABEL (self->priv->cmdResult_lb), "Invalid Command");
	}

	if( (self->priv->updateTimerID == 0) && (self->priv->delaySeconds > 0))
	{
		self->priv->updateTimerID = g_timeout_add_seconds (self->priv->delaySeconds, (GSourceFunc)desktop_cmd_exec_update_content, self);
	}
	
	return found;
}




static void desktop_cmd_exec_dispose (GObject *object)
{
// 	g_warning ("desktop_cmd_exec_dispose");
// 	DesktopCmdExec *self = DESKTOP_CMD_EXEC (object);

	G_OBJECT_CLASS (desktop_cmd_exec_parent_class)->dispose (object);
}

static void desktop_cmd_exec_finalize (GObject *object)
{
// 	g_warning ("desktop_cmd_exec_finalize");
	DesktopCmdExec *self = DESKTOP_CMD_EXEC (object);

	if (self->priv->updateTimerID) {
		g_source_remove (self->priv->updateTimerID);
	}

	g_strfreev(self->priv->c_titles);
	g_strfreev(self->priv->c_commands);

	G_OBJECT_CLASS (desktop_cmd_exec_parent_class)->finalize (object);
}

static void desktop_cmd_exec_realize (GtkWidget *widget)
{
// 	g_warning ("desktop_cmd_exec_realize");
	GdkScreen *screen;

	screen = gtk_widget_get_screen (widget);
	gtk_widget_set_colormap (widget, gdk_screen_get_rgba_colormap (screen));

	gtk_widget_set_app_paintable (widget, TRUE);

	GTK_WIDGET_CLASS (desktop_cmd_exec_parent_class)->realize (widget);
}

static gboolean desktop_cmd_exec_expose_event (GtkWidget *widget, GdkEventExpose *event)
{
// 	g_warning ("desktop_cmd_exec_expose_event");
	DesktopCmdExec *self = DESKTOP_CMD_EXEC (widget);

	cairo_t *cr;
	
	cr = gdk_cairo_create(GDK_DRAWABLE (widget->window));

	GdkColor color;
	if (!self->priv->isPressed) {
		gtk_style_lookup_color (gtk_rc_get_style(widget), "DefaultBackgroundColor", &color);
		cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.75);
	} else {
		gtk_style_lookup_color (gtk_rc_get_style(widget), "SelectionColor", &color);
		cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.6);
	}
	
	gint width, height, x, y;
	gint radius = 5;
	width = widget->allocation.width;
	height = widget->allocation.height;
	x = widget->allocation.x;
	y = widget->allocation.y;

	cairo_move_to(cr, x + radius, y);
	cairo_line_to(cr, x + width - radius, y);
	cairo_curve_to(cr, x + width - radius, y, x + width, y, x + width,y + radius);
	cairo_line_to(cr, x + width, y + height - radius);
	cairo_curve_to(cr, x + width, y + height - radius, x + width,y + height, x + width - radius, y + height);
	cairo_line_to(cr, x + radius, y + height);
	cairo_curve_to(cr, x + radius, y + height, x, y + height, x,y + height - radius);
	cairo_line_to(cr, x, y + radius);
	cairo_curve_to(cr, x, y + radius, x, y, x + radius, y);

	cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);

	cairo_fill_preserve(cr);
	
	gtk_style_lookup_color (gtk_rc_get_style(widget), "ActiveTextColor", &color);
	cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.5);
	cairo_set_line_width (cr, 1);
	cairo_stroke (cr);
	
	cairo_destroy(cr);

	return GTK_WIDGET_CLASS (desktop_cmd_exec_parent_class)->expose_event (widget, event);
}

static void desktop_cmd_exec_class_init (DesktopCmdExecClass *klass)
{
// 	g_warning ("desktop_cmd_exec_class_init");
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	object_class->dispose = desktop_cmd_exec_dispose;
	object_class->finalize = desktop_cmd_exec_finalize;
	
	widget_class->realize = desktop_cmd_exec_realize;
	widget_class->expose_event = desktop_cmd_exec_expose_event;

	g_type_class_add_private (klass, sizeof (DesktopCmdExecPrivate));
}

static void desktop_cmd_exec_class_finalize (DesktopCmdExecClass *klass G_GNUC_UNUSED)
{
}

static void desktop_cmd_exec_init (DesktopCmdExec *self)
{
// 	g_warning ("desktop_cmd_exec_init");
	self->priv = DESKTOP_CMD_EXEC_GET_PRIVATE (self);
	self->priv->updateTimerID = 0;
	self->priv->isPressed = FALSE;
	
	hd_home_plugin_item_set_settings (&self->parent, TRUE);
	g_signal_connect (&self->parent, "show-settings", G_CALLBACK (desktop_cmd_exec_settings), self);

	gtk_window_set_default_size (GTK_WINDOW (self), 800, 40);
	
	self->priv->homeWidget = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (self), self->priv->homeWidget);
	gtk_widget_show (self->priv->homeWidget);

	GdkGeometry hints;
	hints.min_width = 800;
	hints.min_height = 40;
	hints.max_width = 800;
	hints.max_height = 40;
	gtk_window_set_geometry_hints (GTK_WINDOW (self), self->priv->homeWidget, &hints, GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);
	
	desktop_cmd_exec_read_settings (self);
	gtk_widget_set_size_request (GTK_WIDGET (self), 800, 40);
	gtk_window_resize (GTK_WINDOW (self), 800, 40);

	
	desktop_cmd_exec_content_create (self);
	desktop_cmd_exec_update_content (self);
	
	g_signal_connect (self, "notify::is-on-current-desktop", G_CALLBACK (desktop_cmd_exec_check_desktop), self);
}

DesktopCmdExec* desktop_cmd_exec_new (void)
{
	return g_object_new (DESKTOP_CMD_EXEC_TYPE, NULL);
}


//Separate window dialogs
void desktop_cmd_exec_settings (HDHomePluginItem *hitem, DesktopCmdExec *self)
{
	// 	g_warning ("desktop_cmd_exec_settings");
	
	int settingsRunning = 1;
	while(settingsRunning)
	{
		GtkWidget *dialog = gtk_dialog_new_with_buttons ("Desktop Command Execution Widget Settings", NULL, 0,
																										 _("wdgt_bd_save"), GTK_RESPONSE_ACCEPT,
																										 "Add Cmd", NON_GTK_RESPONSE_ADD_CMD,
																										 "Edit Cmd", NON_GTK_RESPONSE_EDIT_CMD,
																										 NULL);

		GtkWidget *content_area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
		GtkSizeGroup *group = GTK_SIZE_GROUP (gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL));

		GtkWidget *cmdSelectionLabel = gtk_label_new ("");
		gtk_label_set_markup (GTK_LABEL (cmdSelectionLabel), "<small>Command Selection:</small>");
		gtk_container_add (GTK_CONTAINER (content_area), cmdSelectionLabel);

		//BEGIN Command Selector
		
		GtkWidget *selector = hildon_touch_selector_new_text ();
		int i;
		for(i=0;i<self->priv->c_size;i++)
		{
			hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), self->priv->c_titles[i]);
		}
		hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (selector), 0, self->priv->c_current);


		GtkWidget * cmdSelector = hildon_picker_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
		hildon_button_set_title (HILDON_BUTTON (cmdSelector), "Commands: ");
		hildon_button_add_title_size_group (HILDON_BUTTON (cmdSelector), group);
		hildon_button_set_alignment (HILDON_BUTTON (cmdSelector), 0, 0.5, 0, 0);
		hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (cmdSelector), HILDON_TOUCH_SELECTOR (selector));
		gtk_container_add (GTK_CONTAINER (content_area), cmdSelector);
		//END

		//BEGIN Update Policy
		GtkWidget *updPolicyLabel = gtk_label_new ("");
		gtk_label_set_markup (GTK_LABEL (updPolicyLabel), "<small>Update Policy:</small>");
		gtk_container_add (GTK_CONTAINER (content_area), updPolicyLabel);
		
		GtkWidget *checkBtClick = hildon_check_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
		gtk_button_set_label (GTK_BUTTON (checkBtClick), "Update when clicked");
		gtk_container_add (GTK_CONTAINER (content_area), checkBtClick);
		hildon_check_button_set_active (HILDON_CHECK_BUTTON (checkBtClick), self->priv->updOnClick);

		GtkWidget *checkBtDesktop = hildon_check_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
		gtk_button_set_label (GTK_BUTTON (checkBtDesktop), "Update when switched to the desktop");
		gtk_container_add (GTK_CONTAINER (content_area), checkBtDesktop);
		hildon_check_button_set_active (HILDON_CHECK_BUTTON (checkBtDesktop), self->priv->updOnDesktop);

		GtkWidget *intervalSelector = hildon_touch_selector_new_text ();
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (intervalSelector), "0");
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (intervalSelector), "30 Seconds");
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (intervalSelector), "1 Minute");
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (intervalSelector), "5 Minutes");
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (intervalSelector), "30 Minutes");
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (intervalSelector), "1 Hour");
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (intervalSelector), "6 Hours");
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (intervalSelector), "12 Hours");
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (intervalSelector), "Daily");
		
		hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (intervalSelector), 0, self->priv->delayIndex);
		
		GtkWidget * intervalSelBt = hildon_picker_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
		hildon_button_set_title (HILDON_BUTTON (intervalSelBt), "Update Interval: ");
		hildon_button_add_title_size_group (HILDON_BUTTON (intervalSelBt), group);
		hildon_button_set_alignment (HILDON_BUTTON (intervalSelBt), 0, 0.5, 0, 0);
		hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (intervalSelBt), HILDON_TOUCH_SELECTOR (intervalSelector));
		gtk_container_add (GTK_CONTAINER (content_area), intervalSelBt);

		
		GtkWidget *updIntervalLabel = gtk_label_new ("");
		gtk_label_set_markup (GTK_LABEL (updIntervalLabel), "<small>'0' disables periodic updating</small>");
		gtk_container_add (GTK_CONTAINER (content_area), updIntervalLabel);
		//END

		gtk_widget_show_all (dialog);

		//BEGIN result processing

		int setDialogReturn = 0;

		setDialogReturn = gtk_dialog_run (GTK_DIALOG (dialog));
		if (setDialogReturn == GTK_RESPONSE_ACCEPT)
		{
			int cmdSel = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (selector), 0);
			if(cmdSel < self->priv->c_size && cmdSel >= 0)
			{
				self->priv->c_current = (gint) cmdSel;
			}
			self->priv->updOnClick = hildon_check_button_get_active (HILDON_CHECK_BUTTON (checkBtClick));
			self->priv->updOnDesktop = hildon_check_button_get_active (HILDON_CHECK_BUTTON (checkBtDesktop));
			
			self->priv->delayIndex = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (intervalSelector), 0);
			self->priv->delaySeconds = GetSeconds(self->priv->delayIndex);
			
			if (self->priv->updateTimerID != 0) {
				g_source_remove (self->priv->updateTimerID);
				self->priv->updateTimerID = 0;
			}
			else
			

			desktop_cmd_exec_write_settings (self, FALSE);
			desktop_cmd_exec_update_content (self);
			settingsRunning = 0;
		}
		else if(setDialogReturn == NON_GTK_RESPONSE_ADD_CMD)
		{
			desktop_cmd_exec_edit_add_dialog(self, 1);
		}
		else if(setDialogReturn == NON_GTK_RESPONSE_EDIT_CMD)
		{
			int cmdSel = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (selector), 0);
			if(cmdSel < self->priv->c_size && cmdSel >= 0)
			{
				self->priv->c_current = (gint) cmdSel;
			}
			desktop_cmd_exec_edit_add_dialog(self,0);
		}
		else
			settingsRunning = 0;
		
		gtk_widget_destroy (dialog);

		//END
	}
}

void desktop_cmd_exec_edit_add_dialog ( DesktopCmdExec *self, gboolean new)
{
	GtkWidget *dialog;
	gchar * title;
	if(new)
		title = g_strconcat ("Add new command", NULL);
	else
		title = g_strconcat ("Edit ", self->priv->c_titles[self->priv->c_current] ," command", NULL);

	if(new)
		dialog = gtk_dialog_new_with_buttons (title, NULL, 0, _("wdgt_bd_save"), GTK_RESPONSE_ACCEPT, NULL);
	else
		dialog = gtk_dialog_new_with_buttons (title, NULL, 0, _("wdgt_bd_save"), GTK_RESPONSE_ACCEPT, _("wdgt_bd_delete"), NON_GTK_RESPONSE_DELETE_CMD, NULL);
	
	GtkWidget *content_area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
	GtkSizeGroup *group = GTK_SIZE_GROUP (gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL));

	//BEGIN title
	GtkWidget *titleBox = gtk_hbox_new (FALSE, 0);
	
	GtkWidget *titleLabel = gtk_label_new ("Title:");
	gtk_misc_set_alignment (GTK_MISC (titleLabel), 0, 0.5);
	gtk_misc_set_padding (GTK_MISC (titleLabel), HILDON_MARGIN_DOUBLE, 0);
	gtk_size_group_add_widget (group, titleLabel);
	
	GtkWidget *titleEntry = hildon_entry_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	gtk_box_pack_start (GTK_BOX (titleBox), titleLabel, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (titleBox), titleEntry, TRUE, TRUE, 0);
	if(!new)
		hildon_entry_set_text(HILDON_ENTRY (titleEntry), self->priv->c_titles[self->priv->c_current]);
	
	gtk_container_add (GTK_CONTAINER (content_area), titleBox);
	//END

	//BEGIN command
	GtkWidget *commandBox = gtk_hbox_new (FALSE, 0);
	
	GtkWidget *commandLabel = gtk_label_new ("Command:");
	gtk_misc_set_alignment (GTK_MISC (commandLabel), 0, 0.5);
	gtk_misc_set_padding (GTK_MISC (commandLabel), HILDON_MARGIN_DOUBLE, 0);
	gtk_size_group_add_widget (group, commandLabel);
	
	GtkWidget *commandEntry = hildon_entry_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	gtk_box_pack_start (GTK_BOX (commandBox), commandLabel, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (commandBox), commandEntry, TRUE, TRUE, 0);
	if(!new)
		hildon_entry_set_text(HILDON_ENTRY (commandEntry), self->priv->c_commands[self->priv->c_current]);
	
	gtk_container_add (GTK_CONTAINER (content_area), commandBox);
	//END

	gtk_widget_show_all (dialog);
	int dialogRunResponse = gtk_dialog_run (GTK_DIALOG (dialog));
	switch(dialogRunResponse)
	{
		case GTK_RESPONSE_ACCEPT:
			if(new)
			{
				AddCommand(self, g_strdup(hildon_entry_get_text (HILDON_ENTRY (titleEntry))), g_strdup(hildon_entry_get_text (HILDON_ENTRY (commandEntry))));
			}
			else
			{
				EditCommand(self, g_strdup(hildon_entry_get_text (HILDON_ENTRY (titleEntry))), g_strdup(hildon_entry_get_text (HILDON_ENTRY (commandEntry))), self->priv->c_current);
			}
			break;

		case NON_GTK_RESPONSE_DELETE_CMD:
			DelCommand(self,self->priv->c_current);
			break;

	}
	gtk_widget_destroy (dialog);
	g_free(title);
}





//Auxiliary functions
void AddCommand(DesktopCmdExec *self, gchar * s_title, gchar * s_command)
{
	gchar ** newTitles = malloc( sizeof(gchar*)*(self->priv->c_size+2) );
	gchar ** newCommands = malloc( sizeof(gchar*)*(self->priv->c_size+2) );
	
	int i;
	for(i=0;i<self->priv->c_size;i++)
	{
		newTitles[i] = g_strdup(self->priv->c_titles[i]);
		newCommands[i] = g_strdup(self->priv->c_commands[i]);
	}
	
	g_strfreev(self->priv->c_titles);
	g_strfreev(self->priv->c_commands);
	
	newTitles[self->priv->c_size] = s_title;
	newCommands[self->priv->c_size] = s_command;

	newTitles[self->priv->c_size+1] = NULL;
	newCommands[self->priv->c_size+1] = NULL;
	
	self->priv->c_current = self->priv->c_size;
	self->priv->c_size += 1;
	
	self->priv->c_titles = newTitles;
	self->priv->c_commands = newCommands;
}

void EditCommand(DesktopCmdExec *self, gchar * s_title, gchar * s_command, int index)
{
	//Clean old
	g_free(self->priv->c_titles[index]);
	g_free(self->priv->c_commands[index]);

	self->priv->c_titles[index] = s_title;
	self->priv->c_commands[index] = s_command;
}

void DelCommand(DesktopCmdExec *self, int index)
{
	gchar ** newTitles = malloc( sizeof(gchar*)*(self->priv->c_size) );
	gchar ** newCommands = malloc( sizeof(gchar*)*(self->priv->c_size) );
	g_warning("mallocated %d", self->priv->c_size);

	int i;
	for(i=0;i<self->priv->c_size;i++)
	{
		g_warning("gathering..%d", i);
		//index is the delete target
		if(i < index)
		{
			newTitles[i] = g_strdup(self->priv->c_titles[i]);
			newCommands[i] = g_strdup(self->priv->c_commands[i]);
		}
		else if(i > index)
		{
			newTitles[i-1] = g_strdup(self->priv->c_titles[i]);
			newCommands[i-1] = g_strdup(self->priv->c_commands[i]);
			g_warning("wrote at %d",i-1);
		}
	}
	g_warning("ok.. no prob");
	newTitles[self->priv->c_size-1] = NULL;
	newCommands[self->priv->c_size-1] = NULL;
	g_warning("nulled %d", self->priv->c_size-1);

	//clear old
	g_strfreev(self->priv->c_titles);
	g_strfreev(self->priv->c_commands);
	g_warning("cleared ok");

	//assign new
	self->priv->c_titles = newTitles;
	self->priv->c_commands = newCommands;

	self->priv->c_current = 0;
	self->priv->c_size -= 1;//decrement size
	g_warning("end of del");
}

guint GetSeconds(guint index)
{
	if(index > 0 && index < 9)
	{
		guint intervalTimes[] = {0, 30, 60, 300, 1800, 3600, 21600, 43200, 86400};
		return intervalTimes[index];
	}
	return 0;
}
