#include "youtubeaccountslistmodel.h"
#include "youtube.h"
#include "notifications.h"
#include <QSqlRecord>
#include <QSqlField>

YouTubeAccountsListModel::YouTubeAccountsListModel(QObject *parent) :
    QSqlTableModel(parent, QSqlDatabase::database())
{
    QHash<int, QByteArray> roles;
    roles[DisplayNameRole] = "displayName";
    roles[UsernameRole] = "username";
    roles[AccessTokenRole] = "accessToken";
    roles[RefreshTokenRole] = "refreshToken";
    roles[ActiveRole] = "active";
    this->setRoleNames(roles);
    this->setTable("youtubeAccounts");
    this->setEditStrategy(QSqlTableModel::OnManualSubmit);
    this->select();

    this->connect(YouTube::instance(), SIGNAL(signedIn(QString,QString,QString)), this, SLOT(addAccount(QString,QString,QString)));
    this->connect(this, SIGNAL(accountSelected(QString,QString,QString)), YouTube::instance(), SLOT(setAccount(QString,QString,QString)));
    this->connect(this, SIGNAL(accountAdded(QString,QString,QString)), YouTube::instance(), SLOT(setAccount(QString,QString,QString)));
    this->connect(this, SIGNAL(accountDeleted()), YouTube::instance(), SLOT(signOut()));
    this->connect(this, SIGNAL(info(QString)), Notifications::instance(), SLOT(onInfo(QString)));
    this->connect(this, SIGNAL(error(QString)), Notifications::instance(), SLOT(onError(QString)));
}

YouTubeAccountsListModel::~YouTubeAccountsListModel() {
    this->clear();
}

void YouTubeAccountsListModel::setActiveAccount(int row) {
    this->database().open();
    QSqlField nameField("name", QVariant::String);
    QSqlField usernameField("username", QVariant::String);
    QSqlField tokenField("token", QVariant::String);
    QSqlField refreshField("refresh", QVariant::String);
    QSqlField activeField("active", QVariant::Int);
    nameField.setValue(this->data(this->index(row, 0), DisplayNameRole));
    usernameField.setValue(this->data(this->index(row, 0), UsernameRole));
    tokenField.setValue(this->data(this->index(row, 0), AccessTokenRole));
    refreshField.setValue(this->data(this->index(row, 0), RefreshTokenRole));
    activeField.setValue(1);

    QSqlRecord rec;
    rec.append(nameField);
    rec.append(usernameField);
    rec.append(tokenField);
    rec.append(refreshField);
    rec.append(activeField);

    setRecord(row, rec);

    for (int i = 0; i < this->rowCount(); i++) {
        if (i != row) {
            nameField.setValue(this->data(this->index(i, 0), DisplayNameRole));
            usernameField.setValue(this->data(this->index(i, 0), UsernameRole));
            tokenField.setValue(this->data(this->index(i, 0), AccessTokenRole));
            refreshField.setValue(this->data(this->index(i, 0), RefreshTokenRole));
            activeField.setValue(0);

            rec.clear();
            rec.append(nameField);
            rec.append(usernameField);
            rec.append(tokenField);
            rec.append(refreshField);
            rec.append(activeField);

            this->setRecord(i, rec);
        }
    }

    this->submitAll();
}

void YouTubeAccountsListModel::switchAccount(int row) {
    this->setActiveAccount(row);

    this->database().open();
    QString user = this->data(this->index(row, 0), UsernameRole).toString();
    QString token = this->data(this->index(row, 0), AccessTokenRole).toString();
    QString refresh = this->data(this->index(row, 0), RefreshTokenRole).toString();

    emit accountSelected(user, token, refresh);
}

QVariant YouTubeAccountsListModel::data(const QModelIndex &idx, int role) const {
    return QSqlTableModel::data(this->index(idx.row(), role));
}

QVariant YouTubeAccountsListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row, 0), this->roleNames().key(role));
}

void YouTubeAccountsListModel::addAccount(const QString &displayName, const QString &token, const QString &refresh) {
    this->database().open();
    QSqlField nameField("name", QVariant::String);
    QSqlField usernameField("username", QVariant::String);
    QSqlField tokenField("token", QVariant::String);
    QSqlField refreshField("refresh", QVariant::String);
    QSqlField activeField("active", QVariant::Int);
    nameField.setValue(displayName);
    usernameField.setValue(QVariant::String);
    tokenField.setValue(token);
    refreshField.setValue(refresh);
    activeField.setValue(0);

    QSqlRecord rec;
    rec.append(nameField);
    rec.append(usernameField);
    rec.append(tokenField);
    rec.append(refreshField);
    rec.append(activeField);

    this->insertRecord(-1, rec);
    this->submitAll();
    this->setActiveAccount(this->rowCount() -1);

    emit accountAdded(QString(), token, refresh);
    emit countChanged(this->rowCount());
}

void YouTubeAccountsListModel::deleteAccount(int row) {
    this->database().open();
    bool signOut = this->data(this->index(row, 0), ActiveRole).toBool();

    if ((this->removeRow(row)) && (this->submitAll())) {
        if (signOut) {
            emit accountDeleted();
        }

        emit countChanged(this->rowCount());
        emit info(tr("Account deleted. Please visit the YouTube website to revoke access"));
    }
    else {
        emit error(tr("Database error. Unable to delete account"));
    }
}
