#include "youtubeuserlistmodel.h"
#include "feedurls.h"
#include "youtube.h"
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTimer>

YouTubeUserListModel::YouTubeUserListModel(QObject *parent) :
    AbstractUserListModel(parent),
    m_subscriptions(false),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
}

void YouTubeUserListModel::clear() {
    AbstractUserListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void YouTubeUserListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreUsers();
    }
}

void YouTubeUserListModel::setFeed(const QUrl &feed) {
    if (feed != this->feed()) {
        m_feed = feed;
        m_subscriptions = feed.toString().contains("subscriptions");
    }
}

void YouTubeUserListModel::getUsers(QUrl feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }
    else {
        feed = this->feed();
    }

    feed.addQueryItem("fields", m_subscriptions ? YOUTUBE_SUBSCRIPTION_FIELDS : YOUTUBE_USER_FIELDS);

    this->setLoading(true);
    QNetworkReply *reply = YouTube::instance()->createReply(feed, this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addUsers()));
}

void YouTubeUserListModel::search(const QString &query) {
    if (!query.isEmpty()) {
        this->setQuery(query);
    }

    this->setLoading(true);
    QNetworkReply *reply = YouTube::instance()->createSearchReply(Queries::Users, this->query(), this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addUsers()));
}

void YouTubeUserListModel::addUsers() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(YouTube::instance(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getUsers()));
        this->connect(YouTube::instance(), SIGNAL(refreshError()), this, SLOT(onError()));
        YouTube::instance()->refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNodeList entries = doc.elementsByTagName("entry");

        for (int i = 0; i < entries.size(); i++) {
            UserItem *user = new UserItem;
            user->loadYouTubeUser(entries.at(i), m_subscriptions);
            QMetaObject::invokeMethod(this, "appendUser", Qt::QueuedConnection, Q_ARG(QSharedPointer<UserItem>, QSharedPointer<UserItem>(user)));
        }

        QTimer::singleShot(1000, this, SLOT(stopLoading()));
        this->setMoreResults(!doc.namedItem("feed").namedItem("link").isNull());
        this->setOffset(this->offset() + 30);

        this->disconnect(YouTube::instance(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getUsers()));
        this->disconnect(YouTube::instance(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    emit countChanged(this->rowCount());

    reply->deleteLater();
}

void YouTubeUserListModel::getMoreUsers() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getUsers();
        }
    }
}
