#include "extractorplugin.h"
#include "utils.h"
#include "json.h"
#include <QRegExp>
#include <QStringList>
#include <QUrl>

using namespace QtJson;

ExtractorPlugin::ExtractorPlugin(QObject *parent) :
    QObject(parent)
{
}

ExtractorPlugin::~ExtractorPlugin() {}

ExtractorPlugin* ExtractorPlugin::create(QObject *parent) {
    return new ExtractorPlugin(parent);
}

QString ExtractorPlugin::versionNumber() {
    return CURRENT_PLUGIN_VERSION;
}

QUrl ExtractorPlugin::youtubeRequestUrl(const QString &id) {
    QUrl url("https://www.youtube.com/watch");
    url.addQueryItem("v", id);
    url.addQueryItem("gl", "US");
    url.addQueryItem("hl", "en");
    url.addQueryItem("has_verified", "1");

    return url;
}

QUrl ExtractorPlugin::dailymotionRequestUrl(const QString &id) {
    return QUrl("http://www.dailymotion.com/embed/video/" + id);
}

QUrl ExtractorPlugin::vimeoRequestUrl(const QString &id) {
    return QUrl("http://player.vimeo.com/video/" + id);
}

QString ExtractorPlugin::decryptYouTubeSignature(const QString &s) {
    QString signature;
    QStringList slist = Utils::reverseString(s).split("", QString::SkipEmptyParts);

    if (slist.size() > 65) {
        slist.removeFirst();
        slist.removeFirst();

        QString b = slist.at(0);
        slist.replace(0, slist.at(65 % slist.size()));
        slist.replace(65, b);
        signature = Utils::reverseString(slist.join(""));
    }

    return signature;
}

QVariantList ExtractorPlugin::extractYouTubeFormats(QString page) {

    QVariantList formats;

    if (!page.contains("url_encoded_fmt_stream_map\":")) {
        qDebug() << page;

        QString message = page.section("unavailable-message\" class=\"message\">", 1, 1).section('<', 0, 0).trimmed();
        QVariantMap errorMap;
        errorMap.insert("error", message.isEmpty() ? tr("Unable to retrieve formats. Access may be restricted") : message);
        formats << errorMap;
    }
    else {
        page = page.section("url_encoded_fmt_stream_map\":", 1, 1).section(", \"", 0, 0).trimmed().replace("\\u0026", "&");
        int unescapes = 0;

        while ((page.contains('%')) && (unescapes < 10)) {
            page = QByteArray::fromPercentEncoding(page.toUtf8());
            unescapes++;
        }

        bool encryptedSignatures = !page.contains("sig=");

        if (encryptedSignatures) {
            page = page.replace(QRegExp("[&,\"]s="), "&signature=");
        }
        else {
            page = page.replace(QRegExp("[&,\"]sig="), "&signature=");
        }

        QStringList urlStrings = page.split("url=", QString::SkipEmptyParts);
        QStringList sigs = page.split("signature=");
        QStringList signatures;

        if (encryptedSignatures) {
            for (int i = 1; i < sigs.size(); i++) {
                signatures << this->decryptYouTubeSignature(sigs.at(i).section(QRegExp("(&|,)"), 0, 0));
            }
        }
        else {
            for (int i = 1; i < sigs.size(); i++) {
                signatures << sigs.at(i).section(QRegExp("(&|,)"), 0, 0);
            }
        }

        foreach (QString urlString, urlStrings) {
            if ((urlString.startsWith("http")) && (!signatures.isEmpty())) {
                QUrl url(urlString.section(QRegExp("[&,\"]itag="), 0, 1));
                int key = url.queryItemValue("itag").toInt();
                QVariantMap format;
                format["value"] = key;

                switch (key) {
                case 37:
                    url.removeQueryItem("signature");
                    url.addQueryItem("signature", signatures.takeFirst());
                    format["name"] = "1080P AVC1";
                    format["service"] = Services::YouTube;
                    format["url"] = url;
                    formats << format;
                    break;
                case 22:
                    url.removeQueryItem("signature");
                    url.addQueryItem("signature", signatures.takeFirst());
                    format["name"] = "720P AVC1";
                    format["service"] = Services::YouTube;
                    format["url"] = url;
                    formats << format;
                    break;
#ifndef Q_OS_SYMBIAN
                case 35:
                    url.removeQueryItem("signature");
                    url.addQueryItem("signature", signatures.takeFirst());
                    format["name"] = "480P FLV";
                    format["service"] = Services::YouTube;
                    format["url"] = url;
                    formats << format;
                    break;
                case 34:
                    url.removeQueryItem("signature");
                    url.addQueryItem("signature", signatures.takeFirst());
                    format["name"] = "360P AVC1";
                    format["service"] = Services::YouTube;
                    format["url"] = url;
                    formats << format;
                    break;
#endif
                case 18:
                    url.removeQueryItem("signature");
                    url.addQueryItem("signature", signatures.takeFirst());
                    format["name"] = QString("%1 AVC1").arg(tr("Normal"));
                    format["service"] = Services::YouTube;
                    format["url"] = url;
                    formats << format;
                default:
                    signatures.takeFirst();
                    break;
                }
            }
        }
    }

    return formats;
}

QVariantList ExtractorPlugin::extractDailymotionFormats(QString page) {
    QVariantMap info = Json::parse(page.section("var info = ", 1, 1).section(";\n", 0, 0)).toMap();
    QVariantList formats;
    int i = 0;

    QMap<QString, QString> formatNames;
    formatNames["stream_h264_url"] = QString("%1 AVC1").arg(tr("Normal"));
    formatNames["stream_h264_hq_url"] = "480P AVC1";
    formatNames["stream_h264_hd_url"] = "720P AVC1";
    formatNames["stream_h264_hd1080_url"] = "1080P AVC1";

    foreach (QString formatName, formatNames.keys()) {
        QUrl url = info.value(formatName).toUrl();

        if (!url.isEmpty()) {
            QVariantMap format;
            format["name"] = formatNames[formatName];
            format["value"] = formatName;
            format["service"] = Services::Dailymotion;
            format["url"] = url;
            formats << format;
        }

        i++;
    }

    if (formats.isEmpty()) {
        qDebug() << page;
        QString errorString = info.value("error").toMap().value("message").toString();
        QVariantMap errorMap;
        errorMap["error"] = errorString.isEmpty() ? tr("Unable to retrieve video formats. Access may be restricted") : errorString;
        formats << errorMap;
    }
    
    return formats;
}

QVariantList ExtractorPlugin::extractVimeoFormats(QString page) {
    QVariantList formats;
    QString params = page.section("config:", 1, 1).section("};", 0, 0);
    QVariantMap paramMap = Json::parse(params).toMap();

    if (paramMap.isEmpty()) {
        qDebug() << page;
        QVariantMap errorMap;
        errorMap["error"] = tr("Unable to retrieve video formats. Access may be restricted");
        formats << errorMap;
    }
    else {
        QVariantMap requestMap = paramMap.value("request").toMap();
        QVariantMap videoMap = paramMap.value("video").toMap();
        QVariantMap formatMap = videoMap.value("files").toMap();
        QString codec("h264");

        if (!formatMap.isEmpty()) {
            codec = formatMap.keys().first();

            QStringList qualities = formatMap.value(codec).toStringList();
            QMap<QString, QString> formatNames;
            formatNames["mobile"] = tr("Mobile");
            formatNames["sd"] = "360P AVC1";
            formatNames["hd"] = "720P AVC1";

            while (!qualities.isEmpty()) {
                QString quality = qualities.takeFirst();
                QString timeStamp = requestMap.value("timestamp").toString();
                QString signature = requestMap.value("signature").toString();
                QString id = videoMap.value("id").toString();

                QVariantMap format;
                format["name"] = formatNames[quality];
                format["value"] = quality;
                format["service"] = Services::Vimeo;
                QUrl url("http://player.vimeo.com/play_redirect");
                url.addQueryItem("quality", quality);
                url.addQueryItem("clip_id", id);
                url.addQueryItem("time", timeStamp);
                url.addQueryItem("sig", signature);

                if (quality != "mobile") {
                    url.addQueryItem("codecs", codec);
                }

                format["url"] = url;
                formats << format;
            }
        }
        else {
            qDebug() << page;
            QVariantMap errorMap;
            errorMap["error"] = tr("Unable to retrieve video formats. Access may be restricted");
            formats << errorMap;
        }
    }

    return formats;
}

Q_EXPORT_PLUGIN2(cutetubeextractor, ExtractorPlugin)
