#include "socialvideolistmodel.h"
#include "sharing.h"
#include "playbacklistmodel.h"
#include "transfermanager.h"
#ifndef QML_USER_INTERFACE
#include "thumbnailcache.h"
#endif

SocialVideoListModel::SocialVideoListModel(QObject *parent) :
    QAbstractListModel(parent)
  #ifndef QML_USER_INTERFACE
    ,m_cache(new ThumbnailCache),
    m_thumbnailPressedRow(-1)
  #endif
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[VideoIdRole] = "videoId";
    roles[FavouriteIdRole] = "favouriteId";
    roles[PlaylistVideoIdRole] = "playlistVideoId";
    roles[TitleRole] = "title";
    roles[ArtistRole] = "artist";
    roles[DateRole] = "date";
    roles[DurationRole] = "duration";
    roles[ViewCountRole] = "viewCount";
    roles[RatingCountRole] = "ratingCount";
    roles[RatingRole] = "rating";
#ifndef QML_USER_INTERFACE
    roles[ThumbnailRole] = "thumbnail";
    roles[ThumbnailPressedRole] = "thumbnailPressed";
#endif
    roles[ThumbnailUrlRole] = "thumbnailUrl";
    roles[ArchiveRole] = "archive";
    roles[ServiceRole] = "service";
    roles[FavouriteRole] = "favourite";
    roles[SelectedRole] = "selected";
    this->setRoleNames(roles);
#ifndef QML_USER_INTERFACE
    this->connect(m_cache, SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
#endif
    Sharing::instance()->getVideos();
    this->connect(Sharing::instance(), SIGNAL(videoAddedToCache(int)), this, SLOT(onVideoAdded(int)));
}

SocialVideoListModel::~SocialVideoListModel() {
#ifndef QML_USER_INTERFACE
    delete m_cache;
    m_cache = 0;
#endif
}

void SocialVideoListModel::onVideoAdded(int row) {
    this->beginInsertRows(QModelIndex(), row, row);
    this->endInsertRows();
    emit countChanged(this->rowCount());
}

int SocialVideoListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return Sharing::instance()->videos()->size();
}

QVariant SocialVideoListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return Sharing::instance()->videos()->at(index.row()).data()->id();
    case VideoIdRole:
        return Sharing::instance()->videos()->at(index.row()).data()->videoId();
    case TitleRole:
        return Sharing::instance()->videos()->at(index.row()).data()->title();
    case ArtistRole:
        return Sharing::instance()->videos()->at(index.row()).data()->artist();
    case DateRole:
        return Sharing::instance()->videos()->at(index.row()).data()->date();
    case DurationRole:
        return Sharing::instance()->videos()->at(index.row()).data()->duration();
    case ViewCountRole:
        return Sharing::instance()->videos()->at(index.row()).data()->viewCount();
    case RatingCountRole:
        return Sharing::instance()->videos()->at(index.row()).data()->ratingCount();
    case RatingRole:
        return Sharing::instance()->videos()->at(index.row()).data()->rating();
#ifndef QML_USER_INTERFACE
    case ThumbnailRole:
        return m_cache->thumbnail(Sharing::instance()->videos()->at(index.row()).data()->thumbnailUrl(), QSize(160, 120));
    case ThumbnailPressedRole:
        return QVariant(m_thumbnailPressedRow == index.row());
#endif
    case ThumbnailUrlRole:
        return Sharing::instance()->videos()->at(index.row()).data()->thumbnailUrl();
    case ServiceRole:
        return Sharing::instance()->videos()->at(index.row()).data()->service();
    case SelectedRole:
        return QVariant(m_selectedRows.contains(index.row()));
    default:
        return QVariant();
    }
}

QSharedPointer<VideoItem> SocialVideoListModel::get(int row) const {
    return QSharedPointer<VideoItem>(Sharing::instance()->videos()->at(row));
}

#ifdef QML_USER_INTERFACE
VideoItem* SocialVideoListModel::getFromQML(int row) const {
    return Sharing::instance()->videos()->at(row).data();
}
#endif

void SocialVideoListModel::toggleSelected(int row) {
    if (!m_selectedRows.contains(row)) {
        m_selectedRows.append(row);
    }
    else {
        m_selectedRows.removeOne(row);
    }

    emit dataChanged(this->index(row), this->index(row));
}

void SocialVideoListModel::selectAll() {
    for (int i = 0; i < this->rowCount(); i++) {
        m_selectedRows.append(i);
    }

    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}

void SocialVideoListModel::selectNone() {
    m_selectedRows.clear();

    if (this->rowCount() > 0) {
        emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
    }
}

QList< QSharedPointer<VideoItem> > SocialVideoListModel::selectedItems() const {
    QList< QSharedPointer<VideoItem> > checked;

    for (int i = 0; i < m_selectedRows.size(); i++) {
        checked.append(this->get(m_selectedRows.at(i)));
    }

    return checked;
}

#ifdef QML_USER_INTERFACE
QList<VideoItem*> SocialVideoListModel::selectedItemsFromQML() const {
    QList<VideoItem*> items;
    for (int i = 0; i < m_selectedRows.size(); i++) {
        items.append(this->getFromQML(m_selectedRows.at(i)));
    }

    return items;
}
#endif

void SocialVideoListModel::downloadSelectedVideos(bool saveAsAudio) {
    TransferManager::instance()->addDownloadTransfers(this->selectedItems(), saveAsAudio);
    this->selectNone();
}

void SocialVideoListModel::queueSelectedVideos() {
    PlaybackListModel::playbackQueue()->addVideos(this->selectedItems());
    this->selectNone();
}

#ifndef QML_USER_INTERFACE
void SocialVideoListModel::onThumbnailReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
#endif
