#include "session.h"
#include "cookiejar.h"
#include "settings.h"
#include "database.h"
#include "videolauncher.h"
#include "playbacklistmodel.h"
#include "youtube.h"
#include "dailymotion.h"
#include "vimeo.h"
#include "sharing.h"
#include "notifications.h"
#include "transfermanager.h"
#include "clipboardmonitor.h"
#include "urlgrabber.h"
#ifdef GRABBER_PLUGIN
#include "../plugins/pluginmanager.h"
#endif
#ifndef QML_USER_INTERFACE
#include "thumbnailcache.h"
#ifdef Q_WS_MAEMO_5
#include "windowstack.h"
#endif
#endif
#include <QNetworkAccessManager>

Session* sessionInstance = 0;

Session::Session(QObject *parent) :
    QObject(parent),
    m_nam(new QNetworkAccessManager(this)),
    m_jar(new CookieJar(m_nam)),
    m_settings(new Settings(this)),
    m_database(new Database(this)),
    m_launcher(new VideoLauncher(this)),
    m_queue(new PlaybackListModel(this)),
    m_youtube(new YouTube(this)),
    m_dailymotion(new Dailymotion(this)),
    m_vimeo(new Vimeo(this)),
    m_sharing(new Sharing(this)),
    m_grabber(new UrlGrabber(this)),
  #ifdef GRABBER_PLUGIN
    m_pluginManager(new PluginManager(this)),
  #endif
    m_notifications(new Notifications(this)),
    m_transferManager(new TransferManager(this)),
    m_monitor(new ClipboardMonitor(this))
  #ifndef QML_USER_INTERFACE
  ,m_cache(new ThumbnailCache)
  #ifdef Q_WS_MAEMO_5
  ,m_stack(new WindowStack(this))
  #endif
  #endif
{
    if (!sessionInstance) {
        sessionInstance = this;
    }

    this->networkAccessManager()->setCookieJar(m_jar);
    this->onMediaPlayerChanged(m_settings->mediaPlayer());

    this->connect(m_database, SIGNAL(gotDailymotionAccount(QString,QString,QString)), m_dailymotion, SLOT(setAccount(QString,QString,QString)));
    this->connect(m_database, SIGNAL(gotYouTubeAccount(QString,QString,QString)), m_youtube, SLOT(setAccount(QString,QString,QString)));
    this->connect(m_database, SIGNAL(gotVimeoAccount(QString,QString,QString)), m_vimeo, SLOT(setAccount(QString,QString,QString)));
    this->connect(m_database, SIGNAL(gotFacebookAccount(QString)), m_sharing, SLOT(setFacebookAccessToken(QString)));
    this->connect(m_database, SIGNAL(gotTwitterAccount(QString,QString)), m_sharing, SLOT(setTwitterAccount(QString,QString)));
    this->connect(m_sharing, SIGNAL(signedInToFacebook(QString)), m_database, SLOT(storeFacebookAccount(QString)));
    this->connect(m_sharing, SIGNAL(signedInToTwitter(QString,QString)), m_database, SLOT(storeTwitterAccount(QString,QString)));
    this->connect(m_sharing, SIGNAL(signedOutOfFacebook()), m_database, SLOT(deleteFacebookAccount()));
    this->connect(m_sharing, SIGNAL(signedOutOfTwitter()), m_database, SLOT(deleteTwitterAccount()));
    this->connect(m_youtube, SIGNAL(usernameChanged(QString)), m_database, SLOT(setYouTubeUsername(QString)));
    this->connect(m_youtube, SIGNAL(accessTokenRefreshed(QString)), m_database, SLOT(setYouTubeAccessToken(QString)));
    this->connect(m_dailymotion, SIGNAL(usernameChanged(QString)), m_database, SLOT(setDailymotionUsername(QString)));
    this->connect(m_dailymotion, SIGNAL(accessTokenRefreshed(QString,QString)), m_database, SLOT(setDailymotionAccessToken(QString,QString)));
    this->connect(m_vimeo, SIGNAL(usernameChanged(QString)), m_database, SLOT(setVimeoUsername(QString)));
    this->connect(m_settings, SIGNAL(youtubePlaybackFormatsChanged(QSet<int>)), m_grabber, SLOT(setYouTubeFormats(QSet<int>)));
    this->connect(m_settings, SIGNAL(dailymotionPlaybackFormatsChanged(QSet<int>)), m_grabber, SLOT(setDailymotionFormats(QSet<int>)));
    this->connect(m_settings, SIGNAL(vimeoPlaybackFormatsChanged(QSet<int>)), m_grabber, SLOT(setVimeoFormats(QSet<int>)));
    this->connect(m_settings, SIGNAL(safeSearchChanged(bool)), m_youtube, SLOT(setSafeSearch(bool)));
    this->connect(m_settings, SIGNAL(safeSearchChanged(bool)), m_dailymotion, SLOT(setSafeSearch(bool)));
    this->connect(m_settings, SIGNAL(mediaPlayerChanged(QString)), this, SLOT(onMediaPlayerChanged(QString)));
    this->connect(m_settings, SIGNAL(maximumConcurrentTransfersChanged(int,int)), m_transferManager, SLOT(onMaximumConcurrentTransfersChanged(int,int)));
    this->connect(m_sharing, SIGNAL(gotYouTubeIds(QStringList)), m_youtube, SLOT(getVideosFromIds(QStringList)));
    this->connect(m_sharing, SIGNAL(gotDailymotionIds(QStringList)), m_dailymotion, SLOT(getVideosFromIds(QStringList)));
    this->connect(m_sharing, SIGNAL(gotVimeoIds(QStringList)), m_vimeo, SLOT(getVideosFromIds(QStringList)));
    this->connect(m_youtube, SIGNAL(gotVideosFromIds(QList< QSharedPointer<VideoItem> >)), m_sharing, SLOT(addVideosToCache(QList< QSharedPointer<VideoItem> >)));
    this->connect(m_dailymotion, SIGNAL(gotVideosFromIds(QList< QSharedPointer<VideoItem> >)), m_sharing, SLOT(addVideosToCache(QList< QSharedPointer<VideoItem> >)));
    this->connect(m_vimeo, SIGNAL(gotVideosFromIds(QList< QSharedPointer<VideoItem> >)), m_sharing, SLOT(addVideosToCache(QList< QSharedPointer<VideoItem> >)));

    this->connect(m_youtube, SIGNAL(signedIn(QString,QString,QString)), m_jar, SLOT(clearCookies()));
    this->connect(m_dailymotion, SIGNAL(signedIn(QString,QString,QString)), m_jar, SLOT(clearCookies()));
    this->connect(m_vimeo, SIGNAL(signedIn(QString,QString,QString)), m_jar, SLOT(clearCookies()));
    this->connect(m_sharing, SIGNAL(signedInToFacebook(QString)), m_jar, SLOT(clearCookies()));
    this->connect(m_sharing, SIGNAL(signedInToTwitter(QString,QString)), m_jar, SLOT(clearCookies()));

    this->connect(m_database, SIGNAL(alert(QString)), m_notifications, SLOT(onAlert(QString)));
    this->connect(m_database, SIGNAL(error(QString)), m_notifications, SLOT(onError(QString)));
    this->connect(m_database, SIGNAL(info(QString)), m_notifications, SLOT(onInfo(QString)));
    this->connect(m_sharing, SIGNAL(alert(QString)), m_notifications, SLOT(onAlert(QString)));
    this->connect(m_sharing, SIGNAL(error(QString)), m_notifications, SLOT(onError(QString)));
    this->connect(m_sharing, SIGNAL(info(QString)), m_notifications, SLOT(onInfo(QString)));
    this->connect(m_youtube, SIGNAL(alert(QString)), m_notifications, SLOT(onAlert(QString)));
    this->connect(m_youtube, SIGNAL(error(QString)), m_notifications, SLOT(onError(QString)));
    this->connect(m_youtube, SIGNAL(info(QString)), m_notifications, SLOT(onInfo(QString)));
    this->connect(m_dailymotion, SIGNAL(alert(QString)), m_notifications, SLOT(onAlert(QString)));
    this->connect(m_dailymotion, SIGNAL(error(QString)), m_notifications, SLOT(onError(QString)));
    this->connect(m_dailymotion, SIGNAL(info(QString)), m_notifications, SLOT(onInfo(QString)));
    this->connect(m_vimeo, SIGNAL(alert(QString)), m_notifications, SLOT(onAlert(QString)));
    this->connect(m_vimeo, SIGNAL(error(QString)), m_notifications, SLOT(onError(QString)));
    this->connect(m_vimeo, SIGNAL(info(QString)), m_notifications, SLOT(onInfo(QString)));
    this->connect(m_grabber, SIGNAL(error(QString)), m_notifications, SLOT(onError(QString)));
    this->connect(m_queue, SIGNAL(alert(QString)), m_notifications, SLOT(onAlert(QString)));
    this->connect(m_transferManager, SIGNAL(alert(QString)), m_notifications, SLOT(onAlert(QString)));
    this->connect(m_youtube, SIGNAL(busy(QString,int)), m_notifications, SLOT(showProgressDialog(QString,int)));
    this->connect(m_youtube, SIGNAL(busyProgressChanged(int)), m_notifications, SLOT(updateProgressDialog(int)));
    this->connect(m_dailymotion, SIGNAL(busy(QString,int)), m_notifications, SLOT(showProgressDialog(QString,int)));
    this->connect(m_dailymotion, SIGNAL(busyProgressChanged(int)), m_notifications, SLOT(updateProgressDialog(int)));
    this->connect(m_vimeo, SIGNAL(busy(QString,int)), m_notifications, SLOT(showProgressDialog(QString,int)));
    this->connect(m_vimeo, SIGNAL(busyProgressChanged(int)), m_notifications, SLOT(updateProgressDialog(int)));
    this->connect(m_sharing, SIGNAL(busy(QString,int)), m_notifications, SLOT(showProgressDialog(QString,int)));
    this->connect(m_sharing, SIGNAL(busyProgressChanged(int)), m_notifications, SLOT(updateProgressDialog(int)));
#ifndef QML_USER_INTERFACE
    this->connect(m_youtube, SIGNAL(alert(QString)), m_notifications, SLOT(cancelProgressDialog()));
    this->connect(m_youtube, SIGNAL(error(QString)), m_notifications, SLOT(cancelProgressDialog()));
    this->connect(m_youtube, SIGNAL(info(QString)), m_notifications, SLOT(cancelProgressDialog()));
    this->connect(m_youtube, SIGNAL(currentOperationCancelled()), m_notifications, SLOT(cancelProgressDialog()));
    this->connect(m_dailymotion, SIGNAL(alert(QString)), m_notifications, SLOT(cancelProgressDialog()));
    this->connect(m_dailymotion, SIGNAL(error(QString)), m_notifications, SLOT(cancelProgressDialog()));
    this->connect(m_dailymotion, SIGNAL(info(QString)), m_notifications, SLOT(cancelProgressDialog()));
    this->connect(m_dailymotion, SIGNAL(currentOperationCancelled()), m_notifications, SLOT(cancelProgressDialog()));
    this->connect(m_vimeo, SIGNAL(alert(QString)), m_notifications, SLOT(cancelProgressDialog()));
    this->connect(m_vimeo, SIGNAL(error(QString)), m_notifications, SLOT(cancelProgressDialog()));
    this->connect(m_vimeo, SIGNAL(info(QString)), m_notifications, SLOT(cancelProgressDialog()));
    this->connect(m_vimeo, SIGNAL(currentOperationCancelled()), m_notifications, SLOT(cancelProgressDialog()));
    this->connect(m_sharing, SIGNAL(alert(QString)), m_notifications, SLOT(cancelProgressDialog()));
    this->connect(m_sharing, SIGNAL(error(QString)), m_notifications, SLOT(cancelProgressDialog()));
    this->connect(m_sharing, SIGNAL(info(QString)), m_notifications, SLOT(cancelProgressDialog()));
    this->connect(m_sharing, SIGNAL(currentOperationCancelled()), m_notifications, SLOT(cancelProgressDialog()));
#endif

    m_grabber->setNetworkAccessManager(this->networkAccessManager());
    m_youtube->setNetworkAccessManager(this->networkAccessManager());
    m_dailymotion->setNetworkAccessManager(this->networkAccessManager());
    m_vimeo->setNetworkAccessManager(this->networkAccessManager());
    m_sharing->setNetworkAccessManager(this->networkAccessManager());
    m_transferManager->setNetworkAccessManager(this->networkAccessManager());
#ifdef GRABBER_PLUGIN
    m_pluginManager->setNetworkAccessManager(this->networkAccessManager());
    m_pluginManager->loadPlugin();
#endif
    m_settings->restoreSettings();
    m_database->restoreAccounts();
#ifndef QML_USER_INTERFACE
    m_transferManager->restoreStoredDownloads();
#endif
}

Session::~Session() {
#ifndef QML_USER_INTERFACE
    delete m_cache;
    m_cache = 0;
#endif
}

Session* Session::instance() {
    return sessionInstance;
}

void Session::onMediaPlayerChanged(const QString &player) {
    this->disconnect(m_grabber, SIGNAL(gotVideoUrl(QUrl,Videos::VideoFormat)), m_launcher, SLOT(launchVideo(QUrl)));
    this->disconnect(m_grabber, SIGNAL(busy(QString,int)), m_notifications, SLOT(showProgressDialog(QString,int)));
    this->disconnect(m_grabber, SIGNAL(busyProgressChanged(int)), m_notifications, SLOT(updateProgressDialog(int)));
#ifndef QML_USER_INTERFACE
    this->connect(m_grabber, SIGNAL(error(QString)), m_notifications, SLOT(cancelProgressDialog()));
#endif

    if (!player.startsWith("cutetube")) {
        this->connect(m_grabber, SIGNAL(gotVideoUrl(QUrl,Videos::VideoFormat)), m_launcher, SLOT(launchVideo(QUrl)));
        this->connect(m_grabber, SIGNAL(busy(QString,int)), m_notifications, SLOT(showProgressDialog(QString,int)));
        this->connect(m_grabber, SIGNAL(busyProgressChanged(int)), m_notifications, SLOT(updateProgressDialog(int)));
#ifndef QML_USER_INTERFACE
        this->connect(m_grabber, SIGNAL(error(QString)), m_notifications, SLOT(cancelProgressDialog()));
#endif
    }
}
