#include "transferdetailsdialog.h"
#include "transferpriorityselector.h"
#include "../base/utils.h"
#include <QLabel>
#include <QGridLayout>
#include <QScrollArea>
#include <QLineEdit>
#include <QCheckBox>

TransferDetailsDialog::TransferDetailsDialog(QSharedPointer<TransferItem> transfer, QWidget *parent) :
    AbstractDialog(parent),
    m_transfer(transfer),
    m_titleHeader(new QLabel(tr("Details") + ":", this)),
    m_statusHeader(new QLabel(tr("Status") + ":", this)),
    m_sizeHeader(new QLabel(tr("Size") + ":", this)),
    m_progressHeader(new QLabel(tr("Progress") + ":", this)),
    m_statusLabel(new QLabel(this)),
    m_statusIcon(new QLabel(this)),
    m_infoLabel(new QLabel(this)),
    m_sizeLabel(new QLabel(this)),
    m_progressLabel(new QLabel(this)),
    m_speedLabel(new QLabel(this)),
    m_prioritySelector(new TransferPrioritySelector(tr("Priority"), this)),
    m_audioCheckbox(new QCheckBox(tr("Save as audio"), this)),
    m_nameEdit(new QLineEdit(m_transfer.data()->fileName(), this)),
    m_nameButton(new QPushButton(tr("Change"), this)),
    m_pauseResumeButton(new QPushButton(this)),
    m_cancelButton(new QPushButton(tr("Cancel"), this)),
    m_grid(0)
{
    this->setWindowTitle(tr("Download status"));
    this->setAttribute(Qt::WA_DeleteOnClose, true);

    m_nameEdit->setEnabled(m_transfer.data()->transferType() == Transfers::Download);
    m_prioritySelector->setValue(m_transfer.data()->priority());
    m_audioCheckbox->setEnabled(m_transfer.data()->convertibleToAudio());
    m_audioCheckbox->setChecked(m_transfer.data()->saveAsAudio());

    m_infoLabel->setWordWrap(true);
    m_titleHeader->setFixedWidth(100);
    m_statusHeader->setFixedWidth(100);
    m_sizeHeader->setFixedWidth(100);
    m_progressHeader->setFixedWidth(100);
    m_sizeLabel->setFixedWidth(100);
    m_sizeLabel->setAlignment(Qt::AlignCenter);
    m_progressLabel->setFixedWidth(100);
    m_progressLabel->setAlignment(Qt::AlignCenter);
    m_statusIcon->setFixedSize(48, 48);
    m_statusIcon->setScaledContents(true);
    m_nameEdit->setCursorPosition(0);
    m_nameButton->setEnabled(false);
    m_nameButton->setFixedWidth(150);
    m_pauseResumeButton->setFixedWidth(150);
    m_cancelButton->setFixedWidth(150);

    QScrollArea *scrollArea = new QScrollArea(this);
    QWidget *scrollWidget = new QWidget(scrollArea);

    m_grid = new QGridLayout(scrollWidget);
    m_grid->setContentsMargins(0, 0, 0, 0);

    scrollArea->setWidgetResizable(true);
    scrollArea->setWidget(scrollWidget);
    scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    QGridLayout *grid2 = new QGridLayout(this);
    grid2->addWidget(scrollArea);

    this->onTransferStatusChanged(m_transfer.data()->status());
    this->onTransferStatusInfoChanged(m_transfer.data()->statusInfo());
    this->onTransferSizeChanged(m_transfer.data()->size());
    this->onTransferProgressChanged(m_transfer.data()->progress());
    this->onTransferSpeedChanged(m_transfer.data()->speed());

    this->connect(m_transfer.data(), SIGNAL(statusChanged(Transfers::TransferStatus)), this, SLOT(onTransferStatusChanged(Transfers::TransferStatus)));
    this->connect(m_transfer.data(), SIGNAL(statusInfoChanged(QString)), this, SLOT(onTransferStatusInfoChanged(QString)));
    this->connect(m_transfer.data(), SIGNAL(sizeChanged(qint64)), this, SLOT(onTransferSizeChanged(qint64)));
    this->connect(m_transfer.data(), SIGNAL(progressChanged(int)), this, SLOT(onTransferProgressChanged(int)));
    this->connect(m_transfer.data(), SIGNAL(speedChanged(int)), this, SLOT(onTransferSpeedChanged(int)));
    this->connect(m_prioritySelector, SIGNAL(valueChanged(Transfers::TransferPriority)), m_transfer.data(), SLOT(setPriority(Transfers::TransferPriority)));
    this->connect(m_nameEdit, SIGNAL(textEdited(QString)), this, SLOT(onFileNameTextChanged(QString)));
    this->connect(m_nameButton, SIGNAL(clicked()), this, SLOT(setTransferFileName()));
    this->connect(m_audioCheckbox, SIGNAL(clicked(bool)), m_transfer.data(), SLOT(setSaveAsAudio(bool)));
    this->connect(m_pauseResumeButton, SIGNAL(clicked()), this, SLOT(onPauseResumeClicked()));
    this->connect(m_cancelButton, SIGNAL(clicked()), this, SLOT(cancelTransfer()));

    this->onOrientationChanged();
}

void TransferDetailsDialog::setLandscapeLayout() {
    this->setMinimumHeight(340);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_titleHeader, 0, 0);
        m_grid->addWidget(m_nameEdit, 0, 1, 1, 2);
        m_grid->addWidget(m_nameButton, 0, 3);
        m_grid->addWidget(m_statusHeader, 1, 0);
        m_grid->addWidget(m_statusIcon, 1, 1, 1, 1, Qt::AlignHCenter);
        m_grid->addWidget(m_statusLabel, 1, 2);
        m_grid->addWidget(m_infoLabel, 2, 2);
        m_grid->addWidget(m_sizeHeader, 3, 0);
        m_grid->addWidget(m_sizeLabel, 3, 1);
        m_grid->addWidget(m_progressHeader, 4, 0);
        m_grid->addWidget(m_progressLabel, 4, 1);
        m_grid->addWidget(m_speedLabel, 4, 2);
        m_grid->addWidget(m_audioCheckbox, 5, 0, 1, 3);
        m_grid->addWidget(m_pauseResumeButton, 5, 3);
        m_grid->addWidget(m_prioritySelector, 6, 0, 1, 3);
        m_grid->addWidget(m_cancelButton, 6, 3);
    }
}

void TransferDetailsDialog::setPortraitLayout() {
    this->setMinimumHeight(420);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_titleHeader, 0, 0);
        m_grid->addWidget(m_nameEdit, 0, 1, 1, 2);
        m_grid->addWidget(m_nameButton, 0, 3);
        m_grid->addWidget(m_statusHeader, 1, 0);
        m_grid->addWidget(m_statusIcon, 1, 1, 1, 1, Qt::AlignHCenter);
        m_grid->addWidget(m_statusLabel, 1, 2);
        m_grid->addWidget(m_infoLabel, 2, 2);
        m_grid->addWidget(m_sizeHeader, 3, 0);
        m_grid->addWidget(m_sizeLabel, 3, 1);
        m_grid->addWidget(m_progressHeader, 4, 0);
        m_grid->addWidget(m_progressLabel, 4, 1);
        m_grid->addWidget(m_speedLabel, 4, 2);
        m_grid->addWidget(m_audioCheckbox, 5, 0, 1, 3);
        m_grid->addWidget(m_pauseResumeButton, 5, 3);
        m_grid->addWidget(m_prioritySelector, 6, 0, 1, 3);
        m_grid->addWidget(m_cancelButton, 6, 3);
    }    
}

void TransferDetailsDialog::onTransferStatusChanged(Transfers::TransferStatus status) {
    m_statusLabel->setText(m_transfer.data()->statusText());

    switch (status) {
    case Transfers::Downloading:
        m_statusIcon->setPixmap(QPixmap("/usr/share/icons/hicolor/48x48/hildon/as_proxy_attachment_download.png"));
        m_pauseResumeButton->setText(tr("Pause"));
        m_infoLabel->hide();
        break;
    case Transfers::Connecting:
        m_statusIcon->setPixmap(QPixmap("/usr/share/icons/hicolor/48x48/hildon/general_packetdata.png"));
        m_pauseResumeButton->setText(tr("Pause"));
        m_infoLabel->hide();
        break;
    case Transfers::Paused:
        m_statusIcon->setPixmap(QPixmap("/etc/hildon/theme/mediaplayer/Pause.png"));
        m_pauseResumeButton->setText(tr("Resume"));
        m_infoLabel->hide();
        break;
    case Transfers::Converting:
        m_statusIcon->setPixmap(QPixmap("/usr/share/icons/hicolor/48x48/hildon/general_audio_file.png"));
        m_pauseResumeButton->setText(tr("Pause"));
        m_infoLabel->show();
        break;
    case Transfers::Failed:
        m_statusIcon->setPixmap(QPixmap("/usr/share/icons/hicolor/48x48/hildon/general_stop.png"));
        m_pauseResumeButton->setText(tr("Resume"));
        m_infoLabel->show();
        break;
    case Transfers::Completed:
        this->accept();
        break;
    case Transfers::Cancelled:
        this->accept();
        break;
    default:
        m_statusIcon->setPixmap(QPixmap("/etc/hildon/theme/mediaplayer/Stop.png"));
        m_pauseResumeButton->setText(tr("Pause"));
        m_infoLabel->hide();
    }
}

void TransferDetailsDialog::onTransferStatusInfoChanged(const QString &info) {
    m_infoLabel->setText(QString("<font color='red'>%1</font>").arg(info));
}

void TransferDetailsDialog::onTransferSizeChanged(qint64 size) {
    m_sizeLabel->setText(size > 0 ? Utils::fileSizeFromBytes(size) : "-");
}

void TransferDetailsDialog::onTransferProgressChanged(int progress) {
    m_progressLabel->setText(QString("%1%").arg(progress));
}

void TransferDetailsDialog::onTransferSpeedChanged(int speed) {
    m_speedLabel->setText(QString("<font color='#4d4d4d'>%1/s</font>").arg(Utils::fileSizeFromBytes(speed)));
}

void TransferDetailsDialog::onFileNameTextChanged(const QString &text) {
    m_nameButton->setEnabled(!text.isEmpty());
}

void TransferDetailsDialog::setTransferFileName() {
    m_transfer.data()->setFileName(m_nameEdit->text());
    m_nameButton->setEnabled(false);
}

void TransferDetailsDialog::onPauseResumeClicked() {
    if (m_transfer.data()->status() == Transfers::Paused) {
        m_transfer.data()->setStatus(Transfers::Queued);
    }
    else {
        m_transfer.data()->setStatus(Transfers::Paused);
    }
}

void TransferDetailsDialog::cancelTransfer() {
    m_transfer.data()->setStatus(Transfers::Cancelled);
}
