#include "dailymotionplaylistvideoswindow.h"
#include "videolistdelegate.h"
#include "dailymotionvideoinfowindow.h"
#include "dailymotionchannelwindow.h"
#include "dailymotionvideoselectionwindow.h"
#include "dailymotionplaylistdialog.h"
#include "videoplaybackwindow.h"
#include "../base/dailymotionvideolistmodel.h"
#include "../base/thumbnailcache.h"
#include "../base/playlistitem.h"
#include "../base/dailymotion.h"
#include "../base/playbacklistmodel.h"
#include "../base/transfermanager.h"
#include "../base/urlgrabber.h"
#include "../base/settings.h"
#include "listview.h"
#include "thumbnaillabel.h"
#include "separatorlabel.h"
#include <QAction>
#include <QLabel>
#include <QScrollArea>
#include <QPushButton>
#include <QMenuBar>
#include <QGridLayout>
#include <QScrollBar>
#include <QTimer>

DailymotionPlaylistVideosWindow::DailymotionPlaylistVideosWindow(QSharedPointer<PlaylistItem> playlist, QWidget *parent) :
    StackedWindow(parent),
    m_playlist(playlist),
    m_listView(new ListView(this)),
    m_model(new DailymotionVideoListModel(this)),
    m_deleteFromPlaylistAction(this->menuBar()->addAction(tr("Delete from playlist"), this, SLOT(deleteVideosFromPlaylist()))),
    m_addToFavouritesAction(this->menuBar()->addAction(tr("Add to favourites"), this, SLOT(addVideosToFavourites()))),
    m_addToPlaylistAction(this->menuBar()->addAction(tr("Add to playlist"), this, SLOT(addVideosToPlaylist()))),
    m_addToQueueAction(this->menuBar()->addAction(tr("Add to playback queue"), this, SLOT(queueVideos()))),
    m_videoDownloadAction(this->menuBar()->addAction(tr("Download"), this, SLOT(addVideoDownloads()))),
    m_audioDownloadAction(this->menuBar()->addAction(tr("Download (audio)"), this, SLOT(addAudioDownloads()))),
    m_contextMenu(new QMenu(this)),
    m_deleteFromPlaylistContextAction(m_contextMenu->addAction(tr("Delete from playlist"), this, SLOT(deleteVideoFromPlaylist()))),
    m_addToFavouritesContextAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(addVideoToFavourites()))),
    m_addToPlaylistContextAction(m_contextMenu->addAction(tr("Add to playlist"), this, SLOT(showPlaylistDialog()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_titleLabel(new QLabel(QString("<p style='font-size:32px;'>%1</p>").arg(m_playlist.data()->title()))),
    m_descriptionLabel(new QLabel(m_playlist.data()->description().isEmpty() ? tr("No description") : m_playlist.data()->description(), this)),
    m_dateLabel(new QLabel(QString("%1 - %2").arg(m_playlist.data()->date()).arg(m_playlist.data()->artist()), this)),
    m_thumbnail(new ThumbnailLabel(this)),
    m_avatar(new ThumbnailLabel(this)),
    m_playButton(new QPushButton(tr("Play all"), this)),
    m_separator1(new SeparatorLabel(tr("Description"), this)),
    m_separator2(new SeparatorLabel(QString("-- %1").arg(tr("videos")), this)),
    m_scrollArea(new QScrollArea(this))
{
    this->setWindowTitle(tr("Playlist"));
    this->setCentralWidget(m_scrollArea);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);
    m_descriptionLabel->setWordWrap(true);
    m_dateLabel->setWordWrap(true);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    m_thumbnail->setInteractive(false);
    m_thumbnail->setFixedSize(160, 120);
    m_avatar->setFixedSize(60, 60);
    m_playButton->setFixedWidth(150);

    grid->setContentsMargins(0, 0, 0, 0);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Maximum), 0, 0);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Maximum), 1, 0);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Maximum), 3, 0);
    grid->addWidget(m_titleLabel, 0, 1, 1, 3, Qt::AlignTop);
    grid->addWidget(m_thumbnail, 1, 1, Qt::AlignTop);
    grid->addWidget(m_avatar, 1, 2, Qt::AlignTop);
    grid->addWidget(m_dateLabel, 1, 3, Qt::AlignTop);
    grid->addWidget(m_playButton, 2, 3, Qt::AlignRight);
    grid->addWidget(m_separator1, 3, 0, 1, 4);
    grid->addWidget(m_descriptionLabel, 4, 1, 1, 3);
    grid->addWidget(m_separator2, 5, 0, 1, 4);
    grid->addWidget(m_listView, 6, 0, 1, 4);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_avatar, SIGNAL(clicked()), this, SLOT(showUserChannel()));
    this->connect(m_playButton, SIGNAL(clicked()), this, SLOT(playPlaylist()));

    m_model->getPlaylistVideos(m_playlist.data()->playlistId());

    this->setThumbnails();
    this->setMenuActions();
    this->onOrientationChanged();
}

void DailymotionPlaylistVideosWindow::showEvent(QShowEvent *event) {
    StackedWindow::showEvent(event);

    if ((m_model) && (m_model->refreshRequired())) {
        QTimer::singleShot(1000, m_model, SLOT(reset()));
    }
}

void DailymotionPlaylistVideosWindow::setLandscapeLayout() {
    m_scrollArea->widget()->setMaximumWidth(800);
}

void DailymotionPlaylistVideosWindow::setPortraitLayout() {
    m_scrollArea->widget()->setMaximumWidth(480);
}

void DailymotionPlaylistVideosWindow::setThumbnails() {
    QImage thumbnail;

    if (m_model->rowCount() > 0) {
        thumbnail = ThumbnailCache::instance()->thumbnail(m_model->get(0).data()->thumbnailUrl());
    }

    QImage avatar = ThumbnailCache::instance()->thumbnail(m_playlist.data()->artistAvatarUrl(), QSize(60, 60));

    if ((thumbnail.isNull()) || (avatar.isNull())) {
        this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    else {
        this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    if (!thumbnail.isNull()) {
        m_thumbnail->setImage(thumbnail);
    }
    if (!avatar.isNull()) {
        m_avatar->setImage(avatar);
    }
}

void DailymotionPlaylistVideosWindow::setMenuActions() {
    bool signedIn = Dailymotion::instance()->userSignedIn();

    m_addToFavouritesAction->setEnabled(signedIn);
    m_addToPlaylistAction->setEnabled(signedIn);
}

void DailymotionPlaylistVideosWindow::onLoadingChanged(bool loading) {
    this->toggleBusy(loading);
    m_playButton->setEnabled((!loading) && (m_model->rowCount() > 0) && (Settings::instance()->mediaPlayer() == "cutetube"));

    if (!m_model->loading()) {
        m_separator2->setText(QString("%1 %2").arg(m_model->rowCount()).arg(tr("videos")));

        if (m_model->rowCount() > 0) {
            m_listView->setFixedHeight(m_model->rowCount() * 150);
            this->setThumbnails();
        }
    }
}

void DailymotionPlaylistVideosWindow::onCountChanged(int count) {
    if (!m_model->loading()) {
        m_separator2->setText(QString("%1 %2").arg(count).arg(tr("videos")));
    }
}

void DailymotionPlaylistVideosWindow::deleteVideosFromPlaylist() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_model, tr("Delete from playlist"), SLOT(deleteVideosFromPlaylist()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void DailymotionPlaylistVideosWindow::addVideosToFavourites() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_model, tr("Add to favourites"), SLOT(addVideosToFavourites()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void DailymotionPlaylistVideosWindow::addVideosToPlaylist() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_model, tr("Add to playlist"), SLOT(showPlaylistDialog()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void DailymotionPlaylistVideosWindow::addVideoDownloads() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_model, tr("Download"), SLOT(addVideoDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void DailymotionPlaylistVideosWindow::addAudioDownloads() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_model, tr("Download"), SLOT(addAudioDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void DailymotionPlaylistVideosWindow::queueVideos() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_model, tr("Add to queue"), SLOT(queueVideos()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void DailymotionPlaylistVideosWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_deleteFromPlaylistContextAction);
}

void DailymotionPlaylistVideosWindow::deleteVideoFromPlaylist() {
    Dailymotion::instance()->deleteFromPlaylist(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()), m_model->playlistId());
}

void DailymotionPlaylistVideosWindow::addVideoToFavourites() {
    Dailymotion::instance()->addToFavourites(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void DailymotionPlaylistVideosWindow::showPlaylistDialog() {
    DailymotionPlaylistDialog *dialog = new DailymotionPlaylistDialog(this);
    dialog->open();

    this->connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    this->connect(dialog, SIGNAL(addToNewPlaylist(QVariantMap)), this, SLOT(addVideoToNewPlaylist(QVariantMap)));
}

void DailymotionPlaylistVideosWindow::addVideoToPlaylist(const QString &id) {
    Dailymotion::instance()->addToPlaylist(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()), id);
}

void DailymotionPlaylistVideosWindow::addVideoToNewPlaylist(const QVariantMap &playlist) {
    Dailymotion::instance()->createPlaylist(playlist, QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void DailymotionPlaylistVideosWindow::addVideoDownload() {
    TransferManager::instance()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionPlaylistVideosWindow::addAudioDownload() {
    TransferManager::instance()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void DailymotionPlaylistVideosWindow::queueVideo() {
    PlaybackListModel::playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionPlaylistVideosWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        this->showVideoInfo(m_model->get(index.row()));
    }
}

void DailymotionPlaylistVideosWindow::showVideoInfo(QSharedPointer<VideoItem> video) {
    DailymotionVideoInfoWindow *window = new DailymotionVideoInfoWindow(video, this);
    window->show();
}

void DailymotionPlaylistVideosWindow::playVideo(const QModelIndex &index) {
    if (Settings::instance()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(QList< QSharedPointer<VideoItem> >() << m_model->get(index.row()), this);
        window->show();
    }
    else {
        UrlGrabber::instance()->getDailymotionVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
    }
}

void DailymotionPlaylistVideosWindow::playPlaylist() {
    QList< QSharedPointer<VideoItem> > videos;

    for (int i = 0; i < m_model->rowCount(); i++) {
        videos.append(m_model->get(i));
    }

    VideoPlaybackWindow *window = new VideoPlaybackWindow(videos, this);
    window->show();
}

void DailymotionPlaylistVideosWindow::showUserChannel() {
    DailymotionChannelWindow *window = new DailymotionChannelWindow(m_playlist.data()->artistId(), this);
    window->show();
}
