#ifndef SHARING_H
#define SHARING_H

#include "videoitem.h"
#include <QObject>
#include <QStringList>
#include <QUrl>
#include <QDateTime>
#include <qplatformdefs.h>

class QNetworkAccessManager;

class Sharing : public QObject
{
    Q_OBJECT

    Q_PROPERTY(bool userSignedInToFacebook
               READ userSignedInToFacebook
               NOTIFY userSignedInToFacebookChanged)
    Q_PROPERTY(bool userSignedInToTwitter
               READ userSignedInToTwitter
               NOTIFY userSignedInToTwitterChanged)
    Q_PROPERTY(QUrl facebookAuthUrl
               READ facebookAuthUrl
               CONSTANT)
    Q_PROPERTY(bool busy
               READ busy
               NOTIFY busyChanged)

public:
    explicit Sharing(QObject *parent = 0);
    ~Sharing();
    inline QNetworkAccessManager* networkAccessManager() const { return m_nam; }
    inline void setNetworkAccessManager(QNetworkAccessManager *manager) { m_nam = manager; }
    inline bool userSignedInToFacebook() { return !this->facebookToken().isEmpty(); }
    inline bool userSignedInToTwitter() { return (!this->twitterToken().isEmpty()) && (!this->twitterTokenSecret().isEmpty()); }
#ifdef MEEGO_EDITION_HARMATTAN
    Q_INVOKABLE void shareViaHarmattanShareUI(VideoItem *video);
#else
    Q_INVOKABLE void shareViaEmail(const QString &title, const QUrl &url);
    Q_INVOKABLE void copyToClipboard(const QUrl &url);
#endif
    Q_INVOKABLE void signInToFacebook(const QUrl &response);
    Q_INVOKABLE void signOutOfFacebook();
    Q_INVOKABLE void signInToTwitter(const QUrl &response);
    Q_INVOKABLE void signOutOfTwitter();
    Q_INVOKABLE void getTwitterAuthUrl();
    QUrl facebookAuthUrl() const;
    inline QList< QSharedPointer<VideoItem> > * videos() const { return m_videoCache; }
    inline bool busy() const { return m_busy; }

    static Sharing* instance();

public slots:
#ifndef MEEGO_EDITION_HARMATTAN
    void postToFacebook(const QVariantMap &post);
    void postToTwitter(const QVariantMap &post);
#endif
    void cancelFacebookSignIn();
    void cancelTwitterSignIn();
    void setFacebookAccessToken(const QString &token);
    void setTwitterAccount(const QString &token, const QString &secret);
    void getVideos(bool forceRefresh = false);
    void addVideosToCache(QList< QSharedPointer<VideoItem> > videos);
    void cancelCurrentOperation();

private slots:
#ifndef MEEGO_EDITION_HARMATTAN
    void postToFacebook();
    void postToTwitter();
    void facebookPostFinished();
    void twitterPostFinished();
#endif
    void getVideosFromFacebook();
    void getVideosFromTwitter();
    void checkFacebookVideos();
    void checkTwitterVideos();
    void checkTwitterAuthUrl();
    void checkIfSignedInToTwitter();

private:
    void clearCache();
    inline QString facebookToken() const { return m_facebookToken; }
    inline QString twitterToken() const { return m_twitterToken; }
    inline void setTwitterAccessToken(const QString &token) { m_twitterToken = token; }
    inline QString twitterTokenSecret() const { return m_twitterSecret; }
    inline void setTwitterTokenSecret(const QString &secret) { m_twitterSecret = secret; }
    inline QString twitterRequestToken() const { return m_twitterRequestToken; }
    inline void setTwitterRequestToken(const QString &token) { m_twitterRequestToken = token; }
    inline QString twitterRequestSecret() const { return m_twitterRequestSecret; }
    inline void setTwitterRequestSecret(const QString &secret) { m_twitterRequestSecret = secret; }
    QByteArray getTwitterOAuthHeader(const QString &method, const QUrl &url, QMap<QString, QString> params);
    QString lastTweet() const { return m_lastTweetId; }
    void setLastTweet(const QString &id) { m_lastTweetId = id; }
    inline qint64 facebookTimeStamp() const { return m_facebookTimestamp; }
    inline void setFacebookTimeStamp(qint64 timestamp) { m_facebookTimestamp = timestamp; }
    inline bool videoRefreshIsDue() const { return (m_nextVideosRefresh.elapsed() == 0) || (m_nextVideosRefresh.elapsed() > 600000); }
    inline void resetVideoRefreshTime() { m_nextVideosRefresh.restart(); }
    void setBusy(bool isBusy, const QString &message = QString(), int numberOfOperations = 1);

signals:
    void alert(const QString &message);
    void busy(const QString &message, int numberOfOperations = 1);
    void busyProgressChanged(int progress);
    void busyChanged(bool isBusy);
    void info(const QString &message);
    void error(const QString &errorString);
    void userSignedInToFacebookChanged();
    void userSignedInToTwitterChanged();
    void signedInToFacebook(const QString &token);
    void signedOutOfFacebook();
    void gotTwitterAuthUrl(const QUrl &url);
    void signedInToTwitter(const QString &token, const QString &secret);
    void signedOutOfTwitter();
    void reauthenticateForFacebook();
    void reauthenticateForTwitter();
    void gotYouTubeIds(const QStringList &ids);
    void gotDailymotionIds(const QStringList &ids);
    void gotVimeoIds(const QStringList &ids);
    void videoAddedToCache(int row);
    void currentOperationCancelled();

private:
    QNetworkAccessManager *m_nam;
    QString m_facebookToken;
    QString m_twitterToken;
    QString m_twitterSecret;
    QString m_twitterRequestToken;
    QString m_twitterRequestSecret;
    qint64 m_facebookTimestamp;
    QString m_lastTweetId;
    QTime m_nextVideosRefresh;
    QVariantMap m_post;
    QList< QSharedPointer<VideoItem> > *m_videoCache;
    bool m_busy;
};

#endif // SHARING_H
