#include "youtubevideolistwindow.h"
#include "youtubevideoinfowindow.h"
#include "youtubeplaylistdialog.h"
#include "youtubevideoselectionwindow.h"
#include "videolistdelegate.h"
#include "videoplaybackwindow.h"
#include "../base/session.h"
#include "../base/youtubevideolistmodel.h"
#include "listview.h"
#include <QAction>
#include <QLabel>
#include <QMenuBar>
#include <QTimer>

YouTubeVideoListWindow::YouTubeVideoListWindow(Session *session, QWidget *parent) :
    StackedWindow(session, parent),
    m_listView(new ListView(this)),
    m_model(new YouTubeVideoListModel(m_session, this)),
    m_deleteFromUploadsAction(this->menuBar()->addAction(tr("Delete from uploads"), this, SLOT(deleteVideosFromUploads()))),
    m_deleteFromWatchLaterAction(this->menuBar()->addAction(tr("Delete from playlist"), this, SLOT(deleteVideosFromWatchLaterPlaylist()))),
    m_addToFavouritesAction(this->menuBar()->addAction(tr("Add to favourites"), this, SLOT(addVideosToFavourites()))),
    m_deleteFromFavouritesAction(this->menuBar()->addAction(tr("Delete from favourites"), this, SLOT(deleteVideosFromFavourites()))),
    m_addToPlaylistAction(this->menuBar()->addAction(tr("Add to playlist"), this, SLOT(addVideosToPlaylist()))),
    m_addToWatchLaterAction(this->menuBar()->addAction(tr("Watch later"), this, SLOT(addVideosToWatchLaterPlaylist()))),
    m_addToQueueAction(this->menuBar()->addAction(tr("Add to playback queue"), this, SLOT(queueVideos()))),
    m_videoDownloadAction(this->menuBar()->addAction(tr("Download"), this, SLOT(addVideoDownloads()))),
    m_audioDownloadAction(this->menuBar()->addAction(tr("Download (audio)"), this, SLOT(addAudioDownloads()))),
    m_contextMenu(new QMenu(this)),
    m_deleteFromUploadsContextAction(m_contextMenu->addAction(tr("Delete from uploads"), this, SLOT(deleteVideoFromUploads()))),
    m_deleteFromWatchLaterContextAction(m_contextMenu->addAction(tr("Delete from playlist"), this, SLOT(deleteVideoFromWatchLaterPlaylist()))),
    m_addToFavouritesContextAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(addVideoToFavourites()))),
    m_deleteFromFavouritesContextAction(m_contextMenu->addAction(tr("Delete from favourites"), this, SLOT(deleteVideoFromFavourites()))),
    m_addToPlaylistContextAction(m_contextMenu->addAction(tr("Add to playlist"), this, SLOT(showPlaylistDialog()))),
    m_addToWatchLaterContextAction(m_contextMenu->addAction(tr("Watch later"), this, SLOT(addVideoToWatchLaterPlaylist()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this))
{
    this->setCentralWidget(m_listView);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);
    m_noResultsLabel->hide();

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreVideos()));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));

    this->onOrientationChanged();
}

void YouTubeVideoListWindow::showEvent(QShowEvent *event) {
    StackedWindow::showEvent(event);

    if ((m_model) && (m_model->refreshRequired())) {
        QTimer::singleShot(1000, m_model, SLOT(reset()));
    }
}

void YouTubeVideoListWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void YouTubeVideoListWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void YouTubeVideoListWindow::setVideoFeed(const QString &feed) {
    m_model->getVideos(feed);
    this->setMenuActions();
}

void YouTubeVideoListWindow::search(const QString &query, Queries::QueryOrder order, Queries::TimeFilter time, Queries::DurationFilter duration, const QString &language) {
    this->setWindowTitle(tr("Search") + QString(" ('%1')").arg(query));
    m_model->search(query, order, time, duration, language);
    this->setMenuActions();
}

void YouTubeVideoListWindow::setMenuActions() {
    bool signedIn = m_session->youtube()->userSignedIn();

    m_deleteFromUploadsAction->setEnabled((signedIn) && (m_model->feed() == YOUTUBE_UPLOADS_FEED));
    m_addToFavouritesAction->setEnabled((signedIn) && (m_model->feed() != YOUTUBE_FAVOURITES_FEED));
    m_deleteFromFavouritesAction->setEnabled((signedIn) && (m_model->feed() == YOUTUBE_FAVOURITES_FEED));
    m_addToPlaylistAction->setEnabled(signedIn);
    m_addToWatchLaterAction->setEnabled((signedIn) && (m_model->feed() != YOUTUBE_WATCH_LATER_FEED));
    m_deleteFromWatchLaterAction->setEnabled((signedIn) && (m_model->feed() == YOUTUBE_WATCH_LATER_FEED));
    m_deleteFromUploadsContextAction->setVisible((signedIn) && (m_model->feed() == YOUTUBE_UPLOADS_FEED));
    m_addToFavouritesContextAction->setVisible((signedIn) && (m_model->feed() != YOUTUBE_FAVOURITES_FEED));
    m_deleteFromFavouritesContextAction->setVisible((signedIn) && (m_model->feed() == YOUTUBE_FAVOURITES_FEED));
    m_addToPlaylistContextAction->setVisible(signedIn);
    m_addToWatchLaterContextAction->setVisible((signedIn) && (m_model->feed() != YOUTUBE_WATCH_LATER_FEED));
    m_deleteFromWatchLaterContextAction->setVisible((signedIn) && (m_model->feed() == YOUTUBE_WATCH_LATER_FEED));
}

void YouTubeVideoListWindow::onLoadingChanged(bool loading) {
    this->toggleBusy(loading);
    m_noResultsLabel->setVisible((!loading) && (m_model->rowCount() == 0));
}

void YouTubeVideoListWindow::onCountChanged(int count) {
    m_noResultsLabel->setVisible((count == 0) && (!m_model->loading()));
}

void YouTubeVideoListWindow::deleteVideosFromUploads() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Delete from uploads"), SLOT(deleteVideosFromUploads()), this);
    window->show();
}

void YouTubeVideoListWindow::addVideosToFavourites() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Add to favourites"), SLOT(addVideosToFavourites()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::deleteVideosFromFavourites() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Delete from favourites"), SLOT(deleteVideosFromFavourites()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::addVideosToPlaylist() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Add to playlist"), SLOT(showPlaylistDialog()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::addVideosToWatchLaterPlaylist() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Watch later"), SLOT(addVideosToWatchLaterPlaylist()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::deleteVideosFromWatchLaterPlaylist() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Delete from playlist"), SLOT(deleteVideosFromWatchLaterPlaylist()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::addVideoDownloads() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Download"), SLOT(addVideoDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::addAudioDownloads() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Download"), SLOT(addAudioDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::queueVideos() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_session, m_model, tr("Add to queue"), SLOT(queueVideos()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_deleteFromUploadsContextAction);
}

void YouTubeVideoListWindow::deleteVideoFromUploads() {
    m_session->youtube()->deleteFromUploads(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void YouTubeVideoListWindow::addVideoToFavourites() {
    m_session->youtube()->addToFavourites(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void YouTubeVideoListWindow::deleteVideoFromFavourites() {
    m_session->youtube()->deleteFromFavourites(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::FavouriteIdRole).toString()));
}

void YouTubeVideoListWindow::showPlaylistDialog() {
    YouTubePlaylistDialog *dialog = new YouTubePlaylistDialog(m_session, this);
    dialog->open();

    this->connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    this->connect(dialog, SIGNAL(addToNewPlaylist(QVariantMap)), this, SLOT(addVideoToNewPlaylist(QVariantMap)));
}

void YouTubeVideoListWindow::addVideoToPlaylist(const QString &id) {
    m_session->youtube()->addToPlaylist(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()), id);
}

void YouTubeVideoListWindow::addVideoToNewPlaylist(const QVariantMap &playlist) {
    m_session->youtube()->createPlaylist(playlist, QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void YouTubeVideoListWindow::addVideoToWatchLaterPlaylist() {
    m_session->youtube()->addToWatchLaterPlaylist(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void YouTubeVideoListWindow::deleteVideoFromWatchLaterPlaylist() {
    m_session->youtube()->deleteFromWatchLaterPlaylist(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::PlaylistVideoIdRole).toString()));
}

void YouTubeVideoListWindow::addVideoDownload() {
    m_session->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void YouTubeVideoListWindow::addAudioDownload() {
    m_session->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void YouTubeVideoListWindow::queueVideo() {
    m_session->playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}

void YouTubeVideoListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        this->showVideoInfo(m_model->get(index.row()));
    }
}

void YouTubeVideoListWindow::showVideoInfo(QSharedPointer<VideoItem> video) {
    YouTubeVideoInfoWindow *window = new YouTubeVideoInfoWindow(m_session, video, this);
    window->show();
}

void YouTubeVideoListWindow::playVideo(const QModelIndex &index) {
    if (m_session->settings()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(m_session, QList< QSharedPointer<VideoItem> >() << m_model->get(index.row()), this);
        window->show();
    }
    else {
        m_session->urlGrabber()->getYouTubeVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
    }
}
