#include "vimeovideodescriptiontab.h"
#include "vimeoalbumdialog.h"
#include "sharingdialog.h"
#include "sharetofacebookdialog.h"
#include "sharetotwitterdialog.h"
#include "facebookauthdialog.h"
#include "twitterauthdialog.h"
#include "vimeovideolistwindow.h"
#include "vimeochannelwindow.h"
#include "videoplaybackwindow.h"
#include "../base/session.h"
#include "../base/videoitem.h"
#include "thumbnaillabel.h"
#include "ratingindicator.h"
#include <QAction>
#include <QLabel>
#include <QScrollArea>
#include <QGridLayout>

VimeoVideoDescriptionTab::VimeoVideoDescriptionTab(Session *session, QSharedPointer<VideoItem> video, QWidget *parent) :
    AbstractTab(parent),
    m_session(session),
    m_video(video),
    m_likeAction(new QAction(m_video.data()->favourite() ? tr("Delete from likes") : tr("Add to likes"), this)),
    m_addToAlbumAction(new QAction(tr("Add to album"), this)),
    m_queueAction(new QAction(tr("Add to playback queue"), this)),
    m_videoDownloadAction(new QAction(tr("Download"), this)),
    m_audioDownloadAction(new QAction(tr("Download (audio)"), this)),
    m_shareAction(new QAction(tr("Share"), this)),
    m_thumbnail(new ThumbnailLabel(m_video.data()->duration(), this)),
    m_avatar(new ThumbnailLabel(this)),
    m_titleLabel(new QLabel(m_video.data()->title(), this)),
    m_dateLabel(new QLabel(QString("%1 - %2").arg(m_video.data()->date()).arg(m_video.data()->artist()), this)),
    m_likesLabel(new QLabel(QString("%1   <font color='#4d4d4d'>%2</font>").arg(tr("Likes")).arg(m_video.data()->likes()), this)),
    m_viewsLabel(new QLabel(QString("%1   <font color='#4d4d4d'>%2</font>").arg(tr("Views")).arg(m_video.data()->viewCount()), this)),
    m_descriptionLabel(new QLabel(QString("Description<br><font color='#4d4d4d'>%1</font>").arg(m_video.data()->description().isEmpty() ? tr("No description") : m_video.data()->description()), this)),
    m_tagsLabel(new QLabel(this->tagsToString(m_video.data()->tags()), this)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    this->addAction(m_likeAction);
    this->addAction(m_addToAlbumAction);
    this->addAction(m_queueAction);
    this->addAction(m_videoDownloadAction);
    this->addAction(m_audioDownloadAction);
    this->addAction(m_shareAction);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);
    QFont smallFont;
    smallFont.setPixelSize(20);
    m_avatar->setFixedSize(60, 60);
    m_titleLabel->setWordWrap(true);
    m_dateLabel->setWordWrap(true);
    m_dateLabel->setFont(smallFont);
    m_likesLabel->setFont(smallFont);
    m_viewsLabel->setFont(smallFont);
    m_descriptionLabel->setWordWrap(true);
    m_descriptionLabel->setFont(smallFont);
    m_tagsLabel->setWordWrap(true);
    m_tagsLabel->setFont(smallFont);

    grid->addWidget(m_titleLabel, 0, 0, 1, 2, Qt::AlignTop);
    grid->addWidget(m_avatar, 1, 0);
    grid->addWidget(m_dateLabel, 1, 1, Qt::AlignTop);
    grid->addWidget(m_likesLabel, 2, 0, 1, 2);
    grid->addWidget(m_viewsLabel, 3, 0, 1, 2, Qt::AlignTop);
    grid->addWidget(m_descriptionLabel, 4, 0, 1, 2, Qt::AlignTop);
    grid->addWidget(m_tagsLabel, 5, 0, 1, 2);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    this->connect(m_thumbnail, SIGNAL(clicked()), this, SLOT(playVideo()));
    this->connect(m_avatar, SIGNAL(clicked()), this, SLOT(showUserChannel()));
    this->connect(m_likeAction, SIGNAL(triggered()), this, SLOT(toggleLike()));
    this->connect(m_addToAlbumAction, SIGNAL(triggered()), this, SLOT(showAlbumDialog()));
    this->connect(m_queueAction, SIGNAL(triggered()), this, SLOT(addVideoToPlaybackQueue()));
    this->connect(m_videoDownloadAction, SIGNAL(triggered()), this, SLOT(addVideoDownload()));
    this->connect(m_audioDownloadAction, SIGNAL(triggered()), this, SLOT(addAudioDownload()));
    this->connect(m_shareAction, SIGNAL(triggered()), this, SLOT(showShareDialog()));
    this->connect(m_tagsLabel, SIGNAL(linkActivated(QString)), this, SLOT(search(QString)));
    this->connect(m_video.data(), SIGNAL(favouriteChanged(bool)), this, SLOT(onLikeChanged(bool)));
    this->connect(m_video.data(), SIGNAL(likesChanged(int)), this, SLOT(onLikesChanged(int)));

    this->setMenuActions();
    this->setThumbnails();
    this->onOrientationChanged();
}

void VimeoVideoDescriptionTab::setLandscapeLayout() {
    m_thumbnail->setFixedSize(280, 210);
    m_grid->setContentsMargins(10, 10, 0, 0);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_thumbnail, 0, 0, Qt::AlignTop | Qt::AlignHCenter);
        m_grid->addWidget(m_scrollArea, 0, 1, Qt::AlignTop);
    }
    else {
        m_grid->removeWidget(m_scrollArea);
        m_grid->addWidget(m_scrollArea, 0, 1, Qt::AlignTop);
    }
}

void VimeoVideoDescriptionTab::setPortraitLayout() {
    m_thumbnail->setFixedSize(460, 345);
    m_grid->setContentsMargins(0, 10, 0, 0);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_thumbnail, 0, 0, Qt::AlignTop | Qt::AlignHCenter);
        m_grid->addWidget(m_scrollArea, 1, 0, Qt::AlignTop);
    }
    else {
        m_grid->removeWidget(m_scrollArea);
        m_grid->addWidget(m_scrollArea, 1, 0, Qt::AlignTop);
    }
}

void VimeoVideoDescriptionTab::setMenuActions() {
    bool signedIn = m_session->vimeo()->userSignedIn();

    m_likeAction->setEnabled(signedIn);
    m_addToAlbumAction->setEnabled(signedIn);
}

QString VimeoVideoDescriptionTab::tagsToString(QStringList tags) {
    QString tagString("Tags<br>");

    if (tags.isEmpty()) {
        tagString.append(QString("<p style='color:#4d4d4d;'>%1</p>").arg(tr("No tags")));
    }
    else {
        while (!tags.isEmpty()) {
            tagString.append(QString("<a href='%1'>%1</a>   ").arg(tags.takeFirst()));
        }
    }

    return tagString;
}

void VimeoVideoDescriptionTab::setThumbnails() {
    QImage thumbnail = m_session->thumbnailCache()->thumbnail(m_video.data()->largeThumbnailUrl());
    QImage avatar = m_session->thumbnailCache()->thumbnail(m_video.data()->artistAvatarUrl(), QSize(60, 60));

    if ((thumbnail.isNull()) || (avatar.isNull())) {
        this->connect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    else {
        this->disconnect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    if (!thumbnail.isNull()) {
        m_thumbnail->setImage(thumbnail);
    }
    if (!avatar.isNull()) {
        m_avatar->setImage(avatar);
    }
}

void VimeoVideoDescriptionTab::toggleLike() {
    if (m_video.data()->favourite()) {
        m_session->vimeo()->deleteFromLikes(QStringList(m_video.data()->videoId()));
    }
    else {
        m_session->vimeo()->addToLikes(QStringList(m_video.data()->videoId()));
    }
}

void VimeoVideoDescriptionTab::onLikeChanged(bool like) {
    m_likeAction->setText(like ? tr("Delete from likes") : tr("Add to likes"));
}

void VimeoVideoDescriptionTab::onLikesChanged(int likes) {
    m_likesLabel->setText(QString("%1   <font color='#4d4d4d'>%2</font>").arg(tr("Likes")).arg(likes));
}

void VimeoVideoDescriptionTab::showAlbumDialog() {
    VimeoAlbumDialog *dialog = new VimeoAlbumDialog(m_session, this);
    dialog->open();
    this->connect(dialog, SIGNAL(albumChosen(QString)), this, SLOT(addVideoToAlbum(QString)));
    this->connect(dialog, SIGNAL(addToNewAlbum(QVariantMap)), this, SLOT(addVideoToNewAlbum(QVariantMap)));
}

void VimeoVideoDescriptionTab::showShareDialog() {
    SharingDialog *dialog = new SharingDialog(this);
    dialog->open();
    this->connect(dialog, SIGNAL(valueChanged(Services::SharingService)), this, SLOT(shareVideo(Services::SharingService)));
}

void VimeoVideoDescriptionTab::shareVideo(Services::SharingService service) {
    switch (service) {
    case Services::Facebook:
        this->showFacebookDialog();
        break;
    case Services::Twitter:
        this->showTwitterDialog();
        break;
    case Services::Email:
        m_session->sharing()->shareViaEmail(m_video.data()->title(), m_video.data()->url());
        break;
    case Services::Clipboard:
        m_session->sharing()->copyToClipboard(m_video.data()->url());
        break;
    default:
        return;
    }
}

void VimeoVideoDescriptionTab::showFacebookDialog() {
    if (m_session->sharing()->userSignedInToFacebook()) {
        ShareToFacebookDialog *dialog = new ShareToFacebookDialog(m_session, m_video, this);
        dialog->open();
    }
    else {
        FacebookAuthDialog *dialog = new FacebookAuthDialog(m_session, this);
        dialog->open();
    }
}

void VimeoVideoDescriptionTab::showTwitterDialog() {
    if (m_session->sharing()->userSignedInToTwitter()) {
        ShareToTwitterDialog *dialog = new ShareToTwitterDialog(m_session, m_video, this);
        dialog->open();
    }
    else {
        TwitterAuthDialog *dialog = new TwitterAuthDialog(m_session, this);
        dialog->open();
    }
}

void VimeoVideoDescriptionTab::addVideoToAlbum(const QString &id) {
    m_session->vimeo()->addToAlbum(QStringList(m_video.data()->videoId()), id);
}

void VimeoVideoDescriptionTab::addVideoToNewAlbum(const QVariantMap &album) {
    m_session->vimeo()->createAlbum(album, QStringList(m_video.data()->videoId()));
}

void VimeoVideoDescriptionTab::addVideoToPlaybackQueue() {
    m_session->playbackQueue()->addVideo(m_video);
}

void VimeoVideoDescriptionTab::addVideoDownload() {
    m_session->addDownloadTransfer(m_video);
}

void VimeoVideoDescriptionTab::addAudioDownload() {
    m_session->addDownloadTransfer(m_video, true);
}

void VimeoVideoDescriptionTab::playVideo() {
    if (m_session->settings()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(m_session, QList< QSharedPointer<VideoItem> >() << m_video, m_session->currentWindow());
        window->show();
    }
    else {
        m_session->urlGrabber()->getVimeoVideoUrl(m_video.data()->videoId());
    }
}

void VimeoVideoDescriptionTab::showUserChannel() {
    VimeoChannelWindow *window = new VimeoChannelWindow(m_session, m_video.data()->artistId(), m_session->currentWindow());
    window->show();
}

void VimeoVideoDescriptionTab::search(const QString &query) {
    VimeoVideoListWindow *window = new VimeoVideoListWindow(m_session, m_session->currentWindow());
    window->search(query, m_session->settings()->searchOrder());
    window->show();
}
