#include "searchdialog.h"
#include "searchhistorydialog.h"
#include "videoserviceselector.h"
#include "searchtypeselector.h"
#include "searchorderselector.h"
#include "searchtimeselector.h"
#include "searchdurationselector.h"
#include "languageselector.h"
#include "searchhistorydialog.h"
#include "../base/searchlistmodel.h"
#include "../base/session.h"
#include <QLineEdit>
#include <QPushButton>
#include <QScrollArea>
#include <QDialogButtonBox>
#include <QGridLayout>
#include <QClipboard>
#include <QRegExp>
#include <QApplication>

SearchDialog::SearchDialog(Session *session, QWidget *parent) :
    AbstractDialog(parent),
    m_session(session),
    m_searchEdit(new QLineEdit(this)),
    m_searchServiceSelector(m_session->settings()->showSearchSite() ? new VideoServiceSelector(tr("Service"), this) : 0),
    m_searchTypeSelector(m_session->settings()->showSearchType() ? new SearchTypeSelector(tr("Search for"), this) : 0),
    m_searchOrderSelector(m_session->settings()->showSearchOrder() ? new SearchOrderSelector(tr("Order by"), this) : 0),
    m_searchTimeSelector(m_session->settings()->showSearchTime() ? new SearchTimeSelector(tr("Show results from"), this) : 0),
    m_searchDurationSelector(m_session->settings()->showSearchDuration() ? new SearchDurationSelector(tr("Show videos of duration"), this) : 0),
    m_searchLanguageSelector(m_session->settings()->showSearchLanguage() ? new LanguageSelector(tr("Show videos with language"), false, this) : 0),
    m_buttonBox(new QDialogButtonBox(Qt::Vertical, this)),
    m_historyButton(m_buttonBox->addButton(tr("History"), QDialogButtonBox::ActionRole)),
    m_searchButton(m_buttonBox->addButton(m_session->clipboardUrl().isEmpty() ? tr("Search") : tr("From URL"), QDialogButtonBox::ActionRole)),
    m_scrollArea(new QScrollArea(this)),
    m_model(new SearchListModel(this)),
    m_historyDialog(0),
    m_grid(new QGridLayout(this))
{
    this->setWindowTitle(tr("Search"));
    m_searchEdit->setPlaceholderText(tr("Search query"));

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);

    vbox->addWidget(m_searchEdit);

    if (m_searchServiceSelector) {
        this->connect(m_searchServiceSelector, SIGNAL(valueChanged(Services::VideoService)), m_session->settings(), SLOT(setSearchSite(Services::VideoService)));
        this->connect(m_session->settings(), SIGNAL(searchSiteChanged(Services::VideoService)), this, SLOT(onServiceChanged(Services::VideoService)));
        m_searchServiceSelector->setValue(m_session->settings()->searchSite());
        vbox->addWidget(m_searchServiceSelector);
    }
    if (m_searchTypeSelector) {
        this->connect(m_searchTypeSelector, SIGNAL(valueChanged(Queries::QueryType)), m_session->settings(), SLOT(setSearchType(Queries::QueryType)));
        this->connect(m_session->settings(), SIGNAL(searchTypeChanged(Queries::QueryType)), this, SLOT(onSearchTypeChanged(Queries::QueryType)));
        m_searchTypeSelector->setValue(m_session->settings()->searchType());
        vbox->addWidget(m_searchTypeSelector);
    }
    if (m_searchOrderSelector) {
        this->connect(m_searchOrderSelector, SIGNAL(valueChanged(Queries::QueryOrder)), m_session->settings(), SLOT(setSearchOrder(Queries::QueryOrder)));
        m_searchOrderSelector->setValue(m_session->settings()->searchOrder());
        vbox->addWidget(m_searchOrderSelector);
    }
    if (m_searchTimeSelector) {
        this->connect(m_searchTimeSelector, SIGNAL(valueChanged(Queries::TimeFilter)), m_session->settings(), SLOT(setSearchTime(Queries::TimeFilter)));
        m_searchTimeSelector->setValue(m_session->settings()->searchTime());
        vbox->addWidget(m_searchTimeSelector);
    }
    if (m_searchDurationSelector) {
        this->connect(m_searchDurationSelector, SIGNAL(valueChanged(Queries::DurationFilter)), m_session->settings(), SLOT(setSearchDuration(Queries::DurationFilter)));
        m_searchDurationSelector->setValue(m_session->settings()->searchDuration());
        vbox->addWidget(m_searchDurationSelector);
    }    
    if (m_searchLanguageSelector) {
        this->connect(m_searchLanguageSelector, SIGNAL(valueChanged(QString)), m_session->settings(), SLOT(setLanguageFilter(QString)));
        m_searchLanguageSelector->setValue(m_session->settings()->languageFilter());
        vbox->addWidget(m_searchLanguageSelector);
    }

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    this->connect(m_searchEdit, SIGNAL(returnPressed()), this, SLOT(parseSearchQuery()));
    this->connect(m_searchEdit, SIGNAL(textChanged(QString)), this, SLOT(onSearchTextChanged(QString)));
    this->connect(m_historyButton, SIGNAL(clicked()), this, SLOT(showSearchHistory()));
    this->connect(m_searchButton, SIGNAL(clicked()), this, SLOT(parseSearchQuery()));

    this->onOrientationChanged();
}

void SearchDialog::onServiceChanged(Services::VideoService service) {
    if (m_searchTypeSelector) {
        m_searchTypeSelector->showGroups(service == Services::Dailymotion);
        m_searchTypeSelector->setEnabled((service == Services::YouTube) || (service == Services::Dailymotion));

        if (((m_session->settings()->searchType() == Queries::Groups) && (service != Services::Dailymotion)) || ((service != Services::YouTube) && (service != Services::Dailymotion))) {
            m_searchTypeSelector->setValue(Queries::Videos);
        }
    }
    if (m_searchTimeSelector) {
        m_searchTimeSelector->setEnabled((service == Services::YouTube) && (m_session->settings()->searchType() == Queries::Videos));
    }
    if (m_searchDurationSelector) {
        m_searchDurationSelector->setEnabled((service == Services::YouTube) && (m_session->settings()->searchType() == Queries::Videos));
    }
    if (m_searchLanguageSelector) {
        m_searchLanguageSelector->setEnabled(((service == Services::YouTube) || (service == Services::Dailymotion)) && (m_session->settings()->searchType() == Queries::Videos));
    }
}

void SearchDialog::onSearchTypeChanged(Queries::QueryType type) {
    if (m_searchOrderSelector) {
        m_searchOrderSelector->setEnabled(type == Queries::Videos);
    }
    if (m_searchTimeSelector) {
        m_searchTimeSelector->setEnabled((m_session->settings()->searchSite() == Services::YouTube) && (type == Queries::Videos));
    }
    if (m_searchDurationSelector) {
        m_searchDurationSelector->setEnabled((m_session->settings()->searchSite() == Services::YouTube) && (type == Queries::Videos));
    }
    if (m_searchLanguageSelector) {
        m_searchLanguageSelector->setEnabled(type == Queries::Videos);
    }
}

void SearchDialog::setLandscapeLayout() {
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }

    this->setFixedHeight(340);
}

void SearchDialog::setPortraitLayout() {
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }

    this->setFixedHeight(qMin(600, m_scrollArea->widget()->height() + 160));
}

void SearchDialog::showSearchHistory() {
    if (!m_historyDialog) {
        m_historyDialog = new SearchHistoryDialog(m_model, this);
        this->connect(m_historyDialog, SIGNAL(queryChosen(QString)), m_searchEdit, SLOT(setText(QString)));
    }

    m_historyDialog->open();
}

void SearchDialog::onSearchTextChanged(const QString &text) {
    if ((text.isEmpty()) && (!m_session->clipboardUrl().isEmpty())) {
        m_searchButton->setText(tr("From URL"));
    }
    else {
        m_searchButton->setText(tr("Search"));
    }
}

void SearchDialog::parseSearchQuery() {
    if ((!m_session->clipboardUrl().isEmpty()) && (m_searchEdit->text().isEmpty())) {
        emit goToVideo(m_session->clipboardUrl());
    }
    else {
        m_model->addSearch(m_searchEdit->text());
        emit search(m_searchEdit->text());
    }

    this->accept();
}
