#include "linkgoogleaccountdialog.h"
#include "textlistdelegate.h"
#include "../base/session.h"
#include <QLineEdit>
#include <QPushButton>
#include <QDialogButtonBox>
#include <QScrollArea>
#include <QGridLayout>
#include <QStringListModel>
#include <QListView>
#include <QLabel>

SuggestedUsernamesDialog::SuggestedUsernamesDialog(QWidget *parent) :
    AbstractDialog(parent),
    m_model(new QStringListModel(this)),
    m_listView(new QListView(this)),
    m_grid(new QGridLayout(this))
{
    this->setWindowTitle(tr("Suggested usernames"));
    this->setAttribute(Qt::WA_DeleteOnClose, false);

    m_grid->addWidget(m_listView, 0, 0);
    m_listView->setModel(m_model);
    m_listView->setItemDelegate(new TextListDelegate(Qt::DisplayRole, Qt::AlignCenter, m_listView));

    this->connect(m_listView, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));

    this->onOrientationChanged();
}

void SuggestedUsernamesDialog::setLandscapeLayout() {
    this->setFixedHeight(340);
}

void SuggestedUsernamesDialog::setPortraitLayout() {
    this->setFixedHeight(600);
}

void SuggestedUsernamesDialog::setSuggestedUsernames(const QStringList &usernames) {
    m_model->setStringList(usernames);
}

void SuggestedUsernamesDialog::onItemClicked(const QModelIndex &index) {
    emit usernameChosen(index.data().toString());
    this->accept();
}

LinkGoogleAccountDialog::LinkGoogleAccountDialog(Session *session, QWidget *parent) :
    AbstractDialog(parent),
    m_session(session),
    m_usernameEdit(new QLineEdit(this)),
    m_messageLabel(new QLabel(tr("This feature requires your Google account to be linked to a YouTube profile. To create a YouTube profile, enter a username above."), this)),
    m_buttonBox(new QDialogButtonBox(Qt::Vertical, this)),
    m_doneButton(m_buttonBox->addButton(tr("Done"), QDialogButtonBox::ActionRole)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this)),
    m_dialog(0)
{
    this->setWindowTitle(tr("Link Google account"));

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);

    vbox->addWidget(m_usernameEdit);
    vbox->addWidget(m_messageLabel, Qt::AlignTop);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_doneButton->setEnabled(false);
    m_usernameEdit->setPlaceholderText(tr("Username"));
    m_messageLabel->setWordWrap(true);

    this->connect(m_usernameEdit, SIGNAL(textChanged(QString)), this, SLOT(onUsernameTextChanged(QString)));
    this->connect(m_doneButton, SIGNAL(clicked()), this, SLOT(checkUsername()));
    this->connect(m_session->youtube(), SIGNAL(gotSuggestedUsernames(QStringList)), this, SLOT(showSuggestedUsernames(QStringList)));
    this->connect(m_session->youtube(), SIGNAL(usernameAvailable()), this, SLOT(onUsernameAvailable()));
    this->connect(m_session->youtube(), SIGNAL(usernameUnavailable()), this, SLOT(onUsernameUnavailable()));
    this->connect(m_session->youtube(), SIGNAL(googleAccountLinked()), this, SLOT(accept()));

    this->onOrientationChanged();
}

void LinkGoogleAccountDialog::setLandscapeLayout() {
    this->setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
}

void LinkGoogleAccountDialog::setPortraitLayout() {
    this->setFixedHeight(600);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
}

void LinkGoogleAccountDialog::onUsernameTextChanged(const QString &text) {
    m_doneButton->setEnabled(!text.isEmpty());
}

void LinkGoogleAccountDialog::onUsernameAvailable() {
    this->toggleBusy(false);

    m_messageLabel->setText(QString("<p style='color:green;'>%1</p>").arg(tr("Username is available. Linking account...")));
}

void LinkGoogleAccountDialog::onUsernameUnavailable() {
    this->toggleBusy(false);

    m_messageLabel->setText(QString("<p style='color:red;'>%1</p>").arg(tr("Username is unavailable. Please try another")));
    m_usernameEdit->clear();
}

void LinkGoogleAccountDialog::showSuggestedUsernames(const QStringList &usernames) {
    this->toggleBusy(false);

    if (!m_dialog) {
        m_dialog = new SuggestedUsernamesDialog(this);

        this->connect(m_dialog, SIGNAL(usernameChosen(QString)), m_usernameEdit, SLOT(setText(QString)));
    }

    m_dialog->setSuggestedUsernames(usernames);
    m_dialog->open();
}

void LinkGoogleAccountDialog::checkUsername() {
    this->toggleBusy(true);

    m_session->youtube()->checkUsernameAvailability(m_usernameEdit->text());
}
