#include "youtubeuserplaylistlistmodel.h"
#include "session.h"
#ifdef QML_USER_INTERFACE
#include <QDeclarativeEngine>
#endif

YouTubeUserPlaylistListModel::YouTubeUserPlaylistListModel(Session *session, QObject *parent) :
    QAbstractListModel(parent),
    m_session(session),
    m_cache(!session ? 0 : session->newThumbnailCache())
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[PlaylistIdRole] = "playlistId";
    roles[TitleRole] = "title";
    roles[ArtistRole] = "artist";
    roles[DateRole] = "date";
    roles[VideoCountRole] = "videoCount";
    roles[ThumbnailRole] = "thumbnail";
    roles[ThumbnailUrlRole] = "thumbnailUrl";
    roles[DescriptionRole] = "description";
    this->setRoleNames(roles);

    if (this->session()) {
        if (!this->session()->youtube()->playlistsLoaded()) {
            this->session()->youtube()->getPlaylists();
        }

        this->connect(this->session()->youtube(), SIGNAL(playlistAddedToCache(int)), this, SLOT(onPlaylistAdded(int)));
        this->connect(this->session()->youtube(), SIGNAL(playlistRemovedFromCache(int)), this, SLOT(onPlaylistRemoved(int)));
        this->connect(this->session()->youtube(), SIGNAL(playlistUpdated(int)), this, SLOT(onPlaylistUpdated(int)));
    }

    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
}

YouTubeUserPlaylistListModel::~YouTubeUserPlaylistListModel() {
    if (this->thumbnailCache()) {
        delete this->thumbnailCache();
    }
}

void YouTubeUserPlaylistListModel::setSession(Session *session) {
    m_session = session;

    if (this->session()) {
        if (!this->session()->youtube()->playlistsLoaded()) {
            this->session()->youtube()->getPlaylists();
        }

        this->connect(this->session()->youtube(), SIGNAL(playlistAddedToCache(int)), this, SLOT(onPlaylistAdded(int)));
        this->connect(this->session()->youtube(), SIGNAL(playlistRemovedFromCache(int)), this, SLOT(onPlaylistRemoved(int)));
        this->connect(this->session()->youtube(), SIGNAL(playlistUpdated(int)), this, SLOT(onPlaylistUpdated(int)));
    }
}

void YouTubeUserPlaylistListModel::setThumbnailCache(ThumbnailCache *cache) {
    m_cache = cache;

    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
}

void YouTubeUserPlaylistListModel::onPlaylistAdded(int row) {
    this->beginInsertRows(QModelIndex(), row, row);
    this->endInsertRows();
    emit countChanged(this->rowCount());
}

void YouTubeUserPlaylistListModel::onPlaylistRemoved(int row) {
    this->beginRemoveRows(QModelIndex(), row, row);
    this->endRemoveRows();
    emit countChanged(this->rowCount());
}

void YouTubeUserPlaylistListModel::onPlaylistUpdated(int row) {
    emit dataChanged(this->index(row), this->index(row));
}

int YouTubeUserPlaylistListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return !this->session() ? 0 : this->session()->youtube()->playlists()->size();
}

QVariant YouTubeUserPlaylistListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return this->session()->youtube()->playlists()->at(index.row()).data()->id();
    case PlaylistIdRole:
        return this->session()->youtube()->playlists()->at(index.row()).data()->playlistId();
    case TitleRole:
        return this->session()->youtube()->playlists()->at(index.row()).data()->title();
    case ArtistRole:
        return this->session()->youtube()->playlists()->at(index.row()).data()->artist();
    case DateRole:
        return this->session()->youtube()->playlists()->at(index.row()).data()->date();
    case VideoCountRole:
        return this->session()->youtube()->playlists()->at(index.row()).data()->videoCount();
#ifndef QML_USER_INTERFACE
    case ThumbnailRole:
        return !this->thumbnailCache() ? QVariant() : this->thumbnailCache()->thumbnail(this->session()->youtube()->playlists()->at(index.row()).data()->thumbnailUrl(), QSize(120, 90));
#endif
    case ThumbnailUrlRole:
        return this->session()->youtube()->playlists()->at(index.row()).data()->thumbnailUrl();
    case DescriptionRole:
        return this->session()->youtube()->playlists()->at(index.row()).data()->description();
    default:
        return QVariant();
    }
}

QVariant YouTubeUserPlaylistListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

QSharedPointer<PlaylistItem> YouTubeUserPlaylistListModel::get(int row) const {
    return QSharedPointer<PlaylistItem>(this->session()->youtube()->playlists()->at(row));
}

PlaylistItem* YouTubeUserPlaylistListModel::getFromQML(int row) const {
#ifdef QML_USER_INTERFACE
    PlaylistItem* playlist = new PlaylistItem(this->session()->youtube()->playlists()->at(row).data());
    QDeclarativeEngine::setObjectOwnership(playlist, QDeclarativeEngine::JavaScriptOwnership);

    return playlist;
#else
    Q_UNUSED(row)
#endif
    return 0;
}

void YouTubeUserPlaylistListModel::onThumbnailReady() {
#ifndef QML_USER_INTERFACE
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
#endif
}
