#include "youtubeuserlistmodel.h"
#include "feedurls.h"
#include "session.h"
#include <QNetworkReply>
#include <QNetworkRequest>

YouTubeUserListModel::YouTubeUserListModel(Session *session, QObject *parent) :
    AbstractUserListModel(!session ? 0 : session->newThumbnailCache(), parent),
    m_session(session),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
}

void YouTubeUserListModel::setSession(Session *session) {
    m_session = session;
}

void YouTubeUserListModel::clear() {
    AbstractUserListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void YouTubeUserListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreUsers();
    }
}

void YouTubeUserListModel::setFeed(const QString &feed) {
    if (feed != this->feed()) {
        m_feed = feed;
        m_subscriptions = feed.contains("subscriptions");
    }
}

void YouTubeUserListModel::getUsers(const QString &feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->youtube()->createReply(QString("%1&fields=%2").arg(this->feed()).arg(m_subscriptions ? QString(YOUTUBE_SUBSCRIPTION_FIELDS) : QString(YOUTUBE_USER_FIELDS)), this->offset());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addUsers()));
}

void YouTubeUserListModel::search(const QString &query) {
    if (!query.isEmpty()) {
        this->setQuery(query);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->youtube()->createSearchReply(Queries::Users, this->query(), this->offset());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addUsers()));
    this->disconnect(this->session()->youtube(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getUsers()));
}

void YouTubeUserListModel::addUsers() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this->session()->youtube(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getUsers()));
        this->connect(this->session()->youtube(), SIGNAL(refreshError()), this, SLOT(onError()));
        this->session()->youtube()->refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNodeList entries = doc.elementsByTagName("entry");

        for (int i = 0; i < entries.size(); i++) {
            UserItem *user = new UserItem;
            user->loadYouTubeUser(entries.at(i), m_subscriptions);
            this->connect(this->session()->youtube(), SIGNAL(subscriptionChanged(QString,bool,QString)), user, SLOT(onSubscriptionChanged(QString,bool,QString)));
            this->appendUser(QSharedPointer<UserItem>(user));
        }

        this->setLoading(false);
        int totalResults = doc.namedItem("feed").firstChildElement("openSearch:totalResults").text().toInt();
        this->setMoreResults(totalResults > this->rowCount());
        this->setOffset(this->offset() + 30);

        this->disconnect(this->session()->youtube(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getUsers()));
        this->disconnect(this->session()->youtube(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    emit countChanged(this->rowCount());

    reply->deleteLater();
}

void YouTubeUserListModel::getMoreUsers() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getUsers();
        }
    }
}

UserItem* YouTubeUserListModel::getFromQML(int row) const {
    if (UserItem *user = AbstractUserListModel::getFromQML(row)) {
        this->connect(this->session()->youtube(), SIGNAL(subscriptionChanged(QString,bool,QString)), user, SLOT(onSubscriptionChanged(QString,bool,QString)));

        return user;
    }

    return 0;
}
