#include "youtubeplaylistlistmodel.h"
#include "feedurls.h"
#include "session.h"
#include <QNetworkReply>
#include <QNetworkRequest>

YouTubePlaylistListModel::YouTubePlaylistListModel(Session *session, QObject *parent) :
    AbstractPlaylistListModel(!session ? 0 : session->newThumbnailCache(), parent),
    m_session(session),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
}

void YouTubePlaylistListModel::setSession(Session *session) {
    m_session = session;
}

void YouTubePlaylistListModel::clear() {
    AbstractPlaylistListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void YouTubePlaylistListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMorePlaylists();
    }
}

void YouTubePlaylistListModel::setFeed(const QString &feed) {
    if (feed != this->feed()) {
        m_feed = feed;
    }
}

void YouTubePlaylistListModel::getPlaylists(const QString &feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->youtube()->createReply(QString("%1&fields=%2").arg(this->feed()).arg(QString(YOUTUBE_PLAYLIST_FIELDS)), this->offset());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addPlaylists()));
}

void YouTubePlaylistListModel::search(const QString &query, int order) {
    if (!query.isEmpty()) {
        this->setQuery(query);
        this->setOrder(order);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->youtube()->createSearchReply(Queries::Playlists, this->query(), this->offset(), this->order());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addPlaylists()));
}

void YouTubePlaylistListModel::addPlaylists() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this->session()->youtube(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getPlaylists()));
        this->connect(this->session()->youtube(), SIGNAL(refreshError()), this, SLOT(onError()));
        this->session()->youtube()->refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNodeList entries = doc.elementsByTagName("entry");

        for (int i = 0; i < entries.size(); i++) {
            PlaylistItem *playlist = new PlaylistItem;
            playlist->loadYouTubePlaylist(entries.at(i));
            this->appendPlaylist(QSharedPointer<PlaylistItem>(playlist));
        }

        this->setLoading(false);
        int totalResults = doc.namedItem("feed").firstChildElement("openSearch:totalResults").text().toInt();
        this->setMoreResults((totalResults > this->rowCount()) && (!entries.isEmpty()));
        this->setOffset(this->offset() + 30);

        this->disconnect(this->session()->youtube(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getPlaylists()));
        this->disconnect(this->session()->youtube(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    emit countChanged(this->rowCount());

    reply->deleteLater();
}

void YouTubePlaylistListModel::getMorePlaylists() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getPlaylists();
        }
    }
}
