#include "vimeocommentlistmodel.h"
#include "feedurls.h"
#include "session.h"
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QDebug>

VimeoCommentListModel::VimeoCommentListModel(Session *session, QObject *parent) :
    AbstractCommentListModel(!session ? 0 : session->newThumbnailCache(), parent),
    m_session(session),
    m_loading(true),
    m_replyRow(0),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
    if (this->session()) {
        this->connect(this->session()->vimeo(), SIGNAL(commentAdded(QSharedPointer<CommentItem>)), this, SLOT(onCommentAdded(QSharedPointer<CommentItem>)));
    }
}

void VimeoCommentListModel::setSession(Session *session) {
    m_session = session;

    if (this->session()) {
        this->connect(this->session()->vimeo(), SIGNAL(commentAdded(QSharedPointer<CommentItem>)), this, SLOT(onCommentAdded(QSharedPointer<CommentItem>)));
    }
}

void VimeoCommentListModel::clear() {
    AbstractCommentListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void VimeoCommentListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreComments();
    }
}

void VimeoCommentListModel::setVideoId(const QString &id) {
    m_videoId = id;
    this->setFeed(QString("%1?method=vimeo.videos.comments.getList&video_id=%2&per_page=30").arg(QString(VIMEO_BASE_URL)).arg(id));
}

void VimeoCommentListModel::getComments(const QString &id) {
    if (!id.isEmpty()) {
        this->setVideoId(id);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->vimeo()->createReply(this->feed(), this->offset());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addComments()));
}

void VimeoCommentListModel::addComments() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        QDomNodeList entries = doc.elementsByTagName("comment");

        for (int i = 0; i < entries.size(); i++) {
            CommentItem *comment = new CommentItem;
            comment->loadVimeoComment(entries.at(i).toElement());
            this->appendComment(QSharedPointer<CommentItem>(comment));
        }

        QDomElement statNode = responseNode.firstChildElement("comments");

        this->setLoading(false);
        this->setMoreResults(statNode.attribute("total").toInt() > this->rowCount());
        this->setOffset(statNode.attribute("page").toInt() + 1);

        emit countChanged(this->rowCount());
    }
    else {
        this->setLoading(false);
        this->setError(true);
    }

    reply->deleteLater();
}

void VimeoCommentListModel::getMoreComments() {
    if ((this->moreResults()) && (!this->loading())) {
        this->getComments();
    }
}

void VimeoCommentListModel::onCommentAdded(QSharedPointer<CommentItem> comment) {
    if (!comment.data()->replyId().isEmpty()) {
        comment.data()->setIndent(this->data(this->index(m_replyRow), IndentRole).toInt() + 1);

        if (m_list.size() > m_replyRow) {
            this->insertComment(m_replyRow + 1, comment);
        }
        else {
            this->appendComment(comment);
        }
    }
    else {
        this->insertComment(0, comment);
    }

    emit countChanged(this->rowCount());
}

void VimeoCommentListModel::onCommentError() {
    if (m_replyRow < m_list.size()) {
        this->get(m_replyRow).data()->setLoading(false);
        emit dataChanged(this->index(m_replyRow), this->index(m_replyRow));
    }

    this->disconnect(this->session()->vimeo(), SIGNAL(error(QString)), this, SLOT(onCommentError()));
}
